/*
 * File: Ltsh.java
 * 
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is
 *      obtained from Adobe Systems Incorporated.
 *
 */

package com.adobe.fontengine.font.opentype;

import java.util.Map;

import com.adobe.fontengine.font.FontByteArray;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;

/** Gives access to the 'LTSH' table.
 * 
 * <h4>Version handling</h4>
 * 
 * <p>'LTSH' tables have only a minor version number.
 * This implementation:
 * <ul> 
 * <li>fully supports version 0 tables,</li>
 * <li>interprets version x > 0  tables as version 0 tables,</li>
 * </ul> 
 * 
 * <p>When some data is not present in the actual version of the table,
 * the accessors return identified values.</p>
 * 
 * 
 * <h4>Synchronization</h4>
 * 
 * <p>Like all tables, these objects are immutable.</p> 
 */

public final class Ltsh extends Table {
 
  protected Ltsh (FontByteArray buffer) 
  throws java.io.IOException, InvalidFontException, UnsupportedFontException {
    super (buffer);
  }

  /** Return the YPels for a given gid. **/
  public int getYPels (int gid) throws InvalidFontException {
    int numGlyphs = data.getuint16 (2);
    if (gid < numGlyphs) {
      return data.getuint8 (4 + gid); }
    else {
      return 1; }
  }
  
  public void subsetAndStream (Subset subset, Map tables)
  throws InvalidFontException {
    OTByteArrayBuilder newData = OTByteArray.getOTByteArrayBuilderInstance (subset.getNumGlyphs () * 4);
    newData.setuint16 (0, 0);
    newData.setuint16 (2, subset.getNumGlyphs ());
    int offset = 4;
    for (int i = 0; i < subset.getNumGlyphs (); i++) {
      int fullGid = subset.getFullGid (i);
      newData.setuint8 (offset, getYPels (fullGid));
      offset += 1; }
    
    tables.put (new Integer (Tag.table_LTSH), newData); 
  }  
}
