/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.i18n;

import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.ServiceReference;
import org.slf4j.LoggerFactory;

import java.util.Locale;
import java.util.IllformedLocaleException;
import com.adobe.granite.toggle.api.ToggleRouter;

/**
 * Utility class for locale handling.
 */
public abstract class LocaleUtil {

    /**
     * Parses the given text to create its corresponding {@link Locale}. This
     * method supports "-" and "_" as locale element separator.
     * 
     * @param text string to parse
     * @return the {@link Locale}
     */
    private static ToggleRouter toggleRouter;
    /**
     * Block to initialize toggleRouter service
     */
    static {
        Bundle bundle = FrameworkUtil.getBundle(LocaleUtil.class);
        if (bundle != null) {
            BundleContext bundleContext = bundle.getBundleContext();
            if (bundleContext != null) {
                ServiceReference serviceReference = bundleContext.getServiceReference(ToggleRouter.class.getName());
                if (serviceReference != null) {
                    toggleRouter = (ToggleRouter) bundleContext.getService(serviceReference);
                }
            }
        }
    }
    public static Locale parseLocale(final String text) {
        String[] elements = text.split("-|_");
        if (toggleRouter != null && !toggleRouter.isEnabled("FT_CQ-4332753")) {
            if (elements.length == 1)
                return new Locale(elements[0]);
    
            if (elements.length == 2)
                return new Locale(elements[0], elements[1]);
    
            if (elements.length >= 3)
                return new Locale(elements[0], elements[1], elements[2]);
    
            throw new IllegalArgumentException("Unparsable text: " + text);
        } else {
            if (elements.length == 0) {
                throw new IllegalArgumentException("Unparsable text: " + text);
            }
    
            if (elements.length == 1) {
                return new Locale(elements[0]);
            }
            
            Locale localeWithBuilder = createLocaleWithBuilder(elements);
            if (localeWithBuilder != null) {
                return localeWithBuilder;
            }
            return createLocaleWithConstructor(elements);
        }
    }
    
    private static Locale createLocaleWithBuilder(String[] parts) {
        if (isScript(parts[1])) {
            try {
                switch (parts.length) {
                    case 2:
                        return new Locale.Builder()
                            .setLanguage(parts[0])
                            .setScript(parts[1])
                            .build();
                    case 3:
                        return new Locale.Builder()
                            .setLanguage(parts[0])
                            .setScript(parts[1])
                            .setRegion(parts[2])
                            .build();
                    default: // case >= 4
                        return processMultipleParts(parts);
                }
            } catch (IllformedLocaleException e) {
                LoggerFactory.getLogger(LocaleUtil.class)
                    .warn("Failed to create locale with LocaleBuilder", e);
            }
        }
        return null;
    }
    
    private static Locale processMultipleParts(String[] parts) {
        Locale.Builder localeBuilder =
            new Locale.Builder().setLanguage(parts[0]).setScript(parts[1]).setRegion(parts[2]);
        try {
            localeBuilder.setVariant(parts[3]);
            return localeBuilder.build();
        } catch (IllformedLocaleException e) {
            // creating locale with language, script and country
            return localeBuilder.build();
        }
    }
    
    private static Locale createLocaleWithConstructor(String[] elements) {
        String language = elements[0];
        String country = elements.length > 1 ? elements[1] : "";
        String variant = elements.length > 2 ? elements[2] : "";
        return new Locale(language, country, variant);
    }
    
    
    private static boolean isAlpha(char c) {
        return (c >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z');
    }
    
    private static boolean isAlphaString(String s) {
        int len = s.length();
        for (int i = 0; i < len; i++) {
            if (!isAlpha(s.charAt(i))) {
                return false;
            }
        }
        return true;
    }
    
    private static boolean isScript(String s) {
        // script        = 4ALPHA              ; ISO 15924 code
        return (s.length() == 4) && isAlphaString(s);
    }

    /**
     * Converts the given locale to RFC 4646 format. e.g. "en_US" to "en-US".
     * 
     * @param locale the {@link Locale} to convert to RFC 4646 format
     * @return the string in RFC 4646 format corresponding to the submitted {@link Locale}
     */
    public static String toRFC4646(final Locale locale) {
    
        if (toggleRouter!=null && !toggleRouter.isEnabled("FT_CQ-4332753")) {
            return locale.toString().replace('_', '-');
        } else {
            boolean country = !locale.getCountry().isEmpty();
            boolean variant = !locale.getVariant().isEmpty();
            boolean script = !locale.getScript().isEmpty();
    
            StringBuilder result = new StringBuilder(locale.getLanguage());
            if (script) {
                result.append('-').append(locale.getScript());
            }
            if (country) {
                result.append('-').append(locale.getCountry());
            }
            if (variant) {
                result.append('-').append(locale.getVariant());
            }
            return result.toString();
        }
    }
}
