// =================================================================================================
// ADOBE SYSTEMS INCORPORATED
// Copyright 2012 Adobe Systems Incorporated
// All Rights Reserved
//
// NOTICE:  Adobe permits you to use, modify, and distribute this file in accordance with the terms
// of the Adobe license agreement accompanying it.
// =================================================================================================
package com.adobe.xmp.core;

import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Locale;
import java.util.TimeZone;

import com.adobe.xmp.core.XMPDateTime;
import com.adobe.xmp.core.XMPException;
import com.adobe.xmp.core.impl.ISO8601Converter;

/**
 * The <code>XMPDateTime</code>-class represents a point in time up to a resolution of nano
 * seconds. Dates and time in the serialized XMP are ISO 8601 strings. There are utility functions
 * to convert to the ISO format, a <code>Calendar</code> or get the Timezone. The fields of
 * <code>XMPDateTime</code> are:
 * <ul>
 * <li> month - The month in the range 1..12.
 * <li> day - The day of the month in the range 1..31.
 * <li> minute - The minute in the range 0..59.
 * <li> hour - The time zone hour in the range 0..23.
 * <li> minute - The time zone minute in the range 0..59.
 * <li> nanoSecond - The nano seconds within a second. <em>Note:</em> if the XMPDateTime is
 * converted into a calendar, the resolution is reduced to milli seconds.
 * <li> timeZone - a <code>TimeZone</code>-object.
 * </ul>
 * DateTime values are occasionally used in cases with only a date or only a time component. A date
 * without a time has zeros for all the time fields. A time without a date has zeros for all date
 * fields (year, month, and day).
 */
public class XMPDateTime implements Comparable<XMPDateTime>
{	
	/** */
	private int year = 0;
	/** */
	private int month = 0;
	/** */
	private int day = 0;
	/** */
	private int hour = 0;
	/** */
	private int minute = 0;
	/** */
	private int second = 0;
	/** Use NO time zone as default */
	private TimeZone timeZone = null;
	/**
	 * The nano seconds take micro and nano seconds, while the milli seconds are in the calendar.
	 */
	private int nanoSeconds;
	/** */
	private boolean hasDate = false;
	/** */
	private boolean hasTime = false;
	/** */
	private boolean hasTimeZone = false;


	/**
	 * Creates an <code>XMPDateTime</code>-instance with the current time in the default time
	 * zone.
	 */
	public XMPDateTime()
	{
		// EMPTY
	}


	/**
	 * Creates an <code>XMPDateTime</code>-instance from a calendar.
	 *
	 * @param calendar a <code>Calendar</code>
	 */
	public XMPDateTime(Calendar calendar)
	{
		// extract the date and timezone from the calendar provided
        Date date = calendar.getTime();
        TimeZone zone = calendar.getTimeZone();

        // put that date into a calendar the pretty much represents ISO8601
        // I use US because it is close to the "locale" for the ISO8601 spec
        GregorianCalendar intCalendar =
        	(GregorianCalendar) Calendar.getInstance(Locale.US);
        intCalendar.setGregorianChange(new Date(Long.MIN_VALUE));
        intCalendar.setTimeZone(zone);
        intCalendar.setTime(date);

		this.year = intCalendar.get(Calendar.YEAR);
		this.month = intCalendar.get(Calendar.MONTH) + 1; // cal is from 0..12
		this.day = intCalendar.get(Calendar.DAY_OF_MONTH);
		this.hour = intCalendar.get(Calendar.HOUR_OF_DAY);
		this.minute = intCalendar.get(Calendar.MINUTE);
		this.second = intCalendar.get(Calendar.SECOND);
		this.nanoSeconds = intCalendar.get(Calendar.MILLISECOND) * 1000000;
		this.timeZone = intCalendar.getTimeZone();

		// object contains all date components
		hasDate = hasTime = hasTimeZone = true;
	}


	/**
	 * Creates an <code>XMPDateTime</code>-instance from
	 * a <code>Date</code> and a <code>TimeZone</code>.
	 *
	 * @param date a date describing an absolute point in time
	 * @param timeZone a TimeZone how to interpret the date
	 */
	public XMPDateTime(Date date, TimeZone timeZone)
	{
		GregorianCalendar calendar = new GregorianCalendar(timeZone);
		calendar.setTime(date);

		this.year = calendar.get(Calendar.YEAR);
		this.month = calendar.get(Calendar.MONTH) + 1; // cal is from 0..12
		this.day = calendar.get(Calendar.DAY_OF_MONTH);
		this.hour = calendar.get(Calendar.HOUR_OF_DAY);
		this.minute = calendar.get(Calendar.MINUTE);
		this.second = calendar.get(Calendar.SECOND);
		this.nanoSeconds = calendar.get(Calendar.MILLISECOND) * 1000000;
		this.timeZone = timeZone;

		// object contains all date components
		hasDate = hasTime = hasTimeZone = true;
	}


	/**
	 * Creates an <code>XMPDateTime</code>-instance from an ISO 8601 string.
	 *
	 * @param strValue an ISO 8601 string
	 * @throws XMPException If the string is a non-conform ISO 8601 string, an exception is thrown
	 */
	public XMPDateTime(String strValue) throws XMPException
	{
		ISO8601Converter.parse(strValue, this);
	}

	/**
	 * Creates an <code>XMPDateTime</code>-object from initial values.
	 * @param year years
	 * @param month months from 1 to 12<br>
	 * <em>Note:</em> Remember that the month in {@link Calendar} is defined from 0 to 11.
	 * @param day days
	 */
	public XMPDateTime(int year, int month, int day)
	{
		this.setYear(year);
		this.setMonth(month);
		this.setDay(day);
	}
	

	/**
	 * Creates an <code>XMPDateTime</code>-object from initial values.
	 * @param year years
	 * @param month months from 1 to 12<br>
	 * <em>Note:</em> Remember that the month in {@link Calendar} is defined from 0 to 11.
	 * @param day days
	 * @param hour hours
	 * @param minute minutes
	 * @param second seconds
	 * @param nanoSecond nanoseconds
	 */
	public  XMPDateTime(int year, int month, int day,
		int hour, int minute, int second, int nanoSecond)
	{
		this.setYear(year);
		this.setMonth(month);
		this.setDay(day);
		this.setHour(hour);
		this.setMinute(minute);
		this.setSecond(second);
		this.setNanoSecond(nanoSecond);
	}
	
	/**
	 * @see XMPDateTime#getYear()
	 */
	public int getYear()
	{
		return year;
	}


	/**
	 * @see XMPDateTime#setYear(int)
	 */
	public void setYear(int year)
	{
		this.year = Math.min(Math.abs(year), 9999);
		this.hasDate = true;
	}


	/**
	 * @see XMPDateTime#getMonth()
	 */
	public int getMonth()
	{
		return month;
	}


	/**
	 * @see XMPDateTime#setMonth(int)
	 */
	public void setMonth(int month)
	{
		if (month < 1)
		{
			this.month = 1;
		}
		else if (month > 12)
		{
			this.month = 12;
		}
		else
		{
			this.month = month;
		}

		this.hasDate = true;
	}


	/**
	 * @see XMPDateTime#getDay()
	 */
	public int getDay()
	{
		return day;
	}


	/**
	 * @see XMPDateTime#setDay(int)
	 */
	public void setDay(int day)
	{
		if (day < 1)
		{
			this.day = 1;
		}
		else if (day > 31)
		{
			this.day = 31;
		}
		else
		{
			this.day = day;
		}

		this.hasDate = true;
	}


	/**
	 * @see XMPDateTime#getHour()
	 */
	public int getHour()
	{
		return hour;
	}


	/**
	 * @see XMPDateTime#setHour(int)
	 */
	public void setHour(int hour)
	{
		this.hour = Math.min(Math.abs(hour), 23);
		this.hasTime = true;
	}


	/**
	 * @see XMPDateTime#getMinute()
	 */
	public int getMinute()
	{
		return minute;
	}


	/**
	 * @see XMPDateTime#setMinute(int)
	 */
	public void setMinute(int minute)
	{
		this.minute = Math.min(Math.abs(minute), 59);
		this.hasTime = true;
	}


	/**
	 * @see XMPDateTime#getSecond()
	 */
	public int getSecond()
	{
		return second;
	}


	/**
	 * @see XMPDateTime#setSecond(int)
	 */
	public void setSecond(int second)
	{
		this.second = Math.min(Math.abs(second), 59);
		this.hasTime = true;
	}


	/**
	 * @see XMPDateTime#getNanoSecond()
	 */
	public int getNanoSecond()
	{
		return nanoSeconds;
	}


	/**
	 * @see XMPDateTime#setNanoSecond(int)
	 */
	public void setNanoSecond(int nanoSecond)
	{
		this.nanoSeconds = nanoSecond;
		this.hasTime = true;
	}


	/**
	 * @see Comparable#compareTo(Object)
	 */
	public int compareTo(XMPDateTime dt)
	{
		long d = getCalendar().getTimeInMillis()
				- ((XMPDateTime) dt).getCalendar().getTimeInMillis();
		if (d != 0)
		{
			return (int) Math.signum(d);
		}
		else
		{
			// if millis are equal, compare nanoseconds
			d = nanoSeconds - ((XMPDateTime) dt).getNanoSecond();
			return (int) Math.signum(d);
		}
	}


	/**
	 * @see XMPDateTime#getTimeZone()
	 */
	public TimeZone getTimeZone()
	{
		return timeZone;
	}


	/**
	 * @see XMPDateTime#setTimeZone(TimeZone)
	 */
	public void setTimeZone(TimeZone timeZone)
	{
		this.timeZone = timeZone;
		this.hasTime = true;
		this.hasTimeZone = true;
	}


	/**
	 * @see XMPDateTime#hasDate()
	 */
	public boolean hasDate()
	{
		return this.hasDate;
	}


	/**
	 * @see XMPDateTime#hasTime()
	 */
	public boolean hasTime()
	{
		return this.hasTime;
	}


	/**
	 * @see XMPDateTime#hasTimeZone()
	 */
	public boolean hasTimeZone()
	{
		return this.hasTimeZone;
	}


	/**
	 * @see XMPDateTime#getCalendar()
	 */
	public Calendar getCalendar()
	{
		GregorianCalendar calendar = (GregorianCalendar) Calendar.getInstance(Locale.US);
		calendar.setGregorianChange(new Date(Long.MIN_VALUE));
		if (hasTimeZone)
		{
			calendar.setTimeZone(timeZone);
		}
		calendar.set(Calendar.YEAR, year);
		calendar.set(Calendar.MONTH, month - 1);
		calendar.set(Calendar.DAY_OF_MONTH, day);
		calendar.set(Calendar.HOUR_OF_DAY, hour);
		calendar.set(Calendar.MINUTE, minute);
		calendar.set(Calendar.SECOND, second);
		calendar.set(Calendar.MILLISECOND, nanoSeconds / 1000000);

		return calendar;
	}


	/**
	 * @see XMPDateTime#getISO8601String()
	 */
	public String getISO8601String()
	{
		return ISO8601Converter.render(this);
	}

	/**
	 * @return Returns the ISO string representation.
	 */
	public String toString()
	{
		return getISO8601String();
	}
	
	
}
