/*************************************************************************
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2011 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 **************************************************************************/

package com.day.cq.replication;

import java.io.Serializable;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import org.apache.sling.event.dea.DEAConstants;
import org.osgi.service.event.Event;
import org.osgi.service.event.EventProperties;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Defines the control information of a replication
 */
public class ReplicationAction implements Serializable {

    private static final Logger LOG = LoggerFactory.getLogger(ReplicationAction.class);

    private static final long serialVersionUID = -151606943486007293L;

    /**
     * Event topic for replication status events.
     */
    public static final String EVENT_TOPIC = "com/day/cq/replication";

    /**
     * The OSGi event property containing the modification date
     */
    public static final String PROPERTY_MODIFICATION_DATE = "modificationDate";

    /**
     * The OSGi event property containing the user id
     */
    public static final String PROPERTY_USER_ID = "userId";

    /**
     * The OSGi event property containing the path
     */
    public static final String PROPERTY_PATH = "path";

    /**
     * The OSGi event property containing the paths
     * @since 5.5
     */
    public static final String PROPERTY_PATHS = "paths";

    /**
     * The OSGi event property containing replication action
     */
    public static final String PROPERTY_TYPE = "type";

    /**
     * The OSGi event property containing the replicated revision (optional)
     */
    public static final String PROPERTY_REVISION = "revision";

    /**
     * Replication path property name.
     */
    public static final String PN_PATH = "cq:repPath";

    /**
     * Replication action type property name.
     */
    public static final String PN_ACTION_TYPE = "cq:repActionType";

    /**
     * the action type
     */
    private final ReplicationActionType type;

    /**
     * all aggregate paths of a replication
     */
    private final String paths[];

    /**
     * The replication time
     */
    private final long time;

    /**
     * The user id triggering the action
     */
    private final String userId;

    /**
     * The revision to replicate (default null)
     */
    private String revision;

    /**
     * Agent config.
     */
    private transient AgentConfig config;

    /**
     * Replication log.
     */
    private transient ReplicationLog log;


    public ReplicationAction(ReplicationActionType type, String path, long time,
                             String userId, String revision) {

        if (type == null) {
            throw new IllegalArgumentException("Type must not be null.");
        }
        if (path == null) {
            throw new IllegalArgumentException("Path must not be null.");
        }
        if (userId == null) {
            throw new IllegalArgumentException("Userid must not be null.");
        }
        this.type = type;
        this.paths = new String[]{path};
        this.time = time == 0 ? System.currentTimeMillis() : time;
        this.userId = userId;
        this.revision = revision;
    }

    public ReplicationAction(ReplicationActionType type, String[] paths, long time,
                             String userId, String revision) {

        if (type == null) {
            throw new IllegalArgumentException("Type must not be null.");
        }
        if (paths == null || paths.length == 0) {
            throw new IllegalArgumentException("Paths must not be null or empty.");
        }
        if (userId == null) {
            throw new IllegalArgumentException("Userid must not be null.");
        }
        this.type = type;
        this.paths = paths;
        this.time = time == 0 ? System.currentTimeMillis() : time;
        this.userId = userId;
        this.revision = revision;
    }

    /**
     * Create a new instance of this class. Used for actions that do not carry
     * content (e.g. DELETE).
     *
     * @param type
     *            action type
     * @param path
     *            path
     */
    public ReplicationAction(ReplicationActionType type, String path) {
        this(type, path, 0, "", null);
    }

    /**
     * Returns the action type
     *
     * @return replication action type
     */
    public ReplicationActionType getType() {
        return type;
    }

    /**
     * Returns the path
     *
     * @return path
     */
    public String getPath() {
        if (paths.length > 1) {
            LOG.warn("Using getPath() while {} paths available", paths.length);
        }
        return paths[0];
    }

    /**
     * Returns the replication paths. Currently the array may only contain more than 1 elements for a delete
     * replication. in that case, the paths are the ones of the deleted aggregates (content nodes).
     *
     * @return the paths
     * @since 5.5
     */
    public String[] getPaths() {
        return paths;
    }

    /**
     * Returns the revision
     *
     * @return the revision
     */
    public String getRevision() {
        return revision;
    }

    /**
     * Returns the replication time
     *
     * @return the replication time
     */
    public long getTime() {
        return time;
    }

    /**
     * Returns the user id.
     *
     * @return the user id.
     */
    public String getUserId() {
        return userId;
    }

    /**
     * Return the agent config.
     *
     * @return agent config
     */
    public AgentConfig getConfig() {
        return config;
    }

    /**
     * Set the agent config.
     *
     * @param config agent config
     */
    public void setConfig(AgentConfig config) {
        this.config = config;
    }

    /**
     * Return the replication log.
     *
     * @return replication log
     */
    public ReplicationLog getLog() {
        return log;
    }

    /**
     * Set the replication log.
     *
     * @param log the log
     */
    public void setLog(ReplicationLog log) {
        this.log = log;
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder();
        sb.append("ReplicationAction");
        sb.append("{type=").append(type);
        sb.append(", path[0]='").append(paths[0]).append('\'');
        sb.append(", time=").append(time);
        sb.append(", userId='").append(userId).append('\'');
        sb.append(", revision='").append(revision).append('\'');
        sb.append('}');
        return sb.toString();
    }

    /**
     * Convert an OSGi event to a replication event.
     *
     * @param evt The OSGi event
     * @return The replication event if the OSGi event contained a replication
     *         event. Otherwise null is returned.
     */
    public static ReplicationAction fromEvent(Event evt) {
        if (!evt.getTopic().equals(EVENT_TOPIC)) {
            return null;
        }
        Object lastMod = evt.getProperty(PROPERTY_MODIFICATION_DATE);
        long time = 0;
        if (lastMod instanceof Date) {
            time = ((Date) lastMod).getTime();
        } else if (lastMod instanceof Calendar) {
            time = ((Calendar) lastMod).getTimeInMillis();
        } else if (lastMod instanceof Long) {
            time = (Long) lastMod;
        }
        String[] paths = (String[]) evt.getProperty(PROPERTY_PATHS);
        if (paths == null) {
            paths = new String[]{(String) evt.getProperty(PROPERTY_PATH)};
        }

        return new ReplicationAction(
                ReplicationActionType.fromName((String) evt.getProperty(PROPERTY_TYPE)),
                paths,
                time,
                (String) evt.getProperty(PROPERTY_USER_ID),
                (String) evt.getProperty(PROPERTY_REVISION)
        );
    }

    /**
     * Create the OSGi event properties.
     *
     * @param distribute Whether this event should be distributed across the cluster.
     * @return OSGi event properties
     * @since 5.18.0
     */
    public Map<String, Object> createEventProperties(final boolean distribute) {
        final Map<String, Object> properties = new HashMap<String, Object>();
        properties.put(PROPERTY_TYPE, type.toString());
        properties.put(PROPERTY_PATHS, paths);
        Calendar c = Calendar.getInstance();
        c.setTimeInMillis(time);
        properties.put(PROPERTY_MODIFICATION_DATE, c);
        properties.put(PROPERTY_USER_ID, userId);
        if (revision != null) {
            properties.put(PROPERTY_REVISION, revision);
        }
        if ( distribute ) {
            properties.put(DEAConstants.PROPERTY_DISTRIBUTE, "");
        }
        return new EventProperties(properties);
    }

    /**
     * Create an OSGi event out of the replication event.
     *
     * @return A new OSGi event.
     */
    public Event toEvent() {
        return toEvent(false);
    }

    /**
     * Create an OSGi event out of the replication event.
     *
     * @param distribute if <code>true</code> a distributed event is created
     * @return A new OSGi event.
     */
    public Event toEvent(final boolean distribute) {
        return new Event(EVENT_TOPIC, createEventProperties(distribute));
    }
}
