/* ****************************************************************************
 *
 *	File: CosDictionary.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.cos;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import com.adobe.internal.io.stream.InputByteStream;
import com.adobe.internal.io.stream.OutputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosParseException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityException;
import com.adobe.internal.pdftoolkit.core.types.ASName;
import com.adobe.internal.pdftoolkit.core.types.ASString;

/**
 * Represents a COS dictionary as defined in section 3.2.6 of the PDF
 * Reference Manual version 1.4.
 */
public class CosDictionary extends CosContainer
{
	Map<ASName, CosObject> mData;				// Underlying dictionary map
	boolean invalidDict = false;	
	
	
	/**
	 * 
	 * Only for Internal Engineering Use. '
	 * This api may change without Notice 
	 * Used for PDFA validation of dictionary.
	 * 
	 * @return true if the dictionary has unequal key and values; false
	 * 			otherwise.
	 */
	public boolean isInvalidDict() {
		return invalidDict;
	}

	/**
	 * Constructs a COS dictionary based on the specified map.
	 * This is used by CosToken after it has read in the dictionary values; the
	 * map is already ASName - CosObject pairings. To insure that the direct
	 * object values get their parent object set, if this is indirect iterate through
	 * the CosObject values and set the value for any direct CosString or CosContainer
	 * objects.
	 *
	 * @param doc		Document containing the dictionary
	 * @param data		Map for the dictionary (ASName key - CosObject value pairs)
	 * @param info		Object info for the dictionary
	 */
	CosDictionary(CosDocument doc, Map<ASName, CosObject> data, CosObjectInfo info)
	{
		super(doc, info);
		
		if(data.containsKey(null))
		{
			data.remove(null);
			invalidDict = true;
		}
		
		mData = data;

		repaired = info != null && doc.isDocumentCosLevelRepaired() && doc.getRepairList() != null && doc.getRepairList().isObjectRepaired(info.getObjNum());
		
		// If this CosDictionary is indirect and there are CosObjects in the map, set
		// the parent object of any direct CosString or CosContainer value to this object
		if (!mData.isEmpty() && isIndirect()) {
			Iterator iter = (mData.values()).iterator();
			while (iter.hasNext()) {
				CosObject obj = (CosObject)iter.next();
				if (!obj.isIndirect()) {
					if (obj instanceof CosString)
						((CosString)obj).setParentObj(this);
					else if (obj instanceof CosContainer)
						if (obj instanceof CosArray)
							((CosArray)obj).setParentObj(this);
						else
							((CosDictionary)obj).setParentObj(this);
				}
			}
		}
	}

	/**
	 * Returns a "direct" dictionary object with all the direct objects cloned but
	 * keys and indirect objects are not cloned.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosDictionary copy()
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		if (isIndirect())
			return this;
		CosDictionary copy = getDocument().createCosDictionary(CosObject.DIRECT);
		Iterator keyiter = keyIterator();
		while (keyiter.hasNext()) {
			ASName key = (ASName)keyiter.next();
			CosObject obj = get(key);
			if (!obj.isIndirect()) {
				if (obj instanceof CosArray)
					obj = ((CosArray)obj).copy();
				else if (obj instanceof CosDictionary)
					obj = ((CosDictionary)obj).copy();
				else if (obj instanceof CosString)
					obj = ((CosString)obj).copy();
			}
			copy.put(key, obj);
		}
		return copy;
	}

	/**
	 * Returns the list of keys of this dictionary.
	 */
	public List<ASName> getKeys()
	{
		List keys = new ArrayList<ASName>();
		Iterator<ASName> iter = keyIterator();
		while (iter.hasNext()) {
			ASName key = iter.next();
			keys.add(key);
		}
		return keys;
	}

	/**
	 * 
	 *
	 * return the type of this CosObject
	 */
	@Override
	public int getType()
	{
		return t_Dictionary;
	}

	/**
	 * 
	 *
	 * Indicates whether the dictionary has any entries.
	 *
	 * @return true if the dictionary does not contain any entries; false
	 * 			otherwise.
	 */
	public boolean isEmpty()
	{
		return mData.isEmpty();
	}

	/**
	 * 
	 *
	 * Returns the number of key/value pairs in the dictionary.
	 *
	 * @return Number of key/value pairs in the dictionary.
	 */
	public int size()
	{
		return mData.size();
	}

	/**
	 * 
	 *
	 * Returns whether this dictionary contains an entry for the given key
	 * @param key
	 */
	public boolean containsKey(Object key)
	{
		return mData.containsKey(key);
	}

	/**
	 * 
	 *
	 * Set this value if this container is direct and is being added to a container.
	 *
	 * @param parent - CosContainer
	 */
	@Override
	void setParentObj(CosContainer parent)
	{
		if (isIndirect())
			return;
		mParentObj = parent;
		List<CosObject> values = values();
		Iterator<CosObject> iter = values.iterator();
		while (iter.hasNext()) {
			CosObject obj = iter.next();
			if (!obj.isIndirect()) {
				if (obj instanceof CosString)
					((CosString)obj).setParentObj(parent);
				else if (obj instanceof CosContainer) {
					if (obj instanceof CosArray)
						((CosArray)obj).setParentObj(parent);
					else
						((CosDictionary)obj).setParentObj(parent);
				}
			}
		}
	}

	/**
	 * 
	 *
	 * Set the specified encryption state for all strings and streams in a dictionary recursively 
	 */
	public void setEncryptionState(boolean state)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			Iterator<ASName> iter = keyIterator();
			while (iter.hasNext()){
				ASName key = iter.next();
				CosObject value = get(key);
				if (value.getType() == CosObject.t_String){
					((CosString) value).setIsEncrypted(state);
					((CosString) value).setToEncrypt(state);
				}
				else if (value.getType() == CosObject.t_Stream){
					((CosStream) value).setIsEncrypted(state);
					((CosStream) value).setToEncrypt(state);
				}
				else if (value.getType() == CosObject.t_Dictionary){
					((CosDictionary) value).setEncryptionState(state);}
				else if (value.getType() == CosObject.t_Array){
					((CosArray) value).setEncryptionState(state);
				}
			}
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	/**
	 * 
	 *
	 * Obtains the object corresponding to the specified key. An indirect
	 * object reference is resolved to the referenced object before being
	 * returned.
	 *
	 * @param key	A key in the dictionary
	 *
	 * @return Object corresponding to the specified key.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosObject get(ASName key)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			CosObject result = null;
			if(repaired)
				result = getDocument().getRepairedValue(this.getObjNum(), key);
			if(result == null)
				result = mData.get(key);
			
			if (result instanceof CosObjectRef)
				result = getDocument().resolveReference((CosObjectRef)result);
			return result;
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	CosObjectRef getRef(ASName key)
	{
		CosObject result = mData.get(key);
		if (result instanceof CosObjectRef)
			return (CosObjectRef)result;
		return null;
	}

	/**
	 * 
	 *
	 * Returns the type of the value associated with the given key
	 * @param key
	 */
	public int getType(ASName key)
	{
		if (!mData.containsKey(key)) {
			return t_KeyAbsent;
		}
		CosObject val = mData.get(key);
		if (val == null)
			return t_Null; //treat null as CosNull
		return val.getType();
	}

	/**
	 * 
	 *
	 * Accesor functions to get the scalar value from the object associated
	 * with a given key. These should only be called if the key is known
	 * to be present and the type is known to be compatible with scalar
	 * result
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public ASName getName(ASName key)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject val = get(key);
		if (val == null)
			return null;
		return val.nameValue();
	}

	/**
	 * Accesor function to get the scalar value from the object associated
	 * with a given key. These should only be called if the key is known
	 * to be present and the type is known to be compatible with scalar
	 * result.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public Boolean getBoolean(ASName key)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject val = get(key);
		if (val == null)
			return null;
		return Boolean.valueOf(val.booleanValue());
	}

	/**
	 * Accesor function to get the scalar value from the object associated
	 * with a given key. These should only be called if the key is known
	 * to be present and the type is known to be compatible with scalar
	 * result.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public Double getDouble(ASName key)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject val = get(key);
		if (val == null)
			return null;
		return new Double(val.doubleValue());
	}

	/**
	 * Accesor function to get the scalar value from the object associated
	 * with a given key. These should only be called if the key is known
	 * to be present and the type is known to be compatible with scalar
	 * result.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public Integer getInt(ASName key)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject val = get(key);
		if (val == null)
			return null;
		return Integer.valueOf(val.intValue());
	}

	/**
	 * Accesor function to get the scalar value from the object associated
	 * with a given key. These should only be called if the key is known
	 * to be present and the type is known to be compatible with scalar
	 * result.
	 * 
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public Long getLong(ASName key)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject val = get(key);
		if (val == null)
			return null;
		return Long.valueOf(val.longValue());
	}

	/**
	 * Accesor function to get the scalar value from the object associated
	 * with a given key. These should only be called if the key is known
	 * to be present and the type is known to be compatible with scalar
	 * result.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public ASString getString(ASName key)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject val = get(key);
		if (val == null)
			return null;
		return val.stringValue();
	}

	/**
	 * Accesor function to get the scalar value from the object associated
	 * with a given key. These should only be called if the key is known
	 * to be present and the type is known to be compatible with scalar
	 * result.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosString getCosString(ASName key)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject obj = get(key);
		if(obj == null || obj instanceof CosNull)
			return null;
		return (CosString)get(key);
	}

	/**
	 * Accesor function to get the array value from the object associated
	 * with a given key. These should only be called if the key is known
	 * to be present and the type is known to be compatible with result.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosArray getCosArray(ASName key)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject obj = get(key);
		if(obj == null || obj instanceof CosNull)
			return null;
		return (CosArray)get(key);
	}

	/**
	 * Accesor function to get the dictionary value from the object associated
	 * with a given key. These should only be called if the key is known
	 * to be present and the type is known to be compatible with result.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosDictionary getCosDictionary(ASName key)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject obj = get(key);
		if(obj == null || obj instanceof CosNull)
			return null;
		return (CosDictionary)get(key);
	}

	/**
	 * Accesor function to get the stream value from the object associated
	 * with a given key. These should only be called if the key is known
	 * to be present and the type is known to be compatible with result.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosStream getCosStream(ASName key)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject obj = get(key);
		if(obj == null || obj instanceof CosNull)
			return null;
		return (CosStream)get(key);
	}

	/**
	 * Accesor function to get the decoded stream value from the object associated
	 * with a given key. These should only be called if the key is known
	 * to be present and the type is known to be compatible with result.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public InputByteStream getStream(ASName key)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject val = get(key);
		if (val == null)
			return null;
		return ((CosStream)val).getStreamDecoded();
	}

	/**
	 * Set the repaired value for this cos dictionary. 
	 * @param key
	 * @param cosObject
	 */
	public void setRepairedValue(ASName key, CosObject cosObject)
	{
		repaired = true;
		getDocument().setRepairedValue(this, key, cosObject);
	}
	
	/**
	 * Places the key/value pair into the dictionary. If the value is an
	 * indirect object, the object's reference is used as the value. This will
	 * overwrite the value associated with the key if there already is an entry for
	 * the key in the dictionary. The removed value will have its reference count
	 * decremented if it's on the dirty list. The added value will have its reference
	 * count incremented if it's on the dirty list.
	 *
	 * @param key		Key for the entry
	 * @param cosObject	 CosObject value for the entry
	 *
	 * @return CosObject value added to the dictionary.
	 * @throws PDFIOException
	 * @throws PDFCosParseException
	 * @throws PDFSecurityException
	 */
	public CosObject put(ASName key, CosObject cosObject)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			if (getDocument() != cosObject.getDocument())
				throw new PDFCosParseException("Object and container must be in same document");
			if (cosObject.isIndirect()) {
				if (!(cosObject instanceof CosObjectRef))
					cosObject = getDocument().getObjectRef(cosObject.getInfo());
				mData.put(key, cosObject);
			} else {
				// We're adding a direct object; if it's a CosContainer or CosString set parent object
				CosContainer parentObj = isIndirect() ? this : mParentObj;
				CosContainer oldParentObj;
				if (cosObject instanceof CosContainer) {
					// We're adding a direct CosContainer
					// If this container already belongs to another collection,
					// make a copy and add the copy
					oldParentObj = ((CosContainer)cosObject).getParentObj();
					if (oldParentObj != null) {
						if (cosObject instanceof CosArray)
							cosObject = ((CosArray)cosObject).copy();
						else
							cosObject = ((CosDictionary)cosObject).copy();
					}
					((CosContainer)cosObject).setParentObj(parentObj);
				} else if (cosObject instanceof CosString) {
					// We're adding a direct CosString
					oldParentObj = ((CosString)cosObject).getParentObj();
					if (oldParentObj != null)
						cosObject = ((CosString)cosObject).copy();
					((CosString)cosObject).setParentObj(parentObj);
				}
				mData.put(key, cosObject);
			}
			if (isIndirect())
				getInfo().markDirty();
			else {
				if (mParentObj != null)
					mParentObj.getInfo().markDirty();
			}
			return cosObject;
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	/**
	 * Places the key/value pair into the dictionary. This will
	 * overwrite the value associated with the key if there already is an entry for
	 * the key in the dictionary. The removed value will have its reference count
	 * decremented if it's on the dirty list. The added value will have its reference
	 * count incremented if it's on the dirty list.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosObject put(ASName key, ASName value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosName cosValue = getDocument().createCosName(value);
		return put(key, cosValue);
	}

	/**
	 * Places the key/value pair into the dictionary. This will
	 * overwrite the value associated with the key if there already is an entry for
	 * the key in the dictionary. The removed value will have its reference count
	 * decremented if it's on the dirty list. The added value will have its reference
	 * count incremented if it's on the dirty list.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosObject put(ASName key, boolean value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosBoolean cosValue = getDocument().createCosBoolean(value);
		return put(key, cosValue);
	}

	/**
	 * Places the key/value pair into the dictionary. This will
	 * overwrite the value associated with the key if there already is an entry for
	 * the key in the dictionary. The removed value will have its reference count
	 * decremented if it's on the dirty list. The added value will have its reference
	 * count incremented if it's on the dirty list.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosObject put(ASName key, double value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosNumeric cosValue = getDocument().createCosNumeric(value);
		return put(key, cosValue);
	}

	/**
	 * Places the key/value pair into the dictionary. This will
	 * overwrite the value associated with the key if there already is an entry for
	 * the key in the dictionary. The removed value will have its reference count
	 * decremented if it's on the dirty list. The added value will have its reference
	 * count incremented if it's on the dirty list.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosObject put(ASName key, long value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosNumeric cosValue = getDocument().createCosNumeric(value);
		return put(key, cosValue);
	}

	/**
	 * Places the key/value pair into the dictionary. This will
	 * overwrite the value associated with the key if there already is an entry for
	 * the key in the dictionary. The removed value will have its reference count
	 * decremented if it's on the dirty list. The added value will have its reference
	 * count incremented if it's on the dirty list.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosObject put(ASName key, int value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosNumeric cosValue = getDocument().createCosNumeric(value);
		return put(key, cosValue);
	}

	/**
	 * Places the key/value pair into the dictionary. This will
	 * overwrite the value associated with the key if there already is an entry for
	 * the key in the dictionary. The removed value will have its reference count
	 * decremented if it's on the dirty list. The added value will have its reference
	 * count incremented if it's on the dirty list.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosObject put(ASName key, ASString value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosString cosValue = getDocument().createCosString(value);
		cosValue.setParentObj(isIndirect() ? this : mParentObj);
		return put(key, cosValue);
	}

	/**
	 * Places the key/value pair into the dictionary. This will
	 * overwrite the value associated with the key if there already is an entry for
	 * the key in the dictionary. The removed value will have its reference count
	 * decremented if it's on the dirty list. The added value will have its reference
	 * count incremented if it's on the dirty list.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosObject put(ASName key, String value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosName cosValue = getDocument().createCosName(ASName.create(value));
		return put(key, cosValue);
	}

	/**
	 * 
	 *
	 * Put an array of double into a Dictionary under the given ASName key.
	 * @param key
	 * @param value an array of values
	 * @throws PDFIOException
	 * @throws PDFCosParseException
	 * @throws PDFSecurityException
	 */
	public CosObject put(ASName key, double[] value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosArray cosArray = getDocument().createCosArray(CosObject.DIRECT);
		for (int i = 0; i < value.length; i++)
			cosArray.addDouble(value[i]);
		return put(key, cosArray);
	}

	/**
	 * Put an array of long into a Dictionary under the given ASName key.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosObject put(ASName key, long[] value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosArray cosArray = getDocument().createCosArray(CosObject.DIRECT);
		for (int i = 0; i < value.length; i++)
			cosArray.addLong(value[i]);
		return put(key, cosArray);
	}

	/**
	 * 
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosObject put(ASName key, ASName[] value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosArray cosArray = getDocument().createCosArray(CosObject.DIRECT);
		for (int i = 0; i < value.length; i++)
			cosArray.addName(value[i]);
		return put(key, cosArray);
	}

	/**
	 * Put an array of int into a Dictionary under the given ASName key.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosObject put(ASName key, int[] value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosArray cosArray = getDocument().createCosArray(CosObject.DIRECT);
		for (int i = 0; i < value.length; i++)
			cosArray.addInt(value[i]);
		return put(key, cosArray);
	}

	/**
	 * Put an array of string into a Dictionary under the given ASName key.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosObject put(ASName key, String[] value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosArray cosArray = getDocument().createCosArray(CosObject.DIRECT);
		cosArray.setParentObj(isIndirect() ? this : mParentObj);
		for (int i = 0; i < value.length; i++)
			cosArray.addName(ASName.create(value[i]));
		return put(key, cosArray);
	}

	/**
	 * Put an array of boolean into a Dictionary under the given ASName key.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosObject put(ASName key, boolean[] value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosArray cosArray = getDocument().createCosArray(CosObject.DIRECT);
		for (int i = 0; i < value.length; i++)
			cosArray.addBoolean(value[i]);
		return put(key, cosArray);
	}

	/**
	 * Removes all entries from the CosDictionary.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public void clear()
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			if (mData.isEmpty())
				return;
			mData.clear();
			if (isIndirect())
				getInfo().markDirty();
			else {
				if (mParentObj != null)
					mParentObj.getInfo().markDirty();
			}
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	/**
	 * 
	 *
	 * Removes the key mapping from the dictionary.
	 *
	 * @param key		Key mapping to remove
	 *
	 * @return If an object was already present in the dictionary using the
	 * 			specified key, that object is returned. Otherwise, null is
	 * 			returned.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public CosObject remove(ASName key)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			CosObject rslt = mData.remove(key);
			if (rslt != null) {
				if (rslt instanceof CosObjectRef)
					rslt = getDocument().resolveReference((CosObjectRef)rslt);
				if (isIndirect())
					getInfo().markDirty();
				else {
					if (mParentObj != null)
						mParentObj.getInfo().markDirty();
				}
			}
			return rslt;
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	/**
	 * Returns the key set of this dictionary.
	 */
	public Set<ASName> keySet()
	{
		return mData.keySet();
	}
	
	/**
	 * 
	 */
	Set<Map.Entry<ASName, CosObject>> entrySet()
	{
		return mData.entrySet();
	}

	/**
	 * Returns an iterator over keys of this dictionary.
	 */
	public Iterator<ASName> keyIterator()
	{
		return mData.keySet().iterator();
	}

	@Override
	public CosContainerValuesIterator getValuesIterator()
	{
		Map.Entry[] entries = new Map.Entry[size()];
		entries = mData.entrySet().toArray(entries);
		return new CosDictionaryValuesIterator(entries);
	}

	/**
	 * Returns a list of values in this dictionary.
	 */
	public List<CosObject> values()
	{
		return new ArrayList(mData.values());
	}

	
	@Override
	public Object getValue()
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		Map<String, Object> result = new LinkedHashMap<String, Object>();
		if (mData != null) {
			Iterator iter = keyIterator();
			while (iter.hasNext()) {
				ASName key = (ASName)iter.next();
				CosObject value = get(key);
				result.put(key.asString(true), value.getValue());
			}
		}
		return result;
	}

	/**
	 * Returns key corresponding to the value passed here.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	public ASName getKeyForValue(CosObject value)
	throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		try {
			Iterator iter = mData.entrySet().iterator();
			while (iter.hasNext()) {
				Map.Entry entry = (Map.Entry)iter.next();
				CosObject thisValue = (CosObject)entry.getValue();
				if (thisValue instanceof CosObjectRef)
					thisValue = getDocument().resolveReference((CosObjectRef)thisValue);
				if (value == thisValue)
					return (ASName)entry.getKey();
			}
			return null;
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	@Override
	void writeOut(OutputByteStream outStream, boolean inString, boolean inDebugMode)
	throws PDFCosParseException, IOException, PDFSecurityException, PDFIOException
	{
		outStream.write('<');
		outStream.write('<');
		Iterator iter = mData.entrySet().iterator();
		while (iter.hasNext()) {
			Map.Entry entry = (Map.Entry)iter.next();
			((ASName)entry.getKey()).write(outStream);
			CosObject value = repaired ? /*get the repaired value*/get((ASName)entry.getKey()) : (CosObject) entry.getValue();
			if (value.isIndirect() || value instanceof CosBoolean || value instanceof CosNumeric || value instanceof CosNull)
				outStream.write(' ');
			value.writeOut(outStream, inString, inDebugMode);
		}
		outStream.write('>');
		outStream.write('>');
	}
	
	
	/**
	 *  
	 *  This method checks the inside key-value pairs of CosDictionary and returns true if they have same number of 
	 *  key-value pairs and also equal data for each key.
	 *  Returns false if passed CosObject is not an instance of CosDictionary.
	 *  This maintains a continuously growing list of indirect CosObject pairs, which have already been compared, to get rid of infinite recursion.
	 *  This method can take some time if called on large Cos Objects.
	 *  @param value
	 *  @return boolean
	 */
	@Override
	public boolean equals(CosObject value){
		HashMap<Integer, HashSet<Integer>> alreadyComparedCosObjectPairsList = new HashMap<Integer, HashSet<Integer>>();
		return safeEquals(value, alreadyComparedCosObjectPairsList);
	}
	
	/**
	 * 
	 * This method will be used for internal communication only after CosObject.equals() is called.
	 * @param value
	 * @param alreadyComparedCosObjectPairsList which maintains the indirect CosObject pairs which have already been compared.
	 * @return boolean
	 */
	@Override
	boolean safeEquals(CosObject value , HashMap<Integer, HashSet<Integer>> alreadyComparedCosObjectPairsList){
		if(!(value instanceof CosDictionary) || value.getDocument() != this.getDocument())
			return false;
		if(value == this)
			return true;
		
		////if pair successfully added then do further comparisons otherwise return true because this pair is already compared.
		if(!addCosObjectPair(value.getObjNum(), alreadyComparedCosObjectPairsList))
			return true;
		
		CosDictionary dict = (CosDictionary)value;
		Iterator<Entry<ASName, CosObject>> entriesItr1 = mData.entrySet().iterator();
		Iterator<Entry<ASName, CosObject>> entriesItr2 = dict.mData.entrySet().iterator();
		int size1 = 0, size2 = 0;
		ArrayList<ASName> compoundEntriesKeyList = new ArrayList<ASName>();
		ASName key = null;
		CosObject dictValue = null;
		CosObject thisValue = null;
		///// we can't remove null values as it will change original dictionaries so checking the sizes just pretending here like we have removed them.
		//// Here also we are comparing scalar cos objects keeping keys 
		/////corresponding to compound cos objects in list which will be compared later.
		try{
			while(entriesItr1.hasNext()){
				key = entriesItr1.next().getKey();
				if(!((thisValue = this.get(key)) instanceof CosNull)){
						size1++;
					if(thisValue instanceof CosScalar){
						dictValue = dict.get(key);
						if(!thisValue.equals(dictValue))
							return false;
					}else{
						compoundEntriesKeyList.add(key);
					}
				}
			}
			
			while(entriesItr2.hasNext()){
				if(!(entriesItr2.next().getValue() instanceof CosNull))
					size2++;
			}
			
			if(size1 != size2)
				return false;
			
			////// we have reached up to here. This means both dictionaries have same number of key-value pairs and also equal scalar value entries.
			///// Now comparing compound entries (dictionary or stream or array)
			Iterator<ASName> compoundKeysIter = compoundEntriesKeyList.iterator();
			while(compoundKeysIter.hasNext()){
				key = compoundKeysIter.next();
				if(!((CosContainer)(this.get(key))).safeEquals
								(dict.get(key), alreadyComparedCosObjectPairsList))
					return false;			
			}
		}catch(Exception e){
			throw new RuntimeException("problem occured while equating " + value.getObjNum() + " with " + this.getObjNum(), e);
		}
		return true;
	}
	
	/**
	 * 
	 *
	 * Obtains the object corresponding to the specified key. If the object
	 * does not exist, the inheritance chain of this dictionary is walked
	 * using the parentKey to find the parent dictionary. Each parent dictionary
	 * is subsequently searched until either the key is found, or we run out
	 * of parents. An indirect object reference is resolved to the referenced
	 * object before being returned.
	 *
	 * @param name	A key in the dictionary
	 * @param parentKey	The key that specifies the "parent" dictionary
	 *
	 * @return Object corresponding to the specified key.
	 * @throws PDFCosParseException 
	 * @throws PDFIOException 
	 * @throws PDFSecurityException 
	 */
	public CosObject getInheritable(ASName name, ASName parentKey) throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		CosObject result = this.get(name);
		if (result == null || result instanceof CosNull) {
			CosObject parent = this.get(parentKey);
			if (parent instanceof CosDictionary)
			{
				return ((CosDictionary)parent).getInheritable(name, parentKey);
			}
		}
		return result;
	}
}
