/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa;


import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.FindBugsSuppress;
import com.adobe.xfa.ut.MsgFormat;
import com.adobe.xfa.ut.MsgFormatPos;
import com.adobe.xfa.ut.ResId;


/**
 * A base class for all model schema definitions.  Elements that are common
 * to all models may be defined here.
 */
public abstract class Schema {

   	private static final NodeSchema gNullSchema = new NodeSchema(0, 0, 0, 0);
   	
   	/**
	 * Sum of all availability masks.  Same as Dynamic.
	 *
   	 * @exclude from published api.
	 */
	public static final int XFAAVAILABILITY_ALL = 63;
   	/**
   	 * @exclude from published api.
	 */
	public static final int XFAAVAILABILITY_CORE = 1;
   	/*
	 * Masks for availability.
	 */
	/**
   	 * @exclude from published api.
	 */
	public static final int XFAAVAILABILITY_DEPRECATED = 0;
   	/**
   	 * @exclude from published api.
	 */
	public static final int XFAAVAILABILITY_DYNAMIC = 16;
   	/**
   	 * @exclude from published api.
	 */
	public static final int XFAAVAILABILITY_PLUGIN = 8;
   	/**
   	 * @exclude from published api.
	 */
	public static final int XFAAVAILABILITY_XFAF = 4;
   	/**
   	 * @exclude from published api.
	 */
	public static final int XFAAVAILABILITY_XFASUBSET = 2;
   	/**
   	 * @exclude from published api.
	 */
	public static final int XFAAVAILABILITY_XFADESIGNER = 32; //TBD: we need to rework all the availability masks
  	/**
   	 * XFA version 1.0.
   	 */
   	public static final int XFAVERSION_10 = 10;


	/**
   	 * XFA version 2.1.
   	 */
   	public static final int XFAVERSION_21 = 21;
	/**
   	 * XFA version 2.2.
   	 */
   	public static final int XFAVERSION_22 = 22;
	/**
   	 * XFA version 2.3.
   	 */
   	public static final int XFAVERSION_23 = 23;
	/**
   	 * XFA version 2.4.
   	 */
   	public static final int XFAVERSION_24 = 24;
	/**
   	 * XFA version 2.5.
   	 */
   	public static final int XFAVERSION_25 = 25;
	/**
   	 * XFA version 2.6.
   	 */
   	public static final int XFAVERSION_26 = 26;
	/**
   	 * XFA version 2.7.
   	 */
   	public static final int XFAVERSION_27 = 27;
   	/**
   	 * XFA version 2.8.
   	 */
   	public static final int XFAVERSION_28 = 28;
   	/**
   	 * XFA version 2.9.
   	 */
   	public static final int XFAVERSION_29 = 29;
   	/**
   	 * XFA version 3.0.
   	 */
   	public static final int XFAVERSION_30 = 30;
   	/**
   	 * XFA version 3.1.
   	 */
   	public static final int XFAVERSION_31 = 31;
   	/**
   	 * XFA version 3.1.
   	 */
   	public static final int XFAVERSION_32 = 32;
   	/**
   	 * XFA version 3.3.
   	 */
   	public static final int XFAVERSION_33 = 33;
   	/**
   	 * XFA version 3.4.
   	 */
   	public static final int XFAVERSION_34 = 34;   	
   	/**
   	 * XFA version 3.5.
   	 */
   	public static final int XFAVERSION_35 = 35;   	
	/**
   	 * Head XFA version.
   	 */
   	public static final int XFAVERSION_HEAD = XFAVERSION_35; 
	/**
   	 * LocaleSetHead XFA version.
   	 */
   	public static final int XFAVERSION_LOCALESETHEAD = XFAVERSION_27; 
	/**
   	 * SourceSetHead XFA version.
   	 */
   	public static final int XFAVERSION_SOURCESETHEAD = XFAVERSION_28; 
	/**
   	 * ConnectionSetHead XFA version.
   	 */
   	public static final int XFAVERSION_CONNECTIONSETHEAD = XFAVERSION_28; 
	/**
   	 * ConfigurationHead XFA version.
   	 */
   	public static final int XFAVERSION_CONFIGURATIONHEAD = XFAVERSION_35;

	/**
   	 * Obsolete XFA version.
   	 */
   	public static final int XFAVERSION_OBS = XFAVERSION_10;
	
	/**
	 * Return a default (null) schema.
	 *
   	 * @exclude from published api.
	 */
	public static final NodeSchema nullSchema() {
		return gNullSchema;
	}

	private boolean mbInitiated;

	private boolean mbSchemaUsesProtos;

	/** 
     * Eventually these should be lists of foreign schemas...
     * But as long as we have only one implementation of a foreign schema
     * (svg) we'll leave it as a single entry for efficiency
     */
    private Schema      mForeignSchemas;

	/** 
   	 * @exclude from published api.
	 */
	protected final int mnAttributeMax;

	/** 
   	 * @exclude from published api.
	 */
	protected final int mnAttributeMin;

	/** 
   	 * @exclude from published api.
	 */
	protected final int mnElementMax;
	
	/** 
   	 * @exclude from published api.
	 */
	protected final int mnElementMin;

	private final NodeSchema mNodeSchemas[] = new NodeSchema[XFA.XFA_ELEMENT_COUNT];
	
    private final String mSchemaNS;	
	
	/**
	 * Create the base schema class
	 * @param sNS The namespace of the model using this schema. This String must be interned.
   	 *
   	 * @exclude from published api.
	 */
	protected Schema(String sNS,
			 int nAttributeMin /* XFA.XFA_ATTRIBUTE_MIN */, 
			 int nAttributeMax /* XFA.XFA_ATTRIBUTE_MAX */, 
			 int nElementMin /* XFA.XFA_ELEMENT_MIN */, 
			 int nElementMax /* XFA.XFA_ELEMENT_MAX */) {
		
		mnAttributeMin = nAttributeMin;
		mnAttributeMax = nAttributeMax;
		mnElementMin = nElementMin;
		mnElementMax = nElementMax;
		mbSchemaUsesProtos = true;
		mSchemaNS = sNS;
	}

	/**
   	 * @exclude from published api.
	 */
	public void addForeignSchema(Schema schema) {
		assert mForeignSchemas == null || mForeignSchemas == schema;
		
		mForeignSchemas = schema;
	}
	
	/** @exclude from published api */
	static void checkVersion(int eClassTag, Model model, Element parent) {
		if (parent != null && model != null) {
			final ChildRelnInfo info = parent.getNodeSchema().getChildRelnInfo(eClassTag);
			
			// need to call getNodeSchema instead of going to mpElements[] because we could 
			// be adding a foreign node
			if (info != null) {
				if (!model.validateUsage(info.getVersionIntroduced(), info.getAvailability(), true)) {
					// child first then parent
					MsgFormatPos reason = new MsgFormatPos(ResId.InvalidChildVersionException);
					reason.format(XFA.getAtom(eClassTag));
					reason.format(parent.getClassAtom());
					
					if (model.validateUsageFailedIsFatal(info.getVersionIntroduced(), info.getAvailability()))
						throw new ExFull(reason);
					else // warn
						model.addErrorList(new ExFull(reason), LogMessage.MSG_WARNING, parent);
				}
			
				// check if the node is deprecated
				int nTargetVer = model.getCurrentVersion();
				if (info.getVersionDeprecated() != 0 && 
					info.getVersionDeprecated() <= nTargetVer) {
					MsgFormatPos reason = new MsgFormatPos(ResId.DeprecatedChildException);
					reason.format(XFA.getAtom(eClassTag));
					reason.format(parent.getClassAtom());
				
					if (model.isLoading())
						model.addXMLLoadErrorContext(parent, new ExFull(reason));
				}
			}
		}
	}

	/**
	 * Return a default attribute value
	 * 
	 * @param attrName
	 *            the name of the attribute. For efficiency, this name should
	 *            come from XFANamespace
	 * @param eClassTag
	 *            Identify which schema element this attribute belongs to
	 * @return The default value for this attribute.
   	 *
   	 * @exclude from published api.
	 */
	public Attribute defaultAttribute(int attrName, int eClassTag) {

		Schema schema = findSchema(eClassTag);
		if (schema == null) {
			// Nobody recognizes this tag .... error!
			throw new ExFull(ResId.InvalidNodeTypeException, "");
		}
		
		if (schema != this) {
			return schema.defaultAttribute(attrName, eClassTag);
		}

		Attribute attr = null;

		NodeSchema nodeSchema = mNodeSchemas[eClassTag - mnElementMin];
		if (nodeSchema != null) {
			AttributeInfo attrInfo = nodeSchema.getAttributeInfo(attrName);
			if (attrInfo != null)
				attr = attrInfo.getDefault();

		}

		if (attr == null)
			throw new ExFull(ResId.InvalidAttributeException, " ("
					+ XFA.getString(attrName) + STRS.RIGHTBRACE);

		// Note there's no special processing for an invalid attribute name.
		// We're assuming that if we get this far, we don't need to validate.
		return attr;


	}
	
	/**
	 * @return number of elements in this schema definition
	 */
	private final int ELEMENT_COUNT() {
		return mnElementMax - mnElementMin + 1;
	}
	

    /**
     * Find a Foreign schema based on a namespace and the tag of the parent
     * element.
     * @param aNS The namespace of the child element
     * @param eTag The tag of the parent element
     * @return If found, the schema.  If not found, NULL.
	 * @exclude from published api.
     */
	@FindBugsSuppress(code="ES")
	protected Schema findForeignSchema(String aNS, int eTag) {
		if (mForeignSchemas != null && mForeignSchemas.getNS() == aNS)
			return mForeignSchemas;
		
		return null;
	}
	
	/**
 	 * Find the schema associated with this class tag.
 	 * This schema might be the main schema or it could 
 	 * be an associated foreign schema.
 	 * @param classTag The class tag to check.
 	 * @return the associated schema, or nulL if this tag
	 * isn't associated with this schema.
 	 */
	private Schema findSchema(int classTag) {
		if (validTag(classTag))
			return this;
		
		if (mForeignSchemas != null && mForeignSchemas.validTag(classTag))
			return mForeignSchemas;
		
		return null;
	}
	
    /**
     * Get the Atom corresponding to this class tag
     * Note that this is a schema-specific look-up!
     * We no longer assume all class Tags are defined in XFA
     * @param classTag the class tag to look up
     * @return the corresponding Atom
   	 *
   	 * @exclude from published api.
     */
	public String getAtom(int classTag) {
	    Schema schema = findSchema(classTag);
	    if (schema == null) {
	        // Nobody recognizes this tag .... error!
	        throw new ExFull(new MsgFormat(ResId.InvalidNodeTypeException, ""));
	    }

	    if (schema == this)
	        return XFA.getAtom(classTag);

	    return schema.getAtom(classTag);
	}

	/**
     * Find a schema tag given an atom representing an attribute name
     * @param aNS the namespace for this attribute. This String must be interned.
     * @param aAttrName the input attribute name to search for. This String must be interned.
     * @return integer value of the tag. -1 if not found.
     * @exclude from published api.
     */
	@FindBugsSuppress(code="ES")
	public int getAttributeTag(String aNS, String aAttrName) {
	    if (mForeignSchemas != null && aNS == mForeignSchemas.getNS())
	        return mForeignSchemas.getAttributeTag(aNS, aAttrName);

	    return XFA.getTagImpl(aAttrName, Boolean.FALSE);
	}
	

    /**
     * Find a schema tag given an atom representing an element name
     * @param aNS the namespace for this element
     * @param aNodeName the input element name to search for
     * @return tag number. -1 if not found.
     * @exclude from published api.
     */
	@FindBugsSuppress(code="ES")
    public int getElementTag(String aNS, String aNodeName) {
    	
    	if (mForeignSchemas != null && aNS == mForeignSchemas.getNS())
    		return mForeignSchemas.getElementTag(aNS, aNodeName);

    	return XFA.getTagImpl(aNodeName, Boolean.TRUE);
    }

    /**
	 * Find an element tag in the XFA namespace by name given a parent element.
	 *
	 * The default implementation effectively does nothing.  Derived classes
	 * may override this method to do context-sensitive searches.
	 * @param parent the parent element context.
	 * @return the INVALID_ELEMENT tag value.
   	 *
   	 * @exclude from published api.
	 */
	protected int getElementTag(Element parent) {
		// the config schema overrides this method to allow arbitrary syntax in places 
		// where it's allowed in the config file.
		return XFA.INVALID_ELEMENT;
	}

	/**
	 * Create an instance of a node.
	 * 
	 * @param eTag
	 *            The class tag of the XFA node to create.
	 * @param model
	 *            the model where this node will belong
	 * @param parent
	 *            the parent node to add the new child to
	 * @param prevSibling
	 *            the previous sibling -- specifying this makes adding a child
	 *            more efficient
	 * 
	 * @return a new node
   	 *
   	 * @exclude from published api.
	 */
	public final Element getInstance(int eTag, Model model, Element parent, Node prevSibling, boolean bDoVersionCheck) {
		
		assert eTag >= 0;
		assert model != null;
		
		// Check if this class tag is handled by this schema or by one of our 
		// Foreign schemas
		Schema schema = findSchema(eTag);
		if (schema == null) {
			// Nobody recognizes this tag .... error!
			throw new ExFull(ResId.InvalidNodeTypeException, "");
		}
		
		Element element = null;
		if (schema != this) {
			return schema.getInstance(eTag, model, parent, prevSibling, bDoVersionCheck); 
		}
		
		NodeSchema nodeSchema = null;
		if ( eTag >= mnElementMin && eTag <= mnElementMax) {
			nodeSchema = mNodeSchemas[eTag - mnElementMin];
		}
		if ( nodeSchema == null) {
			String className = getAtom(eTag);
			throw new ExFull(ResId.InvalidNodeTypeException, className);
		}
		
		// JavaPort: If we're creating a child to the AppModel (a packet) then don't
		// bother validating it.  We accept all packets.
		boolean bValidate = !(schema instanceof AppSchema);
		
		// if we have a parent see if we can create the node.
		if (parent != null && bValidate) {
			ChildRelnInfo info = parent.getNodeSchema().getChildRelnInfo(eTag);
			
			// need to call getNodeSchema instead of going to mpElements[] because we could 
			// be adding a foreign node
			if (info != null) {
				
				//Do a version check? We don't need to when querying default elements.			
				if (bDoVersionCheck)
					checkVersion(eTag, model, parent);				
			}
			else {
				// JavaPort:
				// Another chance for user-defined config nodes to escape being
				// classified as errors...
				if (schema.getElementTag(parent) == XFA.INVALID_ELEMENT) {
					// parent then child
					MsgFormatPos message = new MsgFormatPos(ResId.InvalidChildAppendException, parent.getClassAtom());
					message.format(XFA.getAtom(eTag));
					throw new ExFull(message);
				}
			}
		}

		String className = getAtom(eTag);
		//
		// Fixed Watson 1480554 -- avoid causing NullPointerExceptions.
		//
		if (mNodeSchemas[eTag] == null) {
			throw new ExFull(ResId.InvalidNodeTypeException, className);
		}
		
		element = newElement(eTag, parent, prevSibling);
		
		if (element == null) {
			
			// These would previously have been added to the schema in initSchema
			
			if (mbSchemaUsesProtos) {
				switch (eTag) {
				
				// TextNode is not an Element!
				// case XFA.TEXTNODETAG:
				
				case XFA.RICHTEXTNODETAG: 
					element = new com.adobe.xfa.RichTextNode(parent, prevSibling);
					break;
					
				case XFA.XMLMULTISELECTNODETAG:
					element = new com.adobe.xfa.XMLMultiSelectNode(parent, prevSibling);
					break;
					
				case XFA.DSIGDATATAG:
					element = new com.adobe.xfa.DSigData(parent, prevSibling);
					break;
				}
			}
			
        	if (element == null)
    			throw new ExFull(ResId.InvalidNodeTypeException, className);
		}
		
		element.setLocalName(className);
		element.setQName(className);
		element.setNS(model);
		
		if (element instanceof GenericNode || element instanceof GenericTextContainer) {
			// Fill in the class information that's otherwise missing for
			// generic nodes
			element.setClass(className, eTag);
		}
		
		// The constructor relies on parent to set the model for a child.
		// If there's no parent specified, fill in the model here.
		if (parent == null) {
			element.setModel(model);
			element.setDocument(model.getDocument());
		}

		return element;
	}

	/**
	 * Get the schema for a node.
	 * 
	 * @param eClassTag
	 *            The name of the schema element to query. This method gets best
	 *            performance if the provided name is from XFANamespace
	 * @return An NodeSchema.
   	 *
   	 * @exclude from published api.
	 */
	public final NodeSchema getNodeSchema(int eClassTag) {
		Schema schema = findSchema(eClassTag);
		if (schema == null)
			return gNullSchema;
		
		NodeSchema nodeSchema = schema.mNodeSchemas[eClassTag - schema.mnElementMin];
			
		if (nodeSchema != null)
			return nodeSchema;
			
		return gNullSchema;
	}
	
   /**
	 * Get the namespace string for the model 
	 * @return an interned namespace string
	 *
	 * @exclude from published api.
	 */
	public String getNS() {
	    return mSchemaNS;
	}	
	
	/**
	 * Return the class tag that's used to represent #text nodes in this schema.
	 * For most schemas this returns XFA::TEXTNODETAG. For the SVG schema it
	 * will return SVG::TEXTDATATAG
	 * 
	 * @return a tag used to represent text data
	 * @exclude from published api.
	 */
	protected int getTextTag() {
		return XFA.TEXTNODETAG;
	}

	/**
	 * Called "import()" in C+++ -- but "import" is a reserved word in Java.
	 * 
	 * @exclude from published api.
	 */
	protected void importSchema(Schema sourceSchema) {
		
		mForeignSchemas = sourceSchema.mForeignSchemas;
		int srcCount = sourceSchema.mnElementMax - sourceSchema.mnElementMin;
		for (int nPos = 0; nPos < srcCount; nPos++) {
			NodeSchema src = sourceSchema.mNodeSchemas[nPos];
			NodeSchema dest;
			if (src == null)
				dest = null;
			else {
				dest = new NodeSchema(src,
									  mnAttributeMin, 
									  mnAttributeMax, 
									  mnElementMin, 
									  mnElementMax);
			}
			
			mNodeSchemas[nPos] = dest;
		}
		
		mbInitiated = true;
	}

	/**
   	 * @exclude from published api.
	 */
	protected void initSchema() {
		// If we have already initialized this schema then don't do it again.
		if (mbInitiated)
			return;

		mbInitiated = true;
		
		if (mbSchemaUsesProtos) {
			//
			// populate the schema relationships
			//
			// add elements defined in XFASchema
			
			// Don't add TextNode since it is not an Element
			//addSample(XFA.TEXTNODETAG, "com.adobe.xfa.TextNode", true);
			
			// These Elements don't have interesting schemas, but they need to be declared
			// so they won't be rejected as invalid when loading.
			mNodeSchemas[XFA.RICHTEXTNODETAG - mnElementMin] = new NodeSchema(mnAttributeMin, mnAttributeMax, mnElementMin, mnElementMax);
			mNodeSchemas[XFA.XMLMULTISELECTNODETAG - mnElementMin] = new NodeSchema(mnAttributeMin, mnAttributeMax, mnElementMin, mnElementMax);
	
			mNodeSchemas[XFA.DSIGDATATAG - mnElementMin] = new NodeSchema(mnAttributeMin, mnAttributeMax, mnElementMin, mnElementMax);
			putAttribute(XFA.DSIGDATATAG, XFA.IDTAG, null, XFAVERSION_10, XFAAVAILABILITY_ALL, /* 0, */ 0);
		}
	}

	/**
	 * Create a new attribute value
	 * 
	 * @param attrName
	 *            The name of the attribute to create. For efficiency, this name
	 *            should come from XFANamespace
	 * @param value
	 *            the value to assign to the new attribute
	 * @param eClassTag
	 *            Identify which schema element this attribute belongs to
	 * @return a new Attribute.
   	 *
   	 * @exclude from published api.
	 */
	public Attribute newAttribute(int attrName, String value, int eClassTag) {
		Schema schema = findSchema(eClassTag);
		if (schema == null) {
			// Nobody recognizes this tag .... error!
			throw new ExFull(ResId.InvalidNodeTypeException, "");
		}
		
		if (schema != this)
			return schema.newAttribute(attrName, value, eClassTag);
		
		
		Attribute attr = null;

		NodeSchema nodeSchema = mNodeSchemas[eClassTag - mnElementMin];
		if (nodeSchema != null) {
			
			AttributeInfo attrInfo = nodeSchema.getAttributeInfo(attrName);
			if (attrInfo != null)
				attr = attrInfo.getDefault();

		}
		
		if (attr == null) {
			throw new ExFull(ResId.InvalidAttributeException, " ("
				+ XFA.getString(attrName) + "=" + value + STRS.RIGHTBRACE);
		}

		return attr.newAttribute(value);
	}
	
	/**
	 * Create a new Element instance. 
	 * This replaces the previous scheme where elements were constructed using Reflection,
	 * which turns out to be very expensive.
	 *
	 * @exclude from published api.
	 */
	protected abstract Element newElement(int eTag,
							           Element parent,
							           Node prevSibling);

	/**
   	 * @exclude from published api.
	 */
	protected void putAttribute(int eParent,
								EnumValue value,
								int nVersionIntro /* = 0 */,
								int nAvailability /* = XFAAVAILABILITY_ALL */,
								// JavaPort: only used by Designer
								// int descriptionResId /* = 0 */, JavaPort: only used by Designer
								int nVersionDep /* = 0 */) {
		
		putAttribute(eParent, value.getAttrTag(), value, nVersionIntro, nAvailability, /* descriptionResId, */ nVersionDep);
	}

	/**
   	 * @exclude from published api.
	 */
	protected void putAttribute(int eParent,
								int eAttr,
								Attribute defaultValue /* = null */,
								int nVersionIntro /* = 0 */,
								int nAvailability /* = XFAAVAILABILITY_ALL */,
								// JavaPort: only used by Designer
								// int descriptionResId /* = 0 */, JavaPort: only used by Designer
								int nVersionDep /* = 0 */) {

		assert (eAttr >= XFA.XFA_PARTIAL_ELEMENT_COUNT);

		NodeSchema nodeSchema = mNodeSchemas[eParent - mnElementMin];
		if (nodeSchema == null) {
			nodeSchema = new NodeSchema(
								mnAttributeMin, 
								mnAttributeMax, 
								mnElementMin, 
								mnElementMax);
			mNodeSchemas[eParent - mnElementMin] = nodeSchema;
		}
			
		nodeSchema.addAttr(
				eAttr, 
				defaultValue == null ? new StringAttr("", "") : defaultValue, 
				nVersionIntro, 
				nVersionDep, 
				nAvailability, 
				0);
	}

	/**
   	 * @exclude from published api.
	 */
	protected void putChildAttrs(int eTag) {
		putAttribute(eTag, XFA.NAMETAG, null, XFAVERSION_10, XFAAVAILABILITY_CORE | XFAAVAILABILITY_XFASUBSET | XFAAVAILABILITY_XFAF, /* 0, */ 0 );
		putAttribute(eTag, XFA.IDTAG, null, XFAVERSION_10, XFAAVAILABILITY_CORE | XFAAVAILABILITY_XFASUBSET | XFAAVAILABILITY_XFAF, /* 0, */ 0 );
		putAttribute(eTag, XFA.USETAG, null, XFAVERSION_10, XFAAVAILABILITY_CORE | XFAAVAILABILITY_XFASUBSET, /* 0, */ 0 );
		putAttribute(eTag, XFA.USEHREFTAG, null, XFAVERSION_26, XFAAVAILABILITY_CORE | XFAAVAILABILITY_XFASUBSET, /* 0, */ 0 ); // Added with CL183802 Mar 15, 2005
	}

	/**
   	 * @exclude from published api.
	 */
	protected void putElement(int eParent,
							  int eChild,
							  ChildReln oRelation,
							  int nVersionIntro /* = 0 */,
							  int nAvailability /* = XFAAVAILABILITY_ALL */,
							  // JavaPort: only used by Designer
							  // int descriptionResId /* = 0 */,
							  int nVersionDep /* = 0 */) {

		assert eChild - mnElementMin < ELEMENT_COUNT();

		NodeSchema nodeSchema = mNodeSchemas[eParent - mnElementMin];
		if (nodeSchema == null) {
			nodeSchema = new NodeSchema(
								mnAttributeMin,
								mnAttributeMax,
								mnElementMin,
								mnElementMax);
			mNodeSchemas[eParent - mnElementMin] = nodeSchema;
		}					

		nodeSchema.addChild(eChild, oRelation, nVersionIntro, nVersionDep, nAvailability, 0);

		if (mbSchemaUsesProtos) {
			NodeSchema protoNodeSchema = mNodeSchemas[XFA.PROTOTAG - mnElementMin];
			if (protoNodeSchema == null) {
				protoNodeSchema = new NodeSchema(
										mnAttributeMin,
										mnAttributeMax,
										mnElementMin,
										mnElementMax);
				mNodeSchemas[XFA.PROTOTAG - mnElementMin] = protoNodeSchema;
			}					

			// only add the child because the parent could be a model
			if (eChild != XFA.PROTOTAG        &&
				eChild != XFA.TEXTNODETAG     &&
				eChild != XFA.RICHTEXTNODETAG &&
				eChild != XFA.XMLMULTISELECTNODETAG) {
				protoNodeSchema.addChild(eChild, ChildReln.getZeroOrMore(), nVersionIntro, nVersionDep, nAvailability, 0);
			}
		}
	}

	/**
   	 * @exclude from published api.
	 */
	public void putForeignElement(int eParent,
								  int eChild,
								  ChildReln relation,
								  Schema foreignSchema,
								  int nVersionIntro /* = 0 */,
								  int nAvailability /* = XFAAVAILABILITY_ALL */,
								  // JavaPort: only used by Designer
								  // int descriptionResId /* = 0 */,
								  int nVersionDep /* = 0 */) {
		
		assert foreignSchema.validTag(eChild);

		NodeSchema nodeSchema = mNodeSchemas[eParent - mnElementMin];
		if (nodeSchema == null) {
			nodeSchema = new NodeSchema(
								mnAttributeMin, 
								mnAttributeMax, 
								mnElementMin, 
								mnElementMax);
			mNodeSchemas[eParent-mnElementMin] = nodeSchema;
		}					

		nodeSchema.addForeignChild(eChild, relation, nVersionIntro, nVersionDep, nAvailability, 0);
	}

 	/**
   	 * @exclude from published api.
	 */
	protected void putPropAttrs(int eTag) {
		putAttribute(eTag, XFA.IDTAG, null, XFAVERSION_10, XFAAVAILABILITY_CORE | XFAAVAILABILITY_XFASUBSET, /* 0, */ 0);
		putAttribute(eTag, XFA.USETAG, null, XFAVERSION_10, XFAAVAILABILITY_CORE | XFAAVAILABILITY_XFASUBSET, /* 0, */ 0);
		putAttribute(eTag, XFA.USEHREFTAG, null, XFAVERSION_26, XFAAVAILABILITY_CORE | XFAAVAILABILITY_XFASUBSET, /* 0, */ 0); // Added with CL183802 Mar 15, 2005

	}
	
	/**
	 * allow a derived schema to dictate whether they get the extra baggage associated with protos
	 * @exclude from published api.
	 */
	protected final void schemaUsesProtos(boolean bUseProto) {
	    mbSchemaUsesProtos = bUseProto;
	}

 	/**
 	 * Validate a class tag.  Check if it's in the range handled by this schema
 	 * @param classTag Class Tag to validate
 	 * @return TRUE if valid
 	 */
 	private boolean validTag(int classTag) {
		return (classTag >= mnElementMin) && (classTag <= mnAttributeMax);
	}
}
