package com.day.cq.dam.commons.thumbnail.provider;

import com.day.cq.commons.thumbnail.ThumbnailProvider;
import com.day.cq.dam.api.Asset;
import com.day.cq.dam.api.Rendition;
import org.apache.commons.lang.StringUtils;
import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Property;
import org.apache.felix.scr.annotations.Service;
import org.apache.jackrabbit.util.Text;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ResourceUtil;
import org.apache.sling.api.resource.ValueMap;

import java.util.Map;
import java.util.regex.Pattern;
import java.util.regex.Matcher;

import static com.day.cq.dam.api.DamConstants.PREFIX_ASSET_THUMBNAIL;

/**
 * The <code>FileThumbnailProvider</code> class does currently only consider nt:files. If the file is part of a DAM
 * Lightbox, it attempts to locate the requested thumbnail within the asset referenced by the lightbox entry.
 */
@Component(metatype = false)
@Service
@Property(name = "thumbnail.provider.name", value = "nt:file")
public class FileThumbnailProvider implements ThumbnailProvider {

    private static final String LIGHBOX_ASSETREF_SEPARATOR = "_:_";
    private static final String PROP_NAME_ASSETREFS = "assetRefs";
    private static final Pattern lightboxPattern = Pattern.compile( "/home/users/(.*)/(.*)/profile/lightbox/(.*)");

    public String getThumbnailPath(Resource resource, int width, int height,
                                   Map<String, Object> additionalConf) {
        // here we could add filetype specific icons...
        // this is not used for the moment
        if (isLightBoxAsset(resource)) {
            return getThumbnailPathForLightboxEntry(resource, width, height);
        }

        return null;
    }

    private String getThumbnailPathForLightboxEntry(final Resource entry, final int width, final int height) {

        String path = null;

        final String lightboxPath = entry.getParent().getPath();
        final ResourceResolver resolver = entry.getResourceResolver();
        final ValueMap lightboxProps = ResourceUtil.getValueMap(resolver.getResource(lightboxPath));
        final String[] assetRefs = lightboxProps.get(PROP_NAME_ASSETREFS, new String[0]);

        for (final String ref : assetRefs) {

            final String[] fragments = StringUtils.splitByWholeSeparator(ref, LIGHBOX_ASSETREF_SEPARATOR);
            if (2 == fragments.length && ResourceUtil.getName(entry).equals(fragments[0])) {

                final String assetPath = fragments[1];
                final Resource assetResource = resolver.getResource(assetPath);
                final Asset asset = assetResource.adaptTo(Asset.class);

                if (null != asset) {
                    final String renditionName = PREFIX_ASSET_THUMBNAIL + "." + width + "." + height + ".png";
                    final Rendition rendition = asset.getRendition(renditionName);
                    if (null != rendition) {
                        path = rendition.getPath();
                    }
                }
            }
        }

        return path;
    }

    private boolean isLightBoxAsset(final Resource resource) {
        Matcher matcher = lightboxPattern.matcher(resource.getPath());
        return matcher.matches()? true : false;
        }
}
