/*
 * Copyright 1997-2009 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */

package com.day.cq.wcm.api.components;

import org.apache.jackrabbit.commons.JcrUtils;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.RepositoryException;

import java.util.LinkedHashMap;
import java.util.Map;

/**
 * Implements the basic inplace editing configuration.
 */
public class InplaceEditingConfig {

    /**
     * Name of property containing the flag if inplace editing is active
     */
    public static final String PN_ACTIVE = "active";

    /**
     * Name of property containing the inplace editor's type
     */
    public static final String PN_EDITOR_TYPE = "editorType";

    /**
     * Name of property containing the path to the editor's configuration node
     */
    public static final String PN_CONFIG_PATH = "configPath";

    /**
     * Default path to the editor-specific config node
     */
    public static final String NN_CONFIG_DEFAULT = "/config";

    /**
     * Default path to the child editor to the child editor's configuration node
     */
    static final String NN_CHILD_EDITORS = "cq:childEditors";

    /**
     * Flag that determines if inplace editing is activated
     */
    private final boolean isActive;

    /**
     * Type of inplace editor
     */
    private final String editorType;

    /**
     * Path to the editor-specific configuration (will be requested by the editor using
     * JSON)
     */
    private final String configPath;

    /**
     * Inplace child editors for subcomponents
     */
    private  Map<String, ChildEditor> childEditors;

    /**
     * Creates a new basic inplace editing configuration from the given configuration
     * {@link Node}.
     * @param configNode The node to generate the basic configuration from.
     * @throws javax.jcr.RepositoryException This exception is thrown for repository access related issues.
     */
    public InplaceEditingConfig(Node configNode) throws RepositoryException {
        this.editorType = JcrUtils.getStringProperty(configNode, PN_EDITOR_TYPE, null);
        this.isActive = (this.editorType != null) && JcrUtils.getBooleanProperty(configNode, PN_ACTIVE, false);
        String configPath = JcrUtils.getStringProperty(configNode, PN_CONFIG_PATH, configNode.getPath() + NN_CONFIG_DEFAULT);
        if (!configPath.startsWith("/")) {
            // resolve relative paths
            String basePath = configNode.getPath();
            configPath = basePath + "/" + configPath;
        }
        this.configPath = configPath;

        //create basic inplace child editors config
        Map<String, ChildEditor> myChildEditors = null;
        if (configNode.hasNode(NN_CHILD_EDITORS)) {
            NodeIterator iter = configNode.getNode(NN_CHILD_EDITORS).getNodes();
            while (iter.hasNext()) {
                if (myChildEditors == null) {
                    myChildEditors = new LinkedHashMap<String, ChildEditor>();
                }
                ChildEditor ce = new ChildEditor(iter.nextNode());
                myChildEditors.put(ce.getId(), ce);
            }
        }
        if (myChildEditors != null) {
            childEditors = myChildEditors;
        }
    }

    /**
     * Creates a new basic inplace editing configuration from the specified configuration.
     * @param config Existing configuration to clone
     */
    public InplaceEditingConfig(InplaceEditingConfig config) {
        this.isActive = config.isActive();
        this.editorType = config.getEditorType();
        this.configPath = config.getConfigPath();
        this.childEditors = config.getChildEditors();
    }

    /**
     * Determines if inplace editing is active for this component.
     * @return <code>true</code> if inplace editing is active for this component
     */
    public boolean isActive() {
        return this.isActive;
    }

    /**
     * Gets the type of inplace editor to be used. The returned <code>String</code> must
     * match a inplace editor module that is registered clientside under that type.
     * Note that {@link #isActive} will return <code>false</code> if no inplace editing
     * module is specified (and getInplaceEditor will return <code>null</code>)
     * @return Type of the inplace module; <code>null</code> if no inplace editing module
     *         is defined
     */
    public String getEditorType() {
        return this.editorType;
    }

    /**
     * Gets the path to the editor-specific configuration. The editor will use this
     * path to load its configuration through an JSON request accordingly. Note that
     * there must not necessarily be a node at the specified path; hence the clientside
     * implementation must internally use suitable default values
     * @return Path to the configuration node
     */
    public String getConfigPath() {
        return this.configPath;
    }

    /**
     * Returns a unmodifiable map of configurations of inplace child editors.
     * the keys are the ids of the configs.
     *
     * @return map of inplace child editors configurations.
     */
    public Map<String, ChildEditor> getChildEditors() { return this.childEditors; }

}
