/* ****************************************************************************
 *
 *	File: DecryptingInputStream.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

// @author itenenbo

package com.adobe.internal.pdftoolkit.core.cos;

import java.io.IOException;
import java.io.InputStream;

import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosParseException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityConfigurationException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityException;
import com.adobe.internal.pdftoolkit.core.securityframework.EncryptionHandler;
import com.adobe.internal.pdftoolkit.core.securityframework.EncryptionHandlerState;
import com.adobe.internal.pdftoolkit.core.types.ASName;
import com.adobe.internal.pdftoolkit.core.util.ByteOps;

/**
 * This class decrypts the content of the specified CosStream before delivering it
 * to the caller.
 */
class DecryptingInputStream extends InputStream
{
	private static final int BUF_LEN = 65536;

	private InputStream mReadStream;
	private EncryptionHandlerState mEncryptionHandler;
	private byte[] mEncrypted = new byte[BUF_LEN];
	private byte[] mSingleByte = new byte[1];
	private byte[] mDecrypted;
	private int mDecryptedPos;
	private boolean mNeedsDecryption;
	private boolean mHasBuffer;
	private boolean mDone;
	private boolean mClosed;

	/**
	 * This constructor uses the supplied {@link EncryptionHandlerState} instance for all its
	 * encryption-related operations. The caller may use 
	 * {@link CosEncryption#getStreamDecryptionStateHandler(CosStream, com.adobe.internal.io.stream.InputByteStream)}
	 * to get reusable EncryptionHandlerState instance. This instance may not be 
	 * used with any other EncryptingOutputStream object or DecryptingInputStream object 
	 * until all reading from the specified CosObject's OutputByteStream completes.
	 * @param cosStream CosStream object from which encrypted content is read 
	 * @param content OutputByteStream associated with the cosStream object.
	 * @param encryptionStateHandler EncryptionHandlerState object used for encryption in this
	 * EncryptingOutputStream object
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 * @throws PDFCosParseException
	 * @throws IOException 
	 */
	DecryptingInputStream(CosStream cosStream, InputStream content, EncryptionHandlerState encryptionStateHandler)
	throws PDFIOException, PDFSecurityException, PDFCosParseException, IOException
	{
		mNeedsDecryption = cosStream.needsDecryption(null);
		mEncryptionHandler = encryptionStateHandler;
		initialize(cosStream, content);
	}
	
	/**
	 * This constructor creates a DecryptingInputStream object with its own copy of the 
	 * {@link EncryptionHandlerState}. It is save to use this instance for asynchronous 
	 * reads from the content InputByteStream.
	 * @param cosStream The CosStream from which the content is read.
	 * @param content The InputByteStream associated with the cosStream.
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 * @throws PDFCosParseException
	 * @throws IOException 
	 */
	DecryptingInputStream(CosStream cosStream, InputStream content)
	throws PDFIOException, PDFSecurityException, PDFCosParseException, IOException
	{
		mNeedsDecryption = cosStream.needsDecryption(null);
		if (mNeedsDecryption)
		{
			ASName cryptFilter = cosStream.getCryptFilter();
			EncryptionHandler streamDecryption = cosStream.getDocument().getEncryption().getStreamDecryptionHandler(cryptFilter == null ? null : cryptFilter.asString(true));
			if (streamDecryption == null)
			{
				throw new PDFSecurityConfigurationException("Cannot find Security Handler for a stream");
			}
			mEncryptionHandler = streamDecryption.createEncryptionHandlerState();
		}
		initialize(cosStream, content);
	}

	@Override
	public int read() 
	throws IOException
	{
		if (read(mSingleByte) != 1)
		{
			return -1;
		}
		return mSingleByte[0];
	}

	@Override
	public int read(byte[] buf)
	throws IOException
	{
		return read(buf, 0, buf.length);
	}
	
	@Override
	public int read(byte[] buf, int off, int len)
	throws IOException
	{
		if (mClosed)
		{
			throw new IOException("Cannot read from a closed stream");
		}
		if (!mNeedsDecryption)
		{
			return mReadStream.read(buf, off, len);	
		}
		if (!readBuf())
		{
			return -1;
		}
		int retLen = ByteOps.copy(mDecrypted, mDecryptedPos, buf, off, len);
		mDecryptedPos += retLen;
		if (mDecryptedPos >= mDecrypted.length)
		{
			mHasBuffer = false;
		}
		return retLen;
	}
	
	private void initialize(CosStream cosStream, InputStream content)
	throws PDFIOException, IOException, PDFSecurityException, PDFCosParseException
	{
		mReadStream = content;
		int inLen = mReadStream.read(mEncrypted);
		if (mNeedsDecryption)
		{
			if (mEncryptionHandler == null)
			{
				throw new PDFSecurityConfigurationException("Cannot find Security Handler for a stream");
			}
			mDecrypted = mEncryptionHandler.init(mEncrypted, 0, inLen,
					cosStream.getDocument().getEncryption().getStreamEncryptionKey(cosStream, false), 
					EncryptionHandlerState.DECRYPT);
			if (mDecrypted == null)
				mDecrypted = new byte[0];
		}
		mDecryptedPos = 0;
		mHasBuffer = true;
		mDone = false;
		mClosed = false;
	}
	
	private boolean readBuf()
	throws IOException
	{
		if (mDone)
		{
			return false;
		}
		if (!mHasBuffer)
		{
			try
			{
				int inLen = mReadStream.read(mEncrypted);
				if (inLen < 0)
				{
					mDecrypted = mEncryptionHandler.finish();
					/*
					 * Calling finish means there will be no data left for
					 * decryption after this and since there is no more data in
					 * stream to be read we can mark mDone as true
					 */
					mDone = true;
				}
				else
					mDecrypted = mEncryptionHandler.update(mEncrypted, 0, inLen);
			} catch (PDFSecurityException e)
			{
				IOException excp = new IOException();
				excp.initCause(e);
				throw excp;
			}
			mDecryptedPos = 0;
			mHasBuffer = true;
		}
		return true;
	}
	
	/**
	 * closes associated mReadStream
	 */
	@Override
	public void close() throws IOException
	{
		if(mReadStream != null)
			mReadStream.close();
		mClosed = true;
	}
	
	@Override
	public int available() throws IOException {
		if(!mDone && !mClosed){// neither closed nor finished with encryption state handler.
			int approxBytes = (mDecrypted.length - mDecryptedPos -1);
			if(approxBytes <= 0)
				return 1;// this is because we may still have some bytes which are not even read and present in encrypted stream
						 // OR we may get some bytes when we call doFinal() on mEncryptionHandler.
			else
				return approxBytes;
				
		}
		return 0;
	}
}
