/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Arrays;

// The description for these unsupported options is parked here until such
// time as we implement them.

//* 
//* <dt> Char Range Min
//* <dd> Any characters greater than or equal to this char are
//* encoded with their entity references.
//* <dd> This parameter defaults to the null character (U+00).
//* 
//* <dt> Char Range Max
//* <dd> Any characters less than or equal to this char are encoded
//* with their entity references.
//* <dd> This parameter defaults to the null character (U+00).
//* 
//* <dt> Entity Chars
//* <dd> A list of individual characters to be represent with entity
//* references. The default list will vary according to whether we're writing
//* attributes or text values. This list will be added to the default list.
//* <dd> This parameter defaults to "";
//* 
//* <dt> Disable Entity Chars
//* <dd> Disables the following characters to be encoded with
//* their entity references.
//* <dd> This parameter defaults to "";


/**
 * A class to define DOM saving options.  These include:
 * <dl>
 * <dt> Display Format
 * <dd> When set to RAW_OUTPUT, the document is written unmodified to
 * the output stream.
 * <dd> When set to SIMPLE_OUTPUT, the document is written to the output stream
 * with carriage returns added. This output format does not add white space on
 * subsequent reading.
 * <dd> When set to PRETTY_OUTPUT, the document is written with
 * carriage returns and indented by three spaces (by default) for each node level. This
 * output format will add white space on subsequent reading.
 * <dd> This parameter defaults to SIMPLE_OUTPUT.
 * 
 * <dt> Exclude Preamble
 * <dd> When set to <code>true</code>, this suppresses output of the XML
 * <code>&lt;?xml version="1.0" ... ?&gt;</code>
 * preamble.
 * <dd> This parameter defaults to <code>false</code>.
 * 
 * <dt> Include DTD
 * <dd> When set to <code>true</code>, this suppresses output of the XML DTD
 * declarations.
 * <dd> This parameter defaults to <code>false</code> which will cause the omission
 * of the DTD declarations.
 * 
 * <dt> Indent Level
 * <dd> Specifies the number of spaces to indent for each nested level
 * during pretty print.
 * <dd> This parameter defaults to 3.
 * 
 * <dt> Format Nodes Outside The Document Element
 * <dd> When set to <code>true</code>, a newline (U+0a) is rendered
 * before and after processing instructions and
 * comments which occur outside the document element.
 * <dd> This parameter defaults to <code>false</code>, but defaults to <code>true</code> if
 * PRETTY_OUTPUT is used.
 * 
 * <dt> Expand Elements
 * <dd> When set to <code>true</code>, empty elements are represented by an
 * opening tag and a closing tag.
 * <dd> When set to <code>false</code>, the element is collapsed
 * when empty.
 * <dd> This parameter defaults to <code>false</code>.
 * 
 * <dt> Use Single Quotes
 * <dd> When set to <code>true</code>, attribute values are enclosed with
 * single quotes (name='myName').
 * <dd> When set to <code>false</code>, attribute values are enclosed by
 * double quotes (name="myName").
 * <dd> This parameter defaults to <code>false</code>.
 * 
 * <dt> Use UTF-8 Encoding
 * <dd> When set to <code>true</code>, the output is UTF-8 encoded.
 * <dd> When set to <code>false</code>, the output is encoded in some internal default.
 * <dd> This parameter defaults to <code>false</code>.
 * </dl>
 */
public final class DOMSaveOptions {

	/**
	 * A display format with carriage returns, indented lines and white
	 * space added
	 */
	public static final int PRETTY_OUTPUT = 2;

	/**
	 * A display format where the document is written without any formatting modifications
	 */
	public static final int RAW_OUTPUT = 0;

	/**
	 * A display format with carriage returns added, but no added white
	 * space
	 */
	public static final int SIMPLE_OUTPUT = 1;
	

	private boolean mbExcludePreamble;

	private boolean mbExpandElement;

	private boolean mbFormatOutside;

	private boolean mbIncludeDTD;

	private boolean mbSaveFragment;

	private boolean mbSaveTransient;

	private boolean mbUseSingleQuoteAttr;
	
	private boolean mbCanonicalizeNamespaceOrder;	// Probably want to default to TRUE for Acrobat 10

	private char mcRangeMax;
	
	private char mcRangeMin;
	
	private int mePrintFmt;
	
	private int mnIndentLevel;

	private String msDisabledEntityChars;

	private String msEntityChars;
	
	private static volatile byte[] mIndent;
	private final Object sync = new Object();

	/**
	 * Instantiates a object with default DOM save options.
	 */
	public DOMSaveOptions() {
		mePrintFmt = SIMPLE_OUTPUT;
		mnIndentLevel = 3;
		msEntityChars = "";
		msDisabledEntityChars = "";
	}

	/**
	 * Copy constructor
	 *
	 * @exclude from published api.
	 */
	public DOMSaveOptions(DOMSaveOptions oOptions) {
		mePrintFmt = oOptions.mePrintFmt;

		mbExcludePreamble = oOptions.mbExcludePreamble;
		mbIncludeDTD = oOptions.mbIncludeDTD;
		mbFormatOutside = oOptions.mbFormatOutside;
		mbExpandElement = oOptions.mbExpandElement;
		mbUseSingleQuoteAttr = oOptions.mbUseSingleQuoteAttr;
		mbSaveTransient = oOptions.mbSaveTransient;
		mbSaveFragment = oOptions.mbSaveFragment;
		
		mnIndentLevel = oOptions.mnIndentLevel;
		msEntityChars = oOptions.msEntityChars;
		mcRangeMin = oOptions.mcRangeMin;
		mcRangeMax = oOptions.mcRangeMax;
		msDisabledEntityChars = oOptions.msDisabledEntityChars;
		mbCanonicalizeNamespaceOrder = oOptions.mbCanonicalizeNamespaceOrder;
	}

	/**
	 * Instantiates a object with the given DOM save options.
	 * @param ePrintFmt the serialization format.
	 * @param bExcludePreamble exclude the preamble or not.
	 * @param bIncludeDTD include the DTD or not.
	 * @param nIndentLevel the indentation level.
	 * @param bFormatOutside format nodes outside the document element or not.
	 * @param bExpandElement expand elements or not.
	 * @param bUseSingleQuoteAttr use single quotes or not.
	 * @param sEntityChars characters to encode
	 * using entity references [Not currently supported]
	 * @param cRangeMin minimum range of characters
	 * to encode using entity references [Not currently supported]
	 * @param cRangeMax maximum range of characters
	 * to encode using entity references [Not currently supported]
	 * @param sDisabledEntityChars characters not to encode
	 * using entity references [Not currently supported]
	 * @param bIsUTF8 use UTF-8 encoding or not.
	 */
	public DOMSaveOptions(int ePrintFmt,
						  boolean bExcludePreamble,
						  boolean bIncludeDTD,
						  boolean bFormatOutside,
						  int nIndentLevel,
						  boolean bExpandElement,
						  boolean bUseSingleQuoteAttr,
						  String sEntityChars,
						  char cRangeMin,
						  char cRangeMax,
						  String sDisabledEntityChars,
						  boolean bIsUTF8,
						  boolean bSaveTransient,
						  boolean bSaveFragment) {
		mePrintFmt = ePrintFmt;
		mnIndentLevel = nIndentLevel;
		msEntityChars = sEntityChars;
		mcRangeMin = cRangeMin;
		mcRangeMax = cRangeMax;
		msDisabledEntityChars = sDisabledEntityChars;
		mbSaveTransient = bSaveTransient;
		mbSaveFragment = bSaveFragment;

		mbExcludePreamble = bExcludePreamble;
		mbIncludeDTD = bIncludeDTD;
		mbFormatOutside = bFormatOutside;
		mbExpandElement = bExpandElement;
		mbUseSingleQuoteAttr = bUseSingleQuoteAttr;
		//mbCanonicalizeNamespaceOrder = false; // Probably want to default to TRUE for Acrobat 10
	}
	
	/**
	 * Determine if an element/attribute should be saved based on its state and save options
	 * @param bFragmentFlag element/attribute fragment flag
	 * @param bDefaultFlag element/attribute default flag
	 * @param bTransientFlag element/attribute transient flag
	 * @return <code>true</code> if it can be saved
	 * @exclude from published api.
	 */
	boolean canBeSaved(boolean bFragmentFlag, boolean bDefaultFlag, boolean bTransientFlag) {
	    // if save fragment is on then save regardless of the default or transient setting.
	    if (!bFragmentFlag || !getSaveFragment()) {
	        if ((bTransientFlag || bDefaultFlag) && !getSaveTransient())
	            return false;
	    }
	    return true;
	}
	
	/**
	 * Gets the flag indicating if namespace attributes are to be saved out
	 * in whatever order they exist in, or in the order specified by XML
	 * canonicalization (default namespace first, followed by namespace
	 * prefix definitions in lexicographical order).
	 * @return jfBool indicating if namespaces are to be saved out in
	 * canonical order.
	 * @exclude from published api.
	 */
	public boolean getCanonicalizeNamespaceOrder () {
		return mbCanonicalizeNamespaceOrder;
	}

	/**
	 * Returns a string containing the characters that will not be encoded with their entity
	 * references.
	 * @return a String containing the characters that will not be encoded with
	 * their entity references
	 * 
	 * @exclude from published api.
	 */
	public String getDisabledEntityChars() {
		return msDisabledEntityChars;
	}

	/**
	 * Returns the display format.
	 * @return one of: {@link #PRETTY_OUTPUT}, {@link #SIMPLE_OUTPUT} or {@link #RAW_OUTPUT}
	 */
	public int getDisplayFormat() {
		return mePrintFmt;
	}

	/**
	 * Gets the string of characters which will be encoded with their entity
	 * references.
	 * @return the String of characters which will be encoded with their
	 * entity references
	 * 
	 * @exclude from published api.
	 */
	public String getEntityChars() {
		return msEntityChars;
	}

	/**
	 * Determines if the <&#x3f;xml version="1.0" &#x3f;> preamble is excluded.
	 * @return <code>true</code> if the XML declaration preamble is excluded, else <code>false</code>.
	 */
	public boolean getExcludePreamble() {
		return mbExcludePreamble;
	}

	/**
	 * Determines if elements are output in their expanded form if empty.
	 * @return <code>true</code> if empty elements are written with open and closing tags
	 */
	public boolean getExpandElement() {
		return mbExpandElement;
	}

	/**
	 * Determines if nodes outside the the document element are formatted.
	 * @return <code>true</code> if nodes outside the document element are formatted,
	 * else <code>false</code>
	 */
	public boolean getFormatOutside() {
		return mbFormatOutside;
	}

	/**
	 * Determines if DTD declarations are included.
	 * @return <code>true</code> if DTD declarations
	 * are included, else <code>false</code>
	 */
	public boolean getIncludeDTD() {
		return mbIncludeDTD;
	}

	/**
	 * Gets the number of spaces used for each level of indentation.
	 * @return the number of spaces used for each level of indentation.
	 */
	public int getIndentLevel() {
		return mnIndentLevel;
	}

	/**
	 * Gets the character which indicate what the end of the range of characters
	 * to be encoded with their references.
	 * @return the max character in the
	 * range
	 *
	 * @exclude from published api.
	 */
	public char getRangeMax() {
		return mcRangeMax;
	}

	/**
	 * Gets the character which indicate what the start of the range of
	 * characters to be encoded with their references.
	 * @return the min character
	 * in the range
	 *
	 * @exclude from published api.
	 */
	public char getRangeMin() {
		return mcRangeMin;
	}

	/**
	 * Determines if transient fragment nodes are written. Fragments are created
	 * when an external proto reference is resolved. When a fragment is
	 * resolved, it is only marked as transient if
	 * {@link AppModel#setExternalProtosAreTransient(boolean)} has been used to
	 * indicate that external protos are transient.
	 * 
	 * @return <code>true</code> if transient fragment nodes are written.
	 * @see #setSaveFragment(boolean)
	 */
	public boolean getSaveFragment() {
		return mbSaveFragment;
	}

	/**
	 * Determines if transient/default nodes are written
	 * @return <code>true</code> if transient/default nodes are written.
	 */
	public boolean getSaveTransient() {
		return mbSaveTransient;
	}

	/**
	 * Determines whether attribute values are enclosed with single quotes.
	 * @return <code>true</code> if attribute values are enclosed by "'",
	 * <code>false</code> if attribute values are enclosed by """
	 */
	public boolean getUseSingleQuoteAttr() {
		return mbUseSingleQuoteAttr;
	}

	/**
	 * Sets the flag indicating if namespace attributes are to be saved out
	 * in whatever order they exist in, or in the order specified by XML
	 * canonicalization (default namespace first, followed by namespace
	 * prefix definitions in lexicographical order).
	 * @exclude from published api.
	 */
	public void setCanonicalizeNamespaceOrder (boolean bCanonicalizeNamespaceOrder) {
		mbCanonicalizeNamespaceOrder = bCanonicalizeNamespaceOrder;
	}
	
	/**
	 * Sets the string of characters which will not be encoded with their entity
	 * references.
	 * @param sEntityChars the string of characters which will not be
	 * encoded
	 *
	 * @exclude from published api.
	 */
	public void setDisabledEntityChars(String sEntityChars) {
		msDisabledEntityChars = sEntityChars;
	}

	/**
	 * Sets the display format.
	 * @param ePrintFmt one of: {@link #PRETTY_OUTPUT}, {@link #SIMPLE_OUTPUT} or {@link #RAW_OUTPUT}
	 */
	public void setDisplayFormat(int ePrintFmt) {
		mePrintFmt = ePrintFmt;
	}

	/**
	 * Sets the string of characters which will be encoded with their entity
	 * references.
	 * @param sEntityChars String of characters which will be encoded
	 * 
	 * @exclude from published api.
	 */
	public void setEntityChars(String sEntityChars) {
		msEntityChars = sEntityChars;
	}

	/**
	 * Determines whether the <&#x3f;xml version="1.0" &#x3f;> preamble is excluded.
	 * @param bExcludePreamble
	 * <code>true</code> if the preamble is excluded else <code>false</code>
	 */
	public void setExcludePreamble(boolean bExcludePreamble) {
		mbExcludePreamble = bExcludePreamble;
	}

	/**
	 * Determines whether elements are written in their expanded form if empty.
	 * @param bExpandElement <code>true</code> if empty elements written with
	 * open and closing tags
	 */
	public void setExpandElement(boolean bExpandElement) {
		mbExpandElement = bExpandElement;
	}

	/**
	 * Determines whether nodes outside the the document element are formatted.
	 * @param bFormatOutside <code>true</code> if nodes outside the document element
	 * are formatted else <code>false</code>.
	 */
	public void setFormatOutside(boolean bFormatOutside) {
		mbFormatOutside = bFormatOutside;
	}

	/**
	 * Sets if DTD declarations are included.
	 * @param bIncludeDTD <code>true</code> if DTD
	 * declarations are included else <code>false</code>.
	 */
	public void setIncludeDTD(boolean bIncludeDTD) {
		mbIncludeDTD = bIncludeDTD;
	}

	/**
	 * Sets the number of spaces used for each level of indentation.
	 * @param nIndentLevel the number of spaces.
	 */
	public void setIndentLevel(int nIndentLevel) {
		mnIndentLevel = nIndentLevel;
	}

	/**
	 * Sets the character which indicate what the end of the range of characters
	 * to be encoded with their references.
	 * @param cRangeMax the max character in the range.
	 *
	 * @exclude from published api.
	 */
	public void setRangeMax(char cRangeMax) {
		mcRangeMax = cRangeMax;
	}
	
	/**
	 * Sets the character which indicate what the start of the range of
	 * characters to be encoded with their references.
	 * @param cRangeMin the min character in the range.
	 *
	 * @exclude from published api.
	 */
	public void setRangeMin(char cRangeMin) {
		mcRangeMin = cRangeMin;
	}

	/**
	 * Determines if transient fragment nodes are written. Fragments are created
	 * when an external proto reference is resolved. When a fragment is
	 * resolved, it is only marked as transient if
	 * {@link AppModel#setExternalProtosAreTransient(boolean)} has been used to
	 * indicate that external protos are transient.
	 * 
	 * @param bSaveFragment <code>true<code> if transient fragment nodes are written.
	 * @see #getSaveFragment()
	 */
	public void setSaveFragment(boolean bSaveFragment) {
		mbSaveFragment = bSaveFragment;
	}

	/**
	 * Sets the flag indicating if transient/default nodes are saved out.
	 *
	 * @exclude from published api.
	 */
	public void setSaveTransient(boolean bSaveTransient) {
		mbSaveTransient = bSaveTransient;
	}

	/**
	 * Determines whether attribute values are enclosed with single quotes.
	 * @param bUseSingleQuoteAttr <code>true</code> if attribute values are enclosed by "'",
	 * <code><code>false</code></code> if by """
	 */
	public void setUseSingleQuoteAttr(boolean bUseSingleQuoteAttr) {
		mbUseSingleQuoteAttr = bUseSingleQuoteAttr;
	}

	/**
	 * @exclude from published api.
	 */
	void writeIndent(OutputStream out, int level) throws IOException {
		if (mePrintFmt == PRETTY_OUTPUT) {
			out.write('\n');
			
			// This uses a statically allocated buffer filled with ' '
			// to avoid calling OutputStream.write repeatedly, which is
			// expensive. 
			// Accesses to mIndent are NOT synchronized since one array of
			// blanks of a given length is equivalent to any other for
			// our purposes.
			
			// This code also made the assumption that ' ' is the same
			// ASCII character in all encodings.
			
			final int n = level * getIndentLevel();
			byte[] indent = mIndent;
			
			if (indent == null || indent.length < n) {
				
				synchronized (sync) {
					indent = mIndent;
					if (indent == null || indent.length < n) {
						indent = new byte[n];
						Arrays.fill(indent, (byte)' ');
						mIndent = indent;
					}
				}
			}
			
			out.write(indent, 0, n);
		}
	}
}
