/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */

package com.adobe.xfa;

import java.util.ArrayList;
import java.util.List;

import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.MsgFormatPos;
import com.adobe.xfa.ut.ResId;
import com.adobe.xfa.ut.StringUtils;
import com.adobe.xfa.ut.UuidFactory;

/**
 * While this was once a map used to uniquify IDs and ID references when importing
 * template fragments, we now use a much simpler strategy: for each fragment reference,
 * generate a unique prefix (stored here), and prepend all IDs and ID references in
 * the fragment with the unique prefix. 
 * Note: since this class maintains the state of the
 * associated model (once installOnModel() has been called), it is only intended
 * to be used as a stack variable. Any other order of destruction could mess up
 * the state of the model's IDValueMap pointer.
 * @exclude
 */
public final class IDValueMap {
	
	private Model mpoInstalledModelImpl;
	private IDValueMap mpoModelPreviousIDValueMap;
	
	private String msPrefix;
	private final List<String> mReferenceList = new ArrayList<String>();

	public IDValueMap() {
	}
	
	public void destroy() {
		if (mpoInstalledModelImpl != null)
			mpoInstalledModelImpl.setIDValueMap(mpoModelPreviousIDValueMap);
	}

	public void installOnModel(Model poModelImpl) {
		mpoModelPreviousIDValueMap = poModelImpl.getIDValueMap();
		mpoInstalledModelImpl = poModelImpl;
		mpoInstalledModelImpl.setIDValueMap(this);
	}
	
	public String getPrefix() {
		if (StringUtils.isEmpty(msPrefix)) {
			msPrefix = UuidFactory.getUuid();
		}
		return msPrefix;
	}

	public List<String> getReferenceList() {
		return mReferenceList;
	}

	public boolean isActive() {
		return mpoInstalledModelImpl != null;
	}

	/**
	 * Logs a warning if the IDs in this list are not self-contained; meaning there
	 * is a "use" reference for which no matching "id" attribute has been seen in the
     * given fragment tree.
	 */
	public boolean verifySelfContained(String sHRefExpression, String sNodeContainingHRef, Node srcNode) {

		if (mpoInstalledModelImpl == null)
			return true;

		for (int i = 0; i < mReferenceList.size(); i++) {
			String sReference = mReferenceList.get(i);
			// Make sure there's a target in the source document:
			Node targetNode = srcNode.getModel().getNode(sReference);
			// Make sure the target is under the source node:
			while (targetNode != null && targetNode != srcNode)
				targetNode = targetNode.getXMLParent();
				
			if (targetNode == null) {
				MsgFormatPos message = new MsgFormatPos(ResId.FragmentIDNotSelfContained);
				message.format(sReference);
				message.format(sHRefExpression);
				message.format(sNodeContainingHRef);

				mpoInstalledModelImpl.addErrorList(new ExFull(message), LogMessage.MSG_WARNING, mpoInstalledModelImpl);
				return false;
			}
		}
		return true;
	}
}
