/*
 * Copyright 1997-2011 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.dam.api;

import java.util.Arrays;
import java.util.Date;
import java.util.Dictionary;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.osgi.service.event.Event;

/**
 * The <code>DamEvent</code> represents events occurring upon assets within DAM. Such events are sent out from
 * implementations performing any of the specified event {@link com.day.cq.dam.api.DamEvent.Type}s.
 */
public class DamEvent {

    /**
     * Event topic for DAM events.
     */
    public static final String EVENT_TOPIC = "com/day/cq/dam";

    /**
     * The type of event.
     */
    private static final String PROPERTY_TYPE = "type";

    /**
     * The date the event occurred on.
     */
    private static final String PROPERTY_DATE = "date";

    /**
     * The expiration date of asset.
     */
    public static final String PROPERTY_EXPIRATIONDATE = "expirationDate";

    /**
     * The creator of asset.
     */
    public static final String PROPERTY_CREATEDBY = "createdBy";

    /**
     * The path of the asset the event occurred upon.
     */
    private static final String PROPERTY_ASSETPATH = "assetPath";

    /**
     * The user that caused the event.
     */
    private static final String PROPERTY_USER_ID = "userId";

    /**
     * The version created for the asset.
     */
    public static final String PROPERTY_VERSION_ID = "versionId";

    /**
     * The license information (e.g. URL/path) that served for accepting/rejecting events.
     */
    public static final String PROPERTY_LICENSE_INFO = "licenseInfo";

    /**
     * Arbitrary additional information settable for the event.
     */
    public static final String PROPERTY_ADDITIONAL_INFO = "additionalInfo";
    
    private static final String PROPERTY_DISTRIBUTABLE = "event.distribute";

    private static final Map<String, Class<?>> propertyTypesMap = new HashMap<String, Class<?>>() {{
        put(PROPERTY_TYPE, Type.class);
        put(PROPERTY_ASSETPATH, String.class);
        put(PROPERTY_USER_ID, String.class);
        put(PROPERTY_VERSION_ID, String.class);
        put(PROPERTY_LICENSE_INFO, String.class);
        put(PROPERTY_ADDITIONAL_INFO, String.class);
        put(PROPERTY_DATE, Date.class);
        put(PROPERTY_EXPIRATIONDATE, Date.class);
        put(PROPERTY_CREATEDBY, String.class);
    }};

    private final Map<String, Object> propertiesMap = new HashMap<>();

    /**
     * A list of events that are supported by assets.
     */
    public static enum Type {
        /**
         * This event type represents the DRM license of an asset having been accepted.
         */
        ACCEPTED,

        /**
         * This event type represents the DRM license of an asset having been rejected.
         */
        REJECTED,

        /**
         * This event type represents the asset having been downloaded (uncached).
         */
        DOWNLOADED,

        /**
         * This event type represents an asset just having been versioned.
         */
        VERSIONED,

        /**
         * This event type represents an asset just having been restored from a previous version.
         */
        RESTORED,

        /**
         * This event type represents the asset's metadata having been updated.
         */
        METADATA_UPDATED,

        /**
         * This event type represents the asset having been uploaded to an external system. In this case the
         * "additionalInfo" property can be checked for more info.
         */
        PUBLISHED_EXTERNAL,

        /**
         * This event type represents the asset's original having been updated/replaced.
         */
        ORIGINAL_UPDATED,

        /**
         * This event type represents a rendition having been added or updated for the given asset.
         */
        RENDITION_UPDATED,

        /**
         * This event type represents a rendition having been removed from the given asset.
         */
        RENDITION_REMOVED,

        /**
         * This event type represents the asset's rendition been downloaded (uncached).
         */
        RENDITION_DOWNLOADED,

        /**
         * This event type represents a sub-asset having been added or updated for the given asset.
         */
        SUBASSET_UPDATED,

        /**
         * This event type represents a sub-asset having been removed for the given asset.
         */
        SUBASSET_REMOVED,

        /**
         * This event type represents an asset having been created.
         */
        ASSET_CREATED,

        /**
         * This event type represents an asset having been moved
         */
        ASSET_MOVED,

        /**
         * This event type represents an asset having been removed
         */
        ASSET_REMOVED,

        /**
         * This event type represents an asset after expiration. By default 
         * asset expiration is based on jcr:content/offTime
         */
        ASSET_EXPIRED,

        /**
         * This event type represents an asset getting expired in future. By default 
         * asset expiration is based on jcr:content/offTime
         */
        ASSET_EXPIRING,
        
        /**
         * This event type represents an asset being viewed by visiting the
         * assetdetails page.
         */
        ASSET_VIEWED,

        /**
         * This event type represents an asset being shared on the different
         * solutions such as marketing cloud or creative cloud.
         */
        ASSET_SHARED,

        /**
         * This event type represents an asset being published.
         */
        ASSET_PUBLISHED,

        /**
         * This event type represents an asset being commented upon
         */
        ADDED_COMMENT,

        /**
         * This event type represents a project's details being viewed
         */
        PROJECT_VIEWED,

        /**
         * This event type represents a collection being viewed
         */
        COLLECTION_VIEWED,
        
        /**
         * This event type represents an asset has been used in some third party solution or in 
         * <code>Collection</code>.
         */
        ASSET_USAGE,

        /**
         * This event type represents completion of asset processing.  Initially, this was always provided by the DAM
         * update asset workflow.  In cloud environments, this may also be performed by the asset processor service.
         */
        DAM_UPDATE_ASSET_WORKFLOW_COMPLETED,

        /**
         * This event type represents completion of DAM metadata writeback workflow.
         */
        DAM_METADATA_WRITEBACK_WORKFLOW_COMPLETED
    }

    @SuppressWarnings("unchecked")
    private <T> T getTypedProperty(String key, Class<T> type) {
        Object propVal = propertiesMap.get(key);
        if (null == propVal) {
            return null;
        }
        if (type.isAssignableFrom(propVal.getClass())) {
            return (T) propVal;
        }
        return null;
    }

    @SuppressWarnings("unchecked")
    private <T> T getTypedPropertyFromEvent(String key, Class<T> type, Event event) {
        Object propVal = event.getProperty(key);
        if (null == propVal) {
            return null;
        }
        if (type.isAssignableFrom(propVal.getClass())) {
            return (T) propVal;
        }
        return null;
    }

    private DamEvent(final Type type,
                     final String assetPath,
                     final String userId,
                     final String licenseInfo,
                     final String versionId,
                     final String additionalInfo,
                     final Date date,
                     final Date expirationDate,
                     final String createdBy) {

        propertiesMap.put(PROPERTY_TYPE, type);
        propertiesMap.put(PROPERTY_ASSETPATH, assetPath);
        propertiesMap.put(PROPERTY_USER_ID, userId);
        propertiesMap.put(PROPERTY_VERSION_ID, versionId);
        propertiesMap.put(PROPERTY_LICENSE_INFO, licenseInfo);
        propertiesMap.put(PROPERTY_ADDITIONAL_INFO, additionalInfo);
        propertiesMap.put(PROPERTY_DATE, (null != date) ? date : new Date());
        propertiesMap.put(PROPERTY_EXPIRATIONDATE, expirationDate);
        propertiesMap.put(PROPERTY_CREATEDBY, createdBy);
    }

    private DamEvent(final Event event) {
        Arrays.stream(event.getPropertyNames())
            .filter(name -> !StringUtils.startsWith(name, "event."))
            .forEach(key -> {
                Class<?> clazz = propertyTypesMap.get(key);
                if (null == clazz) {
                    clazz = String.class;
                }
                if (null != getTypedPropertyFromEvent(key, clazz, event)) {
                    propertiesMap.put(key, getTypedPropertyFromEvent(key, clazz, event));
                }
            });
    }

    /**
     * The path of the asset the event occurred upon.
     *
     * @return A <code>String</code> representing the asset, folder or collection path.
     */
    public String getAssetPath() {
        return getTypedProperty(PROPERTY_ASSETPATH, String.class);
    }

    /**
     * The date the event occurred on.
     *
     * @return The {@link Date} representing the date/time the event occurred on.
     */
    public Date getDate() {
        return getTypedProperty(PROPERTY_DATE, Date.class);
    }

    /**
     * Information about the license upon which the user accepted or rejected.
     *
     * @return A <code>String</code> representing the license information, or <code>null</code> if n/a.
     */
    public String getLicenseInfo() {
        return getTypedProperty(PROPERTY_LICENSE_INFO, String.class);
    }

    /**
     * The ID of the version that was created upon the asset versioning event.
     *
     * @return A <code>String</code> representing the version ID, or <code>null</code> if n/a.
     */
    public String getVersionId() {
        return getTypedProperty(PROPERTY_VERSION_ID, String.class);
    }

    /**
     * The id of the user generating the event.
     *
     * @return A <code>String</code> representing the user id.
     */
    public String getUserId() {
        return getTypedProperty(PROPERTY_USER_ID, String.class);
    }

    /**
     * The {@link Type} of event that occurred on the asset.
     *
     * @return The {@link Type}
     */
    public Type getType() {
        return getTypedProperty(PROPERTY_TYPE, Type.class);
    }

    /**
     * Any additional information provided for certain event types.
     *
     * @return A <code>String</code> representing the additional information, or <code>null</code> if n/a.
     */
    public String getAdditionalInfo() {
        return getTypedProperty(PROPERTY_ADDITIONAL_INFO, String.class);
    }

    /**
     * Expiration date for the ASSET_EXPIRED event.
     *
     * @return <code>Date</code> representing the expiration date, or <code>null</code> if n/a.
     */

    public Date getExpirationDate() {
        return getTypedProperty(PROPERTY_EXPIRATIONDATE, Date.class);
    }

    /**
     * Creator of the asset.
     *
     * @return <code>String</code> representing the creator of asset, or <code>null</code> if n/a.
     */

    public String getCreatedBy() {
        return getTypedProperty(PROPERTY_CREATEDBY, String.class);
    }

    /**
     * Indicates whether this event represents an asset having been uploaded to an external system.
     *
     * @return <code>true</code> if the event type is {@link Type#PUBLISHED_EXTERNAL}.
     */
    public boolean isPublishedExternally() {
        return Type.PUBLISHED_EXTERNAL == getType();
    }

    /**
     * Translates a given {@link Event} to a {@link DamEvent} if the event topic matches {@link DamEvent#EVENT_TOPIC}.
     *
     * @param event The {@link Event} to translate.
     *
     * @return A {@link DamEvent} or <code>null</code> if the given event is not of the expected topic.
     */
    public static DamEvent fromEvent(final Event event) {
        return EVENT_TOPIC.equals(event.getTopic()) ? new DamEvent(event) : null;
    }

    /**
     * Create a distributable event.
     *
     * @return An event.
     */
    public Event toEvent() {
        return new Event(EVENT_TOPIC, this.getDistributableEventProperties());
    }

    /**
     * Create a non distributable event.
     *
     * @return An event.
     */
    public Event toNonDistributableEvent() {
        return new Event(EVENT_TOPIC,  this.getEventProperties());
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event of the license of a DRM-protected asset having been
     * accepted.
     *
     * @param assetPath   The path of the asset for which the license has been accepted.
     * @param userId      The id of the user that accepted the license.
     * @param licenseInfo Information about the license, such as its path or URL.
     *
     * @return The corresponding <code>DamEvent</code>
     *
     * @see Type#ACCEPTED
     */
    public static DamEvent accepted(final String assetPath, final String userId, final String licenseInfo) {
        return new DamEvent(Type.ACCEPTED, assetPath, userId, licenseInfo, null, null, null, null, null);
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event of the license of a DRM-protected asset having been
     * rejected.
     *
     * @param assetPath   The path of the asset for which the license has been rejected.
     * @param userId      The id of the user that rejected the license.
     * @param licenseInfo Information about the license, such as its path or URL.
     *
     * @return The corresponding <code>DamEvent</code>
     *
     * @see Type#REJECTED
     */
    public static DamEvent rejected(final String assetPath, final String userId, final String licenseInfo) {
        return new DamEvent(Type.REJECTED, assetPath, userId, licenseInfo, null, null, null, null, null);
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event of an asset having been downloaded. In case of a
     * DRM-protected asset, this event can only follow a {@link #accepted(String, String, String)} event.
     *
     * @param assetPath The path of the asset that was downloaded.
     * @param userId    The id of the user that downloaded the asset.
     *
     * @return The corresponding <code>DamEvent</code>
     *
     * @see Type#DOWNLOADED
     */
    public static DamEvent downloaded(final String assetPath, final String userId) {
        return new DamEvent(Type.DOWNLOADED, assetPath, userId, null, null, null, null, null, null);
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event of an asset having been downloaded. In case of a
     * DRM-protected asset, this event can only follow a {@link #accepted(String, String, String)} event.
     *
     * @param assetPath         The path of the asset that was downloaded.
     * @param userId            The id of the user that downloaded the asset.
     * @param additionalInfo    Name of the source workflow of the downloaded asset, empty value means normal download.
     *
     * @return The corresponding <code>DamEvent</code>
     *
     * @see Type#DOWNLOADED
     */
    public static DamEvent downloaded(final String assetPath, final String userId, String additionalInfo) {
        return new DamEvent(Type.DOWNLOADED, assetPath, userId, null, null, additionalInfo, null, null, null);
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event of an asset having been versioned.
     *
     * @param assetPath The path of the asset that was versioned.
     * @param userId    The id of the user that versioned the asset.
     * @param versionId The id of the version that was created.
     *
     * @return The corresponding <code>DamEvent</code>
     *
     * @see Type#VERSIONED
     * @see Asset#createRevision(String, String)
     */
    public static DamEvent versioned(final String assetPath, final String userId, final String versionId) {
        return new DamEvent(Type.VERSIONED, assetPath, userId, null, versionId, null, null, null, null);
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event of an asset having been restored from a previous
     * version.
     *
     * @param assetPath The path of the asset that was restored.
     * @param userId    The id of the user that restored the asset.
     * @param versionId The id of the version from which the asset was restored.
     *
     * @return The corresponding <code>DamEvent</code>
     *
     * @see Type#RESTORED
     * @see AssetManager#restore(String)
     */
    public static DamEvent restored(final String assetPath, final String userId, final String versionId) {
        return new DamEvent(Type.RESTORED, assetPath, userId, null, versionId, null, null, null, null);
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event of an asset's metadata having been updated.
     *
     * @param assetPath The path of the asset the metadata of which was updated.
     * @param userId    The id of the user that updated the metadata.
     *
     * @return The corresponding <code>DamEvent</code>
     *
     * @see Type#METADATA_UPDATED
     */
    public static DamEvent metadataUpdated(final String assetPath, final String userId) {
        return new DamEvent(Type.METADATA_UPDATED, assetPath, userId, null, null, null, null, null, null);
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event of an asset's metadata having been updated.
     *
     * @param assetPath The path of the asset the metadata of which was updated.
     * @param userId    The id of the user that updated the metadata.
     * @param additionalInfo metadata info passed as "property1=value1,property2=value2..."
     *
     * @return The corresponding <code>DamEvent</code>
     *
     * @see Type#METADATA_UPDATED
     */
    public static DamEvent metadataUpdated(final String assetPath, final String userId, String additionalInfo) {
        return new DamEvent(Type.METADATA_UPDATED, assetPath, userId, null, null, additionalInfo, null, null, null);
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event of an asset having been uploaded to an external system.
     *
     * @param assetPath      The path of the asset that was uploaded.
     * @param userId         The id of the user that uploaded the asset.
     * @param additionalInfo Information about the external system to which the asset was uploaded, such as a URL.
     *
     * @return The corresponding <code>DamEvent</code>
     *
     * @see Type#PUBLISHED_EXTERNAL
     */
    public static DamEvent publishedExternally(final String assetPath,
                                               final String userId,
                                               final String additionalInfo) {
        return new DamEvent(Type.PUBLISHED_EXTERNAL, assetPath, userId, null, null, additionalInfo, null, null, null);
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event of an asset's original rendition having been updated.
     *
     * @param assetPath The path of the asset the original of which was updated.
     * @param userId    The id of the user that updated the asset's original.
     *
     * @return The corresponding <code>DamEvent</code>
     *
     * @see Type#ORIGINAL_UPDATED
     * @see Asset#addRendition(String, java.io.InputStream, String)
     */
    public static DamEvent originalUpdated(final String assetPath, final String userId) {
        return new DamEvent(Type.ORIGINAL_UPDATED, assetPath, userId, null, null, null, null, null, null);
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event of a rendition having been added or updated for an
     * asset.
     *
     * @param assetPath     The path of the asset for which a rendition was added or updated.
     * @param userId        The id of the user that added/updated the rendition.
     * @param renditionPath The path of the rendition that was added/updated.
     *
     * @return The corresponding <code>DamEvent</code>
     *
     * @see Type#RENDITION_UPDATED
     * @see Asset#addRendition(String, java.io.InputStream, String)
     */
    public static DamEvent renditionUpdated(final String assetPath, final String userId, final String renditionPath) {
        return new DamEvent(Type.RENDITION_UPDATED, assetPath, userId, null, null, renditionPath, null, null, null);
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event of a rendition having been removed for an asset.
     *
     * @param assetPath     The path of the asset for which a rendition was removed.
     * @param userId        The id of the user that removed the rendition.
     * @param renditionPath The path of the rendition that was removed.
     *
     * @return The corresponding <code>DamEvent</code>
     *
     * @see Type#RENDITION_REMOVED
     * @see Asset#removeRendition(String)
     */
    public static DamEvent renditionRemoved(final String assetPath, final String userId, final String renditionPath) {
        return new DamEvent(Type.RENDITION_REMOVED, assetPath, userId, null, null, renditionPath, null, null, null);
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event of a rendition having been downloaded for an
     * asset.
     *
     * @param assetPath     The path of the asset for which a rendition was downloaded.
     * @param userId        The id of the user that downloaded the rendition.
     * @param renditionPath The path of the rendition that was downloaded.
     *
     * @return The corresponding <code>DamEvent</code>
     *
     * @see Type#RENDITION_DOWNLOADED
     */
    public static DamEvent renditionDownloaded(final String assetPath, final String userId, final String renditionPath) {
        return new DamEvent(Type.RENDITION_DOWNLOADED, assetPath, userId, null, null, renditionPath, null, null, null);
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event of a sub-asset having been added or updated for an
     * asset.
     *
     * @param assetPath    The path of the asset for which a sub-asset was added or updated.
     * @param userId       The id of the user that added/updated the sub-asset.
     * @param subassetPath The path of the sub-asset that was added/updated.
     *
     * @return The corresponding <code>DamEvent</code>
     *
     * @see Type#SUBASSET_UPDATED
     * @see Asset#addSubAsset(String, String, java.io.InputStream)
     */
    public static DamEvent subassetUpdated(final String assetPath, final String userId, final String subassetPath) {
        return new DamEvent(Type.SUBASSET_UPDATED, assetPath, userId, null, null, subassetPath, null, null, null);
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event of a sub-asset having been removed for an asset.
     *
     * @param assetPath    The path of the asset for which a sub-asset was removed.
     * @param userId       The id of the user that removed the sub-asset.
     * @param subassetPath The path of the sub-asset that was removed.
     *
     * @return The corresponding <code>DamEvent</code>
     *
     * @see Type#SUBASSET_REMOVED
     */
    public static DamEvent subassetRemoved(final String assetPath, final String userId, final String subassetPath) {
        return new DamEvent(Type.SUBASSET_REMOVED, assetPath, userId, null, null, subassetPath, null, null, null);
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event of an asset having been created.
     *
     * @param assetPath The path of the asset that was just created.
     * @param userId    The id of the user that created the asset.
     *
     * @return The corresponding <code>DamEvent</code>
     *
     * @see Type#ASSET_CREATED
     * @see AssetManager#createAsset(String, java.io.InputStream, String, boolean)
     */
    public static DamEvent assetCreated(final String assetPath, final String userId) {
        return new DamEvent(Type.ASSET_CREATED, assetPath, userId, null, null, null, null, null, null);
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event of an asset having been ,pved.
     *
     * @param assetPath The path of the asset that was just moved.
     * @param userId    The id of the user that moved the asset.
     *
     * @return The corresponding <code>DamEvent</code>
     *
     * @see Type#ASSET_MOVED
     */
    public static DamEvent assetMoved(final String assetPath, final String userId) {
        return new DamEvent(Type.ASSET_MOVED, assetPath, userId, null, null, null, null, null, null);
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event of an asset having been removed.
     *
     * @param assetPath The path of the asset that was just removed.
     * @param userId    The id of the user that removed the asset.
     *
     * @return The corresponding <code>DamEvent</code>
     *
     * @see Type#ASSET_REMOVED
     */
    public static DamEvent assetRemoved(final String assetPath, final String userId) {
        return new DamEvent(Type.ASSET_REMOVED, assetPath, userId, null, null, null, null, null, null);
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event of an asset after expiration.
     * By default expiration date property that is considered is jcr:content/offTime
     *
     * @param assetPath The path of the asset that was expired.
     * @param expirationDate  The expiration date of the asset.
     * @param createdBy Creator of the asset
     * @param userId The id of the user that expired the asset.
     *
     * @return The corresponding <code>DamEvent</code>
     *
     * @see Type#ASSET_EXPIRING
     */
    public static DamEvent assetExpired(final String assetPath, final String userId, final Date expirationDate, final String createdBy) {
        return new DamEvent(Type.ASSET_EXPIRED, assetPath, userId, null, null, null, null, expirationDate, createdBy);
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event of an asset getting expired in future.
     * By default expiration date property that is considered is jcr:content/offTime
     *
     * @param assetPath The path of the asset that is getting expired.
     * @param expirationDate  The expiration date of the asset.
     * @param createdBy Creator of the asset
     * @param userId The id of the user that expired the asset.
     *
     * @return The corresponding <code>DamEvent</code>
     *
     * @see Type#ASSET_EXPIRED
     */
    public static DamEvent assetExpiring(final String assetPath, final String userId, final Date expirationDate, final String createdBy) {
        return new DamEvent(Type.ASSET_EXPIRING, assetPath, userId, null, null, null, null, expirationDate, createdBy);
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event upon visiting an
     * asset's detail page.
     *
     * @param assetPath The path of the asset whose details were accessed.
     * @param userId The id of the user that accessed the asset details page.
     * @return The corresponding <code>DamEvent</code>
     * @see Type#ASSET_VIEWED
     */
    public static DamEvent assetViewed(final String assetPath,
            final String userId) {
        return new DamEvent(Type.ASSET_VIEWED, assetPath, userId, null, null,
            null, null, null, null);
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event of an asset
     * being shared on different cloud solutions.
     *
     * @param assetPath The path of the asset that was shared.
     * @param userId The id of the user who shared the asset.
     * @param additionalInfo Place where this asset was shared.
     * @return The corresponding <code>DamEvent</code>
     * @see Type#ASSET_SHARED
     */
    public static DamEvent assetShared(final String assetPath,
            final String userId, String additionalInfo) {
        return new DamEvent(Type.ASSET_SHARED, assetPath, userId, null, null,
            additionalInfo, null, null, null);
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event of an asset
     * being published.
     *
     * @param assetPath The path of the asset that was published.
     * @param userId The id of the user that updated the metadata.
     * @return The corresponding <code>DamEvent</code>
     * @see Type#ASSET_PUBLISHED
     */
    public static DamEvent assetPublished(final String assetPath,
            final String userId) {
        return new DamEvent(Type.ASSET_PUBLISHED, assetPath, userId, null,
            null, null, null, null, null);
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event of a project
     * being viewed.
     *
     * @param assetPath The path of the project that was viewed.
     * @param userId The id of the user who viewed the project.
     * @return The corresponding <code>DamEvent</code>
     * @see Type#PROJECT_VIEWED
     */
    public static DamEvent projectViewed(final String assetPath,
            final String userId) {
        return new DamEvent(Type.PROJECT_VIEWED, assetPath, userId, null, null,
            null, null, null, null);
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event of a collection
     * being viewed.
     *
     * @param assetPath The path of the collection that was viewed.
     * @param userId The id of the user who viewed the collection.
     * @return The corresponding <code>DamEvent</code>
     * @see Type#COLLECTION_VIEWED
     */
    public static DamEvent collectionViewed(final String assetPath,
            final String userId) {
        return new DamEvent(Type.COLLECTION_VIEWED, assetPath, userId, null,
            null, null, null, null, null);
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event of addidng a
     * commment to an asset.
     *
     * @param assetPath The path of the asset which was commented upon.
     * @param userId The id of the user who commented on the asset.
     * @param additionalInfo Message of added comment.
     * @return The corresponding <code>DamEvent</code>
     * @see Type#ADDED_COMMENT
     */
    public static DamEvent addedComment(final String assetPath,
            final String userId, String additionalInfo) {
        return new DamEvent(Type.ADDED_COMMENT, assetPath, userId, null, null,
            additionalInfo, null, null, null);
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event that asset is
     * used in some solution.
     *
     * @param assetPath The path of the asset which was used.
     * @param usageType defines the type of usage, whether asset is used in
     *            collection or some third party solution like Campaign, Social.
     * @param date The time when asset was used.
     * @return The corresponding <code>DamEvent</code>
     * @see Type#ASSET_USAGE
     */
    public static DamEvent assetUsage(final String assetPath,
            final String usageType, final Date date) {
        return new DamEvent(Type.ASSET_USAGE, assetPath, null, null, null,
            usageType, date, null, null);
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event that DAM update
     * asset workflow is completed.
     *
     * @param assetPath The path of the asset on which the DAM update asset workflow
     *                  executed.
     * @return The corresponding <code>DamEvent</code>
     * @see Type#DAM_UPDATE_ASSET_WORKFLOW_COMPLETED
     */
    public static DamEvent damUpdateAssetWorkflowCompleted(final String assetPath) {
        return new DamEvent(Type.DAM_UPDATE_ASSET_WORKFLOW_COMPLETED, assetPath, null,
                null, null, null, null, null, null);
    }

    /**
     * Creates a {@link DamEvent} that corresponds to the event that DAM metadata
     * writeback workflow is completed.
     *
     * @param assetPath The path of the asset on which the DAM metadata writeback
     * workflow executed.
     * @return The corresponding <code>DamEvent</code>
     * @see Type#DAM_METADATA_WRITEBACK_WORKFLOW_COMPLETED
     */
    public static DamEvent damMetadataWritebackWorkflowCompleted(final String assetPath) {
        return new DamEvent(Type.DAM_METADATA_WRITEBACK_WORKFLOW_COMPLETED, assetPath, null,
                null, null, null, null, null, null);
    }
    
    private Dictionary<String, Object> getEventProperties() {

        final Dictionary<String, Object> properties = new Hashtable<String, Object>();

        propertiesMap.keySet()
            .forEach(k -> {
                Class<?> clazz = propertyTypesMap.get(k);
                if (null == clazz) {
                    clazz = String.class;
                }
                if (null != getTypedProperty(k, clazz)) {
                    properties.put(k, getTypedProperty(k, clazz));
                }
            });

        return properties;
    }
    
    private Dictionary<String, Object> getDistributableEventProperties() {
        Dictionary<String, Object> properties = this.getEventProperties();
        properties.put( PROPERTY_DISTRIBUTABLE, Boolean.TRUE);
        return properties;
    }
}