/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.security;

import java.security.Principal;
import java.util.Iterator;

import javax.jcr.AccessDeniedException;

import com.day.cq.commons.RangeIterator;

/**
 * UserManager allows access to Users and Groups.
 * These are the Objects that act on {@link org.apache.sling.api.resource.Resource Resources}.
 * They are referred to by the common super-class '{@link Authorizable Authorizable}'.
 * <br>
 * Basic search and creation means are provided.<br>
 * The UserManager for a {@link javax.jcr.Session Session} can be accessed
 * by use of {@link UserManagerFactory UserManagerFactory}
 * <p></p>
 * <i>NOTE</i>: This does not go to the osg.services.UserManger, as it is not existing
 * now...additionly to me its relation to Authentication outside Sling is not clear enough.
 *
 * @see Authorizable
 * @see UserManagerFactory
 * @deprecated cq 5.5 Use org.apache.jackrabbit.api.security.user.UserManager instead.
 */
public interface UserManager {

    /**
     * This method tests if the Manager has any Authorizable with this given
     * ID.<br>
     * If this method evaluates to  <code>true</code> a call to
     * {@link #get(String) get(id)} has to return a non-null
     * {@link Authorizable Authorizable}
     *
     * @param id to be probed
     * @return true if an Authorizable with the given id exists
     * @see #get(String)
     */
    boolean hasAuthorizable(String id);

    /**
     * Accesses an Authorizable by its {@link com.day.cq.security.Authorizable#getID() ID}.
     * if an Authorizable exists a test with
     * {@link #hasAuthorizable(String) hasAuthorizable(id)} has to be <code>true</code>.
     *
     * @param id of the Authorizable requested
     * @return the requests Authorizable
     * @throws com.day.cq.security.NoSuchAuthorizableException in case no Authorizable for that ID exists
     */
    Authorizable get(String id) throws NoSuchAuthorizableException;

    /**
     * Searches for an Authorizable for the given Principal.
     *
     * @param principal the Authorizable should contain
     * @return the Authorizable or <code>null</code> if none found
     * @see Authorizable#getPrincipal()
     */
    Authorizable find(Principal principal);

    /**
     * Search for an Authorizable that contains the given query as a value
     * of the given property.<br>
     * The value doesn't have to match exactly.
     * Property names of Property to be searched.
     * The Authorizable defines default properties, implementations may extend
     * these
     *
     * @param propertyName to be searched in
     * @param substring    to be found
     * @return Iterator of hits, empty if no Authorizable matches
     */
    Iterator<Authorizable> find(String propertyName, String substring);

    /**
     * <p>getUsers.</p>
     *
     * @return all Users this Manager knows. Should be used with care, may be
     *         time-consuming
     */
    Iterator<User> getUsers();

    /**
     * <p>getGroups.</p>
     *
     * @return all Groups this Manager knows.Should be used with care, may be
     *         time-consuming
     */
    Iterator<Group> getGroups();

    /**
     * Creates a new User for given Id, password and PrincipalName.
     *
     * @param userId        Id for the User
     * @param password      plain password, implementations may encrypt
     * @param principalName principalName to be used when authenticated
     * @return the newly created User
     * @throws com.day.cq.security.AuthorizableExistsException in case the ID is already in use
     * @throws javax.jcr.AccessDeniedException
     *                                     in case the requesting session is not allowed to create a User
     */
    User createUser(String userId, String password, String principalName)
            throws AuthorizableExistsException, AccessDeniedException;

    /**
     * Creates a new User for given Id, password and PrincipalName.
     *
     * @param userId           Id for the User
     * @param password         plain password, implementations may encrypt
     * @param principalName    principalName to be used when authenticated
     * @param intermediatePath a {@link java.lang.String} object.
     * @return the newly created User
     * @throws com.day.cq.security.AuthorizableExistsException in case the ID is already in use
     * @throws javax.jcr.AccessDeniedException
     *                                     in case the requesting session is not allowed to create a User
     */
    User createUser(String userId, String password, String principalName, String intermediatePath)
            throws AuthorizableExistsException, AccessDeniedException;


    /**
     * Creates a new Group for the given name and Principal-Name
     *
     * @param groupId       to be assigned to the new Group
     * @param principalName for the new Group
     * @return newly created Group
     * @throws com.day.cq.security.AuthorizableExistsException if the id is already in use
     * @throws javax.jcr.AccessDeniedException
     *                                     in case the requesting session is not allowed to create a Group
     */
    Group createGroup(String groupId, String principalName)
            throws AuthorizableExistsException, AccessDeniedException;

    /**
     * Creates a new Group for the given name and Principal-Name
     *
     * @param groupId          to be assigned to the new Group
     * @param principalName    for the new Group
     * @param intermediatePath a {@link java.lang.String} object.
     * @return newly created Group
     * @throws com.day.cq.security.AuthorizableExistsException if the id is already in use
     * @throws javax.jcr.AccessDeniedException
     *                                     in case the requesting session is not allowed to create a Group
     */
    Group createGroup(String groupId, String principalName, String intermediatePath)
            throws AuthorizableExistsException, AccessDeniedException;

    /** Constant <code>TYPE_USER=1</code> */
    final int TYPE_USER = 1;

    /** Constant <code>TYPE_GROUP=2</code> */
    final int TYPE_GROUP = 2;

    /**
     * Search for {@link Authorizable Authorizables} that contain the given Query.
     * Searches typical ID-Properties, like {@link com.day.cq.security.Authorizable#getID authorizable ID},
     * {@link com.day.cq.security.Authorizable#getPrincipal() principal name}.
     *
     * @param query token to find
     * @param type  Class of Authorizable to take Authorizable to search for any kind
     * @return that match the query
     * @param <AuthType> a AuthType object.
     */
    <AuthType extends Authorizable> RangeIterator<AuthType> find(String query, Class<AuthType> type);

    /**
     * Searches an Authorizable which has the given path as Path of it's
     * Home Path property.
     * Thus that {@code findByHomepath(homePath).getHomePath().equals(homePath)}
     *
     * @param homePath path of the Authorizable's Home
     * @return the Authorizable or <code>null</code>
     * @see Authorizable#getHomePath()
     */
    Authorizable findByHome(String homePath);

    /**
     * If any write operations executed through the User API are automatically
     * persisted this method returns <code>true</code>. In this case there are
     * no pending transient changes left and there is no need to explicitly call
     * {@link javax.jcr.Session#save()}. If this method returns <code>false</code>
     * any changes must be completed by an extra save call on the
     * <code>Session</code> associated with this <code>UserManager</code>.
     *
     * @return <code>true</code> if changes are automatically persisted;
     *         <code>false</code> if changes made through this API (including method
     *         calls on  {@link com.day.cq.security.Authorizable} and subclasses are only transient and
     *         must be persisted using {@link javax.jcr.Session#save()}.
     * @see #autoSave(boolean)
     */
    boolean isAutoSave();

    /**
     * Changes the auto save behavior of this <code>UserManager</code>.
     *
     * @param enable If <code>true</code> changes made through this API will
     *               be automatically saved
     * @return <code>true</code> if the autoSave mode was successfully changed.
     */
    boolean autoSave(boolean enable);
}
