/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 *
 **************************************************************************/
package com.day.cq.wcm.api.policies;

import java.util.List;
import java.util.function.Predicate;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.resource.Resource;
import org.osgi.annotation.versioning.ProviderType;

import com.day.cq.wcm.api.Template;
import com.day.cq.wcm.api.components.ComponentContext;

/**
 * Content Policy Manager API.
 */
@ProviderType
public interface ContentPolicyManager {

    /**
     * Get the <code>ContentPolicy</code> for the given <code>ComponentContext</code>.
     *
     * @param componentContext The mandatory component context to get the content policy for.
     *
     * @return The content policy for the given component context or <code>null</code> if none is configured.
     * @throws java.lang.IllegalArgumentException This exception is thrown, if the given component context is
     * <code>null</code>.
     */
    @Nullable
    public ContentPolicy getPolicy(@Nonnull ComponentContext componentContext);

    /**
     * Get the <code>ContentPolicy</code> for the given <code>Resource</code>.
     *
     * @param contentResource The mandatory resource to get the content policy for.
     *
     * @return The content policy for the given resource or <code>null</code> if none is configured.
     * @throws java.lang.IllegalArgumentException This exception is thrown, if the given resource is
     * <code>null</code>.
     */
    @Nullable
    public ContentPolicy getPolicy(@Nonnull Resource contentResource);

    /**
     * Get the <code>ContentPolicy</code> for the given <code>Resource</code>.
     *
     * @param contentResource The mandatory resource to get the content policy for.
     * @param request Request from which to read specific attributes and parameters. Allows to resolve delegated policy resources
     *
     * @return The content policy for the given resource or <code>null</code> if none is configured.
     * @throws java.lang.IllegalArgumentException This exception is thrown, if the given resource is
     * <code>null</code>.
     */
    @Nullable
    public ContentPolicy getPolicy(@Nonnull Resource contentResource, @Nullable SlingHttpServletRequest request);

    /**
     * Get the list of policies for the given path
     *
     * @param path Path from where to start searching for policies
     *
     * @return the list of policies for the given path.
     */
    public List<ContentPolicy> getPolicies(String path);

    /**
     * Get the list of policies for the given path. Here you can limit search to the given policy resource type
     *
     * Policy Resource Types are relative
     *
     * @param path Path from where to start searching for policies
     * @param policyResourceType Sling resource type associated with the policies
     * @return the list of policies for the given path.
     */
    public List<ContentPolicy> getPolicies(String path, String policyResourceType);

    /**
     * @param originalPolicy policy to be copied
     * @param newTitle The title of the target policy
     * @param newDescription The Description o the target policy
     *
     * @return The policy with new Description.
     */
    public ContentPolicy copyPolicy(ContentPolicy originalPolicy, String newTitle, String newDescription);

    /**
     * @param resource The resource for which policy mapping to be fetched
     * @return <code>ContentPolicyMapping</code>
     */
    public ContentPolicyMapping getPolicyMapping(Resource resource);

    /**
     * @param contentPolicy Policy for which policy mapping to be fetched.
     *
     * @return The List of <code>ContentPolicyMapping</code>
     */
    public List<ContentPolicyMapping> getPolicyMappings(ContentPolicy contentPolicy);


    /**
     * Returns the policy location for the given resource
     *
     * Policies are located following their respective component path location under the settings/wcm/policies directory
     *
     * The returned string is an absolute path to the directory that should contain policies for the given resource
     *
     * This function only returns a path if the given resource is contained in a page making use of the editable template
     * If this requirement is not meet the returned string is null
     *
     * @param resource Resource contained in a page making use of an editable template
     * @return The path to the directory that contains the policies for the given resource
     */
    public String getPolicyLocation(Resource resource);


    /**
     * Get the templates a policy is used in
     *
     * @param policyAbsolutePath absolute path to policy
     * @return list of templates where the policy is used in
     */
    public List<Template> getTemplates(String policyAbsolutePath);

    /**
     * Get a filtered list of templates a policy is used in
     *
     * @param policyAbsolutePath absolute path to policy
     * @param filter for the templates
     * @return list of templates where the policy is used in
     * @deprecated use {@link #getTemplates(String policyAbsolutePath, Predicate filter)} instead
     */
    @Deprecated
    public List<Template> getTemplates(String policyAbsolutePath, @Nullable org.apache.commons.collections.Predicate filter);

    /**
     * Get a filtered list of templates a policy is used in
     *
     * @param policyAbsolutePath absolute path to policy
     * @param filter             for the templates
     * @return list of templates where the policy is used in
     */
    public List<Template> getTemplates(String policyAbsolutePath, @Nullable Predicate filter);

    /**
     * Verifies if there is any Template matching the given predicate and using the policy with given path
     *
     * @param policyAbsolutePath a {@code String} absolute path to policy
     * @param filter a {@code Predicate} used to filter the templates on which lookup is performed; if the {@code filter} is {@code null},
     *               no filtering will be performed
     * @return {@code true} if there is any Template using the Policy with {@code policyAbsolutePath} path and is matching the {@code filter}, if provided.
     */
    public boolean isPolicyUsed(String policyAbsolutePath, @Nullable Predicate filter);

    /**
     * Verifies if there is any Template matching the given predicate and using the policy with given path
     *
     * @param policyAbsolutePath a {@code String} absolute path to policy
     * @param filter             a {@code Predicate} used to filter the templates on which lookup is performed; if the {@code filter} is {@code null},
     *                           no filtering will be performed
     * @return {@code true} if there is any Template using the Policy with {@code policyAbsolutePath} path and is matching the {@code filter}, if provided.
     * @since 1.7
     * @deprecated use {@link #isPolicyUsed(String policyAbsolutePath, Predicate filter)} instead
     */
    @Deprecated
    public boolean isPolicyUsed(String policyAbsolutePath, @Nullable org.apache.commons.collections.Predicate filter);

}
