/*************************************************************************
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2020 Adobe
 *  All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 **************************************************************************/
package com.day.util;

import java.util.Iterator;

/**
 * Implements the <code>Iterator</code> interface and combines two
 * <code>Iterator</code> objects into one single <code>Iterator</code>.
 * <p>
 * The constructor takes two <code>Iterator</code> arguments: <code>parent</code>
 * and <code>child</code>. Calling {@link #next} on this <code>Iterator</code>
 * will first try to return an element from the parent <code>Iterator</code>
 * and once the parent <code>Iterator</code> does not have any more elements
 * it will return elements from the child <code>Iterator</code>.
 *
 * @version $Revision: 1.2 $, $Date: 2004-08-22 06:56:09 +0200 (Sun, 22 Aug 2004) $
 * @author mreutegg
 * @since fennec
 * Audience wad
 */
public class CompoundIterator implements Iterator {

    /**
     * The parent <code>Iterator</code>. Elements from this
     * <code>Iterator</code> are returned first.
     */
    private final Iterator parent;

    /**
     * The child <code>Iterator</code>. Elements from this
     * <code>Iterator</code> are returned, after the parent
     * <code>Iterator</code> does not have elements any more.
     */
    private final Iterator child;

    /**
     * Creates a <code>CompoundIterator</code> based on <code>parent</code>
     * and <code>child</code>. This <code>CompountIterator</code> will first
     * return elements from <code>parent</code> and then elements from
     * <code>child</code>.
     * @param parent the <code>Iterator</code> from where to return the elements
     *  first.
     * @param child the <code>Iterator</code> from where to return the elements
     *  after <code>parent</code> does not have elements any more.
     */
    public CompoundIterator(Iterator parent, Iterator child) {
	this.parent = parent;
	this.child = child;
    }

    /**
     * Returns <code>true</code> if either parent or child iterator
     * has a next element; <code>false</code> otherwise.
     * @return <code>true</code> if either parent or child iterator
     *  has a next element; <code>false</code> otherwise.
     */
    public boolean hasNext() {
	return parent.hasNext() || child.hasNext();
    }

    /**
     * Returns the next element from the parent or the child
     * iterator object.
     * @return the next element from the parent or the child
     *  iterator object.
     */
    public Object next() {
	return (parent.hasNext()) ? parent.next() : child.next();
    }

    /**
     * Always throws <code>UnsupportedOperationException</code>
     * @throws UnsupportedOperationException always!
     */
    public void remove() throws UnsupportedOperationException {
    	throw new UnsupportedOperationException("remove() is not supported in CompoundInterator");
    }
}
