/* ****************************************************************************
 *
 *	File: CosBoolean.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.cos;

import java.io.IOException;

import com.adobe.internal.io.stream.OutputByteStream;
import com.adobe.internal.pdftoolkit.core.util.StringOps;

/**
 * Represents a COS Boolean object as defined in section 3.2.1 of the PDF
 * Reference Manual version 1.4.
 */
public class CosBoolean extends CosScalar
{
	private Boolean mValue;		// Underlying value of the boolean

	/**
	 * Constructs a COS Boolean object.
	 *
	 * @param doc		Document containing the boolean
	 * @param value		Value for the object (true or false)
	 * @param info		Object info for the boolean
	 */
	CosBoolean(CosDocument doc, boolean value, CosObjectInfo info)
	{
		super(doc, info);
		mValue = Boolean.valueOf(value);
	}

	CosBoolean(CosDocument doc, Boolean value, CosObjectInfo info)
	{
		super(doc, info);
		mValue = value;
	}

	/**
	 * 
	 * return the type of this CosObject
	 */
	@Override
	public int getType()
	{
		return t_Boolean;
	}

	/**
	 * 
	 * Returns the value of the boolean.
	 *
	 * @return boolean
	 */
	@Override
	public boolean booleanValue()
	{
		return mValue.booleanValue();
	}

	/**
	 * 
	 * Returns the value as an instance of Boolean
	 * @return Boolean
	 */
	@Override
	public Object getValue()
	{
		return mValue;
	}

	
@Override
//	public String toString()
//	{
//		return mValue.toString();
//	}
	
	void writeOut(OutputByteStream outStream, boolean inString, boolean inDebugMode)
	throws IOException
	{
		outStream.write(StringOps.toByteArray(mValue.toString()));
	}
	
	/**
	 *  This method returns true if both CosObjects have same boolean value or 
	 *  they are the same objects. Returns false if passed CosObject is not an instance of
	 *  CosBoolean.
	 *  @param value
	 *  @return boolean
	 */
	@Override
	public boolean equals(CosObject value){
		if(!(value instanceof CosBoolean) || value.getDocument() != this.getDocument())
			return false;
		if(value == this)
			return true;
		return (value.booleanValue() == this.booleanValue());
	}
}
