/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2015 Adobe Systems Incorporated
 * All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */

package com.adobe.cq.dam.cfm;

import aQute.bnd.annotation.ProviderType;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Calendar;

/**
 * Represents a piece of content (or data) provided by a content fragment.
 *
 * <p>In general, {@code FragmentData} works as a wrapper around a (generic) Java object
 * that represents the actual value. This avoids using {@code Object} in the API and also
 * allows to provide helper functionality to access the generic value in a controlled way.
 * </p>
 *
 * <p>Values are typed. For more information about data types, see {@link DataType}.</p>
 *
 * <p>Values of a given type - represented by an {@code Object} wrapped by
 * {@link FragmentData} - may need to be converted to the needs of the client. Basically,
 * {@link FragmentData#getValue(Class)} supports to convert each value to a {@code String}
 * (and back).</p>
 *
 * <p>For the conversion to a {@code String} the following rules are followed:</p>
 *
 * <ul>
 *     <li>
 *         The conversion of numbers is technical, i.e. based on
 *         {@code Xxxx.toString()}/{@code Xxx.parseXxx} rather than {@code NumberFormat}
 *     </li>
 *     <li>
 *         The conversion of {@code Calendar} is based on ISO-8601.
 *     </li>
 * </ul>
 *
 * @since 1.1
 */
@ProviderType
public interface FragmentData {

    /**
     * Gets the data type of the value.
     *
     * @return The data type
     */
    @NotNull
    DataType getDataType();

    /**
     * Gets the value, converted to the provided type.
     *
     * <p>For information about supported type conversions: see {@link DataType}.</p>
     *
     * <p>Note that this method is supposed to return a suitably converted value for all
     * types where {@link #isTypeSupported(Class)} returns {@code true}</p>
     *
     * @param <T> Describes the type parameter
     * @param type The type the value should be converted to
     * @return The value; {@code null} if the value could not be converted to the provided
     *         type
     */
    @Nullable
    <T> T getValue(Class<T> type);

    /**
     * Checks if the specified Java data type is supported for calls to
     * {@link #getValue(Class)} and {@link #setValue(Object)}.
     *
     * @param type The class object representing the type
     * @return {@code true} if the type is supported
     */
    boolean isTypeSupported(Class type);

    /**
     * Gets the original value as a generic Java object.
     *
     * @return The value
     */
    @Nullable
    Object getValue();

    /**
     * Sets the value.
     *
     * <p>The general rule is that a non-null value that was retrieved using
     * {@link #getValue(Class)} needs to be accepted and converted to a suitable original
     * value. Also, every value of a type where {@link #isTypeSupported(Class)} returns
     * {@code true} needs to be accepted and converted accordingly.</p>
     *
     * <p>Note that for the value to be persisted,
     * {@link ContentElement#setValue(FragmentData)} or
     * {@link ContentVariation#setValue(FragmentData)} have to be called explicitly.
     * This allows for manipulating the value multiple times before actually persisting it.
     * </p>
     *
     * @param value The value
     * @throws ContentFragmentException if the type of the value provided is unsupported
     */
    void setValue(@Nullable Object value) throws ContentFragmentException;

    /**
     * Returns the content type for textbased data types.
     *
     * <p>This will return a MIME type only when applicable. For numbers, dates, booleans,
     * it will typically be {@code null}.</p>
     *
     * @return The content type; {@code null} if not applicable
     */
    @Nullable
    String getContentType();

    /**
     * Sets the content type for text-based data types.
     *
     * <p>For data other than text-based types {@code null} should be provided.</p>
     *
     * @param contentType The content type; {@code null} if not applicable
     */
    void setContentType(@Nullable String contentType);

    /**
     * Returns the date the value was last modified.
     * 
     * @return the last modified date or {@code null} if not available
     */
    @Nullable
    Calendar getLastModified();

    /**
     * Returns the user that last modified this resource.
     *
     * @return String containing the name of the user that last modified this resource.
     */
    @Nullable
    String getLastModifiedBy();
}
