/*
 * File: CombiningSequence.java
 * 
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */
package com.adobe.fontengine;

import java.io.Serializable;
import java.util.Arrays;


/** Represents all the strings canonically equivalent to a combining sequence.
 * This class is used in two contexts: 
 * <ul>
 * <li>when using a cmap, to figure out which entry in a font's cmap we need
 * to look up
 * <li>when building a cmap, to figure out which entries we should populate
 * </ul>
 * 
 * See {@link com.adobe.fontengine.CombiningSequenceResourceBuilder} for more details
 * on how this works.
 */

final class CombiningSequence implements Serializable {
  /* Serialization signature is explicitly set and should be 
   * incremented on each release to prevent compatibility.
   */
    static final long serialVersionUID = 1;
    
  /* When using a cmap, which entry we should look up. */
  private final int composeTo;
  
  /* When building a cmap, which entries we should populate. */
  private final int[] mapFrom;
  
  /* The coded characters we can append to this sequence to form another
   * sequence, in increasing code point order. */
  private final int[] nextChar;
  
  /* The combining sequences we obtain by appending the corresponding 
   * entry in <code>nextChar</code> to this sequence.
   * This array must have the same number of elements are <code>nextChar</code>.
   */
  private final CombiningSequence[] nextSequence;
  
  /* To avoid allocating this all the time. */
  private final static int[] emptyIntArray = new int[0];
  
  /** Construct a CombiningSequence.
   * @param composeTo the cmap entry to look up for this combining sequence.
   * @param mapFrom the cmap entries to populate for this sequence.
   * @param nextChar characters which can be added to this sequence to form other
   * sequences
   * @param nextSequence combining sequences obtained by adding characters
   * in <code>nextChar</code>
   */
  CombiningSequence (int composeTo, int[] mapFrom,
                     int[] nextChar, CombiningSequence[] nextSequence) {
    this.composeTo = composeTo;
    this.mapFrom = mapFrom.clone ();
    this.nextChar = nextChar.clone ();
    this.nextSequence = nextSequence.clone ();
  }
  
  /** Return the entry to lookup in a cmap for a combining sequence.
   * The combining sequence is the concatenation of the sequence represented
   * by this object, and some additional characters passed in. 
   *
   * @param usvs contains the additional characters
   * @param start the index of the first additional character
   * @param limit the index following the last additional character
   * @return the entry to lookup, -1 if none
   */
  final int compose (int[] usvs, int start, int limit) {
    if (start >= limit) {
      return composeTo; }
    else {
      int index = Arrays.binarySearch (nextChar, usvs [start]);
      if (index < 0) {
        return -1; }
      else {
        return nextSequence [index].compose (usvs, start+1, limit); }}
  }
        
  /** Return the entries to populate in a cmap for a combining sequence.
   * The combining sequence is the concatenation of the sequence represented
   * by this object, and some additional characters passed in.
   * 
   * @param usvs contains the additional cahracters
   * @param start the index of the first additional character
   * @param limit the index following the last additional character
   * @return the entries to populate; if none, an empty array
   */
  final int[] map (int[] usvs, int start, int limit) {
    if (start >= limit) {
      return mapFrom.clone (); }
    else {
      int index = Arrays.binarySearch (nextChar, usvs [start]);
      if (index < 0) {
        return emptyIntArray; }
      else {
        return nextSequence [index].map (usvs, start+1, limit); }}
  }
}
