/*
 *	File: Type2Parser.java
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 */

package com.adobe.fontengine.font.cff;

import com.adobe.fontengine.font.InvalidGlyphException;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.postscript.SeacPhase;
import com.adobe.fontengine.font.postscript.StandardEncoding;

/** Parse a Type2 charstring.
 * 
 * A <code>Type2Parser</code> offers a single method, <code>parse</code> which is given 
 * a charstring and a</code> Type2Consumer</code>. As the charString is parsed, the 
 * <code>Type2Consumer</code> is called for each operator. The parser maintains the 
 * Type2 stack and storage area, interpreting the operators which affect them, 
 * as well as interpreting subroutine calls. The intent is that a Consumer 
 * can concentrate on dealing with the path and hints operators, and 
 * essentially ignore the arithmetic, storage, conditional and subroutine 
 * operators. However, all the operators are reported to the Consumer, so 
 * that it is for example possible to write a Consumer that creates an accurate 
 * textual representation of the charstring.
 * 
 * <p>The same parser can be used repeatedly to 
 * parse multiple charstrings; those charstrings do not need to be 
 * in the same font, nor even in the same font set. The parser maintains 
 * no state between invocations of <code>parse</code>.
 * 
 * <p>The consumer must not call <code>parse</code> on the parser that 
 * invoked it.
 * 
 * <h4>Synchronization</h4>
 * 
 * This class is not synchronized. The <code>parse</code> method invokes the
 * Type2Consumer on the thread on which it is called.
 */
final public class Type2Parser {

  /** The interpreter stack. */
  private final double[] stack = new double[48];

  /** The interpreter storage area. */
  private final double[] storage = new double[32];

  /** The number of hints for the charstring being parsed. */
  private int nbHints;

  /** The number of entries on the stack. */
  private int stackDepth;

  /** The number of pending subroutine calls. Used to 
   * verify that return operators are properly nested. */
  private int callDepth;

  /** Indicates whether the stack can legally contain the charstring width. */
  private boolean widthMayBeThere;

  /** Indicates where we are in a seac glyph */
  private SeacPhase seacPhase;

  private double seacAccentX;
  private double seacAccentY;

  /** Check that some condition is true. 
   * This is typically called to verify some property of the interpreter state, 
   * such as the minimum number of arguments on the stack.
   * @param b the condition to verify.
   * @throws InvalidGlyphException if the condition is not true.
   */
  private void check(boolean b) throws InvalidGlyphException {
    if (!b) {
      throw new InvalidGlyphException("invalid Type2 charstring");}
  }

  /** Check that the value is an integer, and return it (as an int).
   * @param d the numeric value
   * @return <code>d</code> as an <code>int</code>.
   * @throws InvalidGlyphException if <code>d</code> is not an integer.
   */
  private int checkInt(double d) throws InvalidGlyphException {
    int i = (int) d;
    if (i != d) {
      throw new InvalidGlyphException("unexpected non-integer value");}
    return i;
  }

  /** Called to pull the width from the stack, if present. 
   * @param c the Type2Consumer to alert.
   * @returns true iff parsing should continue
   * @throws InvalidGlyphException if there is a width, but it is not the 
   * topmost stack element.
   */
  private boolean handleWidth(Type2Consumer c) 
  throws InvalidGlyphException {

    boolean result = true;
    if (stackDepth == 1) {
      if (widthMayBeThere) {
        result = c.width(stack[0]); } 
      else if (seacPhase == SeacPhase.seacNone) {
        throw new InvalidGlyphException("stack is non empty"); }
      stackDepth = 0; }
    widthMayBeThere = false;
    return result;
  }

  /** Decode a Type2 charstring.
   * @param charStrings the <code>CharStrings</code> containing the charstring
   * @param index the index of the charstring in <code>charStrings</code>
   * @param localSubrs the local subroutines
   * @param globalSubrs the global subroutines
   * @param c the Type2Consumer to alert of the parsing events
   * @param font The name-keyed font to which the charStrings belong, if they
   * belong to a name-keyed font. If the font is cid-keyed, null should be passed
   * in. font is only used to resolve seac glyphs, which can only be present in name-keyed
   * fonts.
   * @throws InvalidGlyphException the charstring is malformed or missing
   */
  public void parse(CharStrings charStrings, int index,
      CharStrings localSubrs, CharStrings globalSubrs, 
      Type2Consumer c, NameKeyedFont font)
  throws InvalidGlyphException, UnsupportedFontException {

    widthMayBeThere = true;
    stackDepth = 0;
    callDepth = 0;
    nbHints = 0;
    seacPhase = SeacPhase.seacNone;
    int offset;
    int nextOffset;
    
    try {
    	offset = charStrings.offsetOf(index);
    	nextOffset = charStrings.offsetFollowing(index);
    } catch (ArrayIndexOutOfBoundsException e) {
    	throw new InvalidGlyphException("Glyph past end of charstrings array", e);
    }
    
    try {    
	    boolean keepGoing = parse(charStrings.data,
	    			offset, nextOffset,
	    			localSubrs, globalSubrs, c, font);
	    
	
	    if (keepGoing) {
	      // implicit enchar
	      c.endchar(stack, stackDepth); }
    } catch (InvalidFontException e) {
    	throw new InvalidGlyphException(e);
    }
  }

  /** Decode a Type2 charstring.
   * @param data the <code>CFFByteArray</code> containing the charstring
   * @param offset the offset of the first byte of the charstring in <code>data</code>
   * @param end the byte following the last byte of the charstring in <code>data</code>
   * @param localSubrs the local subroutines
   * @param globalSubrs the global subroutines
   * @param c the Type2Consumer to alert of the parsing events
   * @return true iff parsing should continue
   * @throws InvalidGlyphException the charstring is malformed
   */
  private boolean parse(CFFByteArray data, int offset, int end,
      CharStrings localSubrs, CharStrings globalSubrs, 
      Type2Consumer c, NameKeyedFont font)
  throws InvalidFontException, UnsupportedFontException {

    while (offset < end) {
      int b0 = data.getcard8(offset);
      offset++;

      switch (b0) {
        case Type2CStringOps.HSTEM: { 
          check(stackDepth >= 2);
          nbHints += stackDepth / 2;

          if (seacPhase == SeacPhase.seacAccentPreMove) {
            int i = (stackDepth % 2 != 0) ? 1 : 0;
            stack[i] += seacAccentY; }

          c.hstem(stack, stackDepth);
          stackDepth = stackDepth % 2;
          boolean keepGoing = handleWidth(c);
          if (!keepGoing) {
            return false; }
          break; }

        case Type2CStringOps.VSTEM: { 
          check(stackDepth >= 2);
          nbHints += stackDepth / 2;


          if (seacPhase == SeacPhase.seacAccentPreMove) {
            int i = (stackDepth % 2 != 0) ? 1 : 0;
            stack[i] += seacAccentX; }

          c.vstem(stack, stackDepth);
          stackDepth = stackDepth % 2;
          boolean keepGoing = handleWidth(c);
          if (!keepGoing) {
            return false; }
          break; }

        case Type2CStringOps.VMOVETO: { 
          check(stackDepth == 1 || stackDepth == 2);
          if (seacPhase == SeacPhase.seacAccentPreMove) {
            if (stackDepth == 2) {
              stack[2] = stack[1] + seacAccentY;
              stack[1] = seacAccentX; }
            else  {
              stack[1] = stack[0] + seacAccentY;
              stack[0] = seacAccentX; }
            stackDepth++;
            c.moveto(stack, stackDepth);
            stackDepth -= 2;

            seacPhase = SeacPhase.seacAccentPostMove; }
          else {
            c.vmoveto(stack, stackDepth);
            stackDepth -= 1; }

          boolean keepGoing = handleWidth(c);
          if (!keepGoing) {
            return false; }
          break; }

        case Type2CStringOps.RLINETO: { 
          check(stackDepth >= 2 && stackDepth % 2 == 0);
          c.rlineto(stack, stackDepth);
          stackDepth = 0;
          break; }

        case Type2CStringOps.HLINETO: { 
          check(stackDepth > 0);
          c.hlineto(stack, stackDepth);
          stackDepth = 0;
          break; }

        case Type2CStringOps.VLINETO: {
          check(stackDepth > 0);
          c.vlineto(stack, stackDepth);
          stackDepth = 0;
          break; }

        case Type2CStringOps.RRCURVETO: {
          check(stackDepth > 0 && stackDepth % 6 == 0);
          c.rrcurveto(stack, stackDepth);
          stackDepth = 0;
          break; }

        case Type2CStringOps.CALLSUBR: {
          check(stackDepth >= 1);
          check(localSubrs != null);

          int subr = checkInt(stack[--stackDepth]);

          if (localSubrs.getCount() < 1240) {
            subr += 107; }
          else if (localSubrs.getCount() < 33900) {
            subr += 1131; }
          else {
            subr += 32768; }
          c.callsubr(stack, stackDepth, subr);

          callDepth++;
          boolean keepGoing = parse(localSubrs.data, 
              localSubrs.offsetOf(subr), 
              localSubrs.offsetFollowing(subr),
              localSubrs, globalSubrs, c, font);
          callDepth--;
          if (!keepGoing) {
            return keepGoing; }
          break; }

        case Type2CStringOps.RETURN: {
          check(callDepth > 0);
          c.return_op(stack, stackDepth);
          return true; }

        case Type2CStringOps.ESCAPE: {
          int b1 = data.getcard8(offset);
          offset++;

          switch (b1) {
            case Type2CStringOps.ESC_DOTSECTION:
              break;

            case Type2CStringOps.ESC_AND: {
              check(stackDepth >= 2);
              c.and(stack, stackDepth);
              double num2 = stack[--stackDepth];
              double num1 = stack[--stackDepth];
              stack[stackDepth++] = (num1 == 0 || num2 == 0) ? 0 : 1;
              break; }

            case Type2CStringOps.ESC_OR: {
              check(stackDepth >= 2);
              c.or(stack, stackDepth);
              double num2 = stack[--stackDepth];
              double num1 = stack[--stackDepth];
              stack[stackDepth++] = (num1 == 0 && num2 == 0) ? 0 : 1;
              break; }

            case Type2CStringOps.ESC_NOT: {
              check(stackDepth >= 1);
              c.not(stack, stackDepth);
              double num = stack[--stackDepth];
              stack[stackDepth++] = (num == 0) ? 1 : 0;
              break; }

            case Type2CStringOps.ESC_ABS: {
              check(stackDepth >= 1);
              c.abs(stack, stackDepth);
              double num = stack[--stackDepth];
              stack[stackDepth++] = (num < 0) ? (-num) : num;
              break; }

            case Type2CStringOps.ESC_ADD: {
              check(stackDepth >= 2);
              c.add(stack, stackDepth);
              double num2 = stack[--stackDepth];
              double num1 = stack[--stackDepth];
              stack[stackDepth++] = num1 + num2;
              break; }

            case Type2CStringOps.ESC_SUB: {
              check(stackDepth >= 2);
              c.sub(stack, stackDepth);
              double num2 = stack[--stackDepth];
              double num1 = stack[--stackDepth];
              stack[stackDepth++] = num1 - num2;
              break; }

            case Type2CStringOps.ESC_DIV: {
              check(stackDepth >= 2);
              c.div(stack, stackDepth);
              double num2 = stack[--stackDepth];
              double num1 = stack[--stackDepth];
              stack[stackDepth++] = num1 / num2;
              break; }

            case Type2CStringOps.ESC_NEG: {
              check(stackDepth >= 1);
              c.neg(stack, stackDepth);
              double num = stack[--stackDepth];
              stack[stackDepth++] = -num;
              break; }

            case Type2CStringOps.ESC_EQ: {
              check(stackDepth >= 2);
              c.eq(stack, stackDepth);
              double num2 = stack[--stackDepth];
              double num1 = stack[--stackDepth];
              stack[stackDepth++] = (num1 == num2) ? 1 : 0;
              break; }

            case Type2CStringOps.ESC_DROP: {
              check(stackDepth >= 1);
              c.drop(stack, stackDepth);
              --stackDepth;
              break; }

            case Type2CStringOps.ESC_PUT: {
              check(stackDepth >= 2);
              c.put(stack, stackDepth);
              int i = checkInt(stack[--stackDepth]);
              double val = stack[--stackDepth];
              storage[i] = val;
              break; }

            case Type2CStringOps.ESC_GET: {
              check(stackDepth >= 1);
              c.get(stack, stackDepth);
              int i = checkInt(stack[--stackDepth]);
              stack[stackDepth++] = storage[i];
              break; }

            case Type2CStringOps.ESC_IFELSE: {
              check(stackDepth >= 4);
              c.ifelse(stack, stackDepth);
              double v2 = stack[--stackDepth];
              double v1 = stack[--stackDepth];
              double s2 = stack[--stackDepth];
              double s1 = stack[--stackDepth];
              stack[stackDepth++] = (v1 <= v2) ? s1 : s2;
              break; }

            case Type2CStringOps.ESC_RANDOM: {
              check(stackDepth < stack.length);
              c.random(stack, stackDepth);
              stack[stackDepth++] = 0;
              break; }

            case Type2CStringOps.ESC_MUL: {
              check(stackDepth >= 2);
              c.mul(stack, stackDepth);
              double num2 = stack[--stackDepth];
              double num1 = stack[--stackDepth];
              stack[stackDepth++] = num1 * num2;
              break; }

            case Type2CStringOps.ESC_SQRT: {
              check(stackDepth >= 1);
              c.sqrt(stack, stackDepth);
              double num = stack[--stackDepth];
              stack[stackDepth++] = Math.sqrt(num);
              break; }

            case Type2CStringOps.ESC_DUP: {
              check(stackDepth >= 1);
              check(stackDepth < stack.length);
              c.dup(stack, stackDepth);
              double num = stack[--stackDepth];
              stack[stackDepth++] = num;
              stack[stackDepth++] = num;
              break; }

            case Type2CStringOps.ESC_EXCH: {
              check(stackDepth >= 2);
              c.exch(stack, stackDepth);
              double num2 = stack[--stackDepth];
              double num1 = stack[--stackDepth];
              stack[stackDepth++] = num2;
              stack[stackDepth++] = num1;
              break; }

            case Type2CStringOps.ESC_INDEX: {
              check(stackDepth >= 1);
              int i = checkInt(stack[stackDepth]);
              check(stackDepth >= i + 1);
              c.index(stack, stackDepth);
              stackDepth--; // i
              stack[stackDepth++] = stack[stackDepth - i - 1];
              break; }

            case Type2CStringOps.ESC_ROLL: {
              check(stackDepth >= 2);
              check(stack[stackDepth - 2] > 0);
              check(stackDepth >= stack[stackDepth - 2] + 2);
              c.roll(stack, stackDepth);
              int j = checkInt(stack[--stackDepth]);
              int n = checkInt(stack[--stackDepth]);

              if (j < 0) {
                j = n - (-j % n); }
              j %= n;

              int top = stackDepth - 1;
              int bottom = stackDepth - n;

              reverseStackRange(top - j + 1, top);
              reverseStackRange(bottom, top - j);
              reverseStackRange(bottom, top);
              break; }

            case Type2CStringOps.ESC_HFLEX: {
              check(stackDepth == 7);
              c.hflex(stack, stackDepth);
              stackDepth = 0;
              break; }

            case Type2CStringOps.ESC_FLEX: {
              check(stackDepth == 13);
              c.flex(stack, stackDepth);
              stackDepth = 0;
              break; }

            case Type2CStringOps.ESC_HFLEX1: {
              check(stackDepth == 9);
              c.hflex1(stack, stackDepth);
              stackDepth = 0;
              break; }

            case Type2CStringOps.ESC_FLEX1: {
              check(stackDepth == 11);
              c.flex1(stack, stackDepth);
              stackDepth = 0;
              break; }

            case Type2CStringOps.ESC_GLOBALCOLORME: {
              c.globalColorMe(stack, stackDepth);
              continue;  }

            default: { // reserved
              throw new InvalidGlyphException("Invalid Type2 operator (12, " + b1 + ")"); }}

          break; }

        case Type2CStringOps.ENDCHAR: {
          if (seacPhase != SeacPhase.seacNone)
            return false;

          if (stackDepth == 4 || stackDepth == 5) {
            // we have a seac.
            int startOfSeacArgs = 0;

            if (stackDepth == 5) {
              // temporarily adjust stack depth so the width is handled.
              stackDepth = 1; 
              if (!handleWidth(c)) {
                return false; }
              stackDepth = 5;
              startOfSeacArgs = 1; }

            if (c.seac(stack, stackDepth)) {
              seacPhase = SeacPhase.seacBase;

              seacAccentX = stack[startOfSeacArgs++];
              seacAccentY = stack[startOfSeacArgs++];
              int base = (int)stack[startOfSeacArgs++];
              int accent = (int)stack[startOfSeacArgs++];

              if (font == null)
                throw new InvalidGlyphException("Seac found in non-nk font");

              base = font.glyphName2gid(StandardEncoding.names[base]);
              accent = font.glyphName2gid(StandardEncoding.names[accent]);

              stackDepth = 0;
              parse(font.charStrings.data, 
                  font.charStrings.offsetOf(base), 
                  font.charStrings.offsetFollowing(base), 
                  localSubrs, globalSubrs, c, font);

              seacPhase = SeacPhase.seacAccentPreMove;

              nbHints = 0;
              stackDepth = 0;

              parse(font.charStrings.data, 
                  font.charStrings.offsetOf(accent), 
                  font.charStrings.offsetFollowing(accent), 
                  localSubrs, globalSubrs, c, font); }

            stackDepth = 0; }
          
          // does not matter if the consumer had enough, we had enough anyway
          handleWidth(c);
          c.endchar(stack, stackDepth);
          return false; }

        case Type2CStringOps.HSTEMHM: {
          check(stackDepth >= 2);
          nbHints += stackDepth / 2;

          if (seacPhase == SeacPhase.seacAccentPreMove) {
            stack [stackDepth % 2] += seacAccentY; }

          c.hstemhm(stack, stackDepth);
          stackDepth = stackDepth % 2;
          boolean keepGoing = handleWidth(c);
          if (!keepGoing) {
            return false; }
          break; }

        case Type2CStringOps.HINTMASK: {
          if (stackDepth > 1) {
            nbHints += stackDepth / 2;
            if (seacPhase == SeacPhase.seacAccentPreMove) {
              int i = (stackDepth % 2 != 0) ? 1: 0;
              stack[i] += seacAccentX;}
            c.implicit_vstemhm(stack, stackDepth);
            stackDepth = stackDepth % 2; }

          boolean keepGoing = handleWidth (c);
          if (!keepGoing) {
            return false; }

          int nbBytes = (nbHints + 7) / 8;
          c.hintmask(stack, stackDepth, data, offset, nbBytes);
          offset += nbBytes;
          stackDepth = 0;
          break; }

        case Type2CStringOps.CNTRMASK: {
          if (stackDepth > 1) {
            nbHints += stackDepth / 2;
            if (seacPhase == SeacPhase.seacAccentPreMove) {
              int i = (stackDepth % 2 != 0) ? 1: 0;
              stack[i] += seacAccentX;}
            c.implicit_vstemhm(stack, stackDepth);
            stackDepth = stackDepth % 2; }

          boolean keepGoing = handleWidth (c);
          if (!keepGoing) {
            return false; }

          int nbBytes = (nbHints + 7) / 8;
          c.cntrmask(stack, stackDepth, data, offset, nbBytes);
          offset += nbBytes;
          stackDepth = 0;
          break; }

        case Type2CStringOps.RMOVETO: {
          check(stackDepth == 2 || stackDepth == 3);

          if (seacPhase == SeacPhase.seacAccentPreMove) {
            if (stackDepth == 2) {
              stack[0] += seacAccentX;
              stack[1] +=seacAccentY; }
            else {
              stack[1] += seacAccentX;
              stack[2] +=seacAccentY; }
            seacPhase = SeacPhase.seacAccentPostMove;
            c.moveto(stack, stackDepth); }
          else {
            c.rmoveto(stack, stackDepth); }

          stackDepth -= 2;
          boolean keepGoing = handleWidth(c);
          if (!keepGoing) {
            return false; }
          break; }

        case Type2CStringOps.HMOVETO: {
          check(stackDepth == 1 || stackDepth == 2);

          if (seacPhase == SeacPhase.seacAccentPreMove) {
            if (stackDepth == 2) {
              stack[2] = seacAccentY;
              stack[1] += seacAccentX; }
            else  {
              stack[1] = seacAccentY;
              stack[0] += seacAccentX; }
            stackDepth++;
            c.moveto(stack, stackDepth);
            stackDepth -= 2;

            seacPhase = SeacPhase.seacAccentPostMove; }
          else {
            c.hmoveto(stack, stackDepth);
            stackDepth -= 1; }
          
          boolean keepGoing = handleWidth(c);
          if (!keepGoing) {
            return false; }
          break; }

        case Type2CStringOps.VSTEMHM: {
          check(stackDepth >= 2);
          nbHints += stackDepth / 2;

          if (seacPhase == SeacPhase.seacAccentPreMove) {
            int i = (stackDepth % 2 != 0) ? 1: 0;
            stack[i] += seacAccentX;}
          c.vstemhm(stack, stackDepth);
          stackDepth = stackDepth % 2;
          boolean keepGoing = handleWidth(c);
          if (!keepGoing) {
            return false; }
          break; }

        case Type2CStringOps.RCURVELINE: {
          check(stackDepth >= 8 && stackDepth % 6 == 2);
          c.rcurveline(stack, stackDepth);
          stackDepth = 0;
          break; }

        case Type2CStringOps.RLINECURVE: {
          check(stackDepth >= 8 && stackDepth % 2 == 0);
          c.rlinecurve(stack, stackDepth);
          stackDepth = 0;
          break; }

        case Type2CStringOps.VVCURVETO: {
          check(stackDepth >= 4 && stackDepth % 4 <= 1);
          c.vvcurveto(stack, stackDepth);
          stackDepth = 0;
          break; }

        case Type2CStringOps.HHCURVETO: {
          check(stackDepth >= 4 && stackDepth % 4 <= 1);
          c.hhcurveto(stack, stackDepth);
          stackDepth = 0;
          break; 
        }

        case Type2CStringOps.SHORTINT: {
          check(stackDepth < stack.length);
          int val = data.getint16(offset);
          offset += 2;
          c.integer(stack, stackDepth, val);
          stack[stackDepth++] = val;
          break; }

        case Type2CStringOps.CALLGSUBR: { // callgsubr
          check(stackDepth >= 1);
        int subr = checkInt(stack[--stackDepth]);
        if (globalSubrs.getCount() < 1240) {
          subr += 107;}
        else if (globalSubrs.getCount() < 33900) {
          subr += 1131;}
        else {
          subr += 32768;}
        c.callgsubr(stack, stackDepth, subr);

        callDepth++;
        boolean keepGoing = parse(globalSubrs.data, 
            globalSubrs.offsetOf(subr), 
            globalSubrs.offsetFollowing(subr),
            localSubrs, globalSubrs, c, font);
        callDepth--;
        if (!keepGoing) {
          return keepGoing; }
        break; }

        case Type2CStringOps.VHCURVETO: { // vhcurveto
          check(stackDepth >= 4
              && (stackDepth % 8 == 0 || stackDepth % 8 == 1
                  || stackDepth % 8 == 4 || stackDepth % 8 == 5));
        c.vhcurveto(stack, stackDepth);
        stackDepth = 0;
        break; }

        case Type2CStringOps.HVCURVETO: { // hvcurveto
          check(stackDepth >= 4
              && (stackDepth % 8 == 0 || stackDepth % 8 == 1
                  || stackDepth % 8 == 4 || stackDepth % 8 == 5));
          c.hvcurveto(stack, stackDepth);
          stackDepth = 0;
          break; }

        case Type2CStringOps.FIVE_BYTE: {
          check(stackDepth < stack.length);
          double a = data.getint32(offset)/65536.0;
          offset+=4;
          c.real(stack, stackDepth, a);
          stack[stackDepth++] = a;
          break; }

        default: {
          if (32 <= b0 && b0 <= 246) { // no byte integer
            check(stackDepth < stack.length);
            int val = (b0 - 139);
            c.integer(stack, stackDepth, val);
            stack[stackDepth++] = val; }

          else if (247 <= b0 && b0 <= 250) { // one byte integer
            check(stackDepth < stack.length);
            int b1 = data.getcard8(offset);
            offset++;
            int val = ((b0 - 247) * 256 + b1 + 108);
            c.integer(stack, stackDepth, val);
            stack[stackDepth++] = val; }

          else if (251 <= b0 && b0 <= 254) { // one byte integer
            check(stackDepth < stack.length);
            int b1 = data.getcard8(offset);
            offset++;
            int val = (-(b0 - 251) * 256 - b1 - 108);
            c.integer(stack, stackDepth, val);
            stack[stackDepth++] = val; }

          else {
            throw new InvalidGlyphException("invalid Type2 operator (" + b0 + ")"); }

          break; }}}

    check(false);
    return false;
  }

  private void reverseStackRange(int min, int max) {
    while (min < max) {
      double tmp = stack[min];
      stack[min] = stack[max];
      stack[max] = tmp;
      min++;
      max--; }
  }
}
