/*
 * File: Type2UnhintedRedirector.java
 * 
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is
 *      obtained from Adobe Systems Incorporated.
 *
 */

package com.adobe.fontengine.font.cff;

import com.adobe.fontengine.font.OutlineConsumer;

/** Interpret the Type2 parsing events and pass them along to an OutlineConsumer. */
class Type2UnhintedRedirector extends Type2ConsumerDefaultImpl {

  /** The current x position. */
  protected double x;
  /** The current y position. */
  protected double y;
  /***/
  private boolean pathStarted;
  /** The current component x origin. */
  private double pathStartX;
  /** The current component y origin. */
  private double pathStartY;

  /** The OutlineConsumer to which we deliver the raw outline. */
  protected OutlineConsumer outlineConsumer;

  void reset(OutlineConsumer outlineConsumer) {
    this.outlineConsumer = outlineConsumer;
    this.x = 0;
    this.y = 0;
    this.pathStarted = false;
  }

  /** This method is called everytime we generate a line or curve. */
  void startPathIfNeeded (double x, double y) {
    if (! pathStarted) {
      pathStarted = true;
      pathStartX = x;
      pathStartY = y;  }
  }

  public void endchar (double stack[], int stackDepth) {
    if (    pathStarted
        &&  (   Double.compare(this.x, pathStartX) != 0
             || Double.compare(this.y, pathStartY) != 0)) {
      outlineConsumer.lineto(pathStartX, pathStartY);}
    outlineConsumer.endchar();
    pathStarted = false;
  }

  public void rmoveto (double stack[], int stackDepth) {
    pathStarted = false;
    x += stack [stackDepth - 2];
    y += stack [stackDepth - 1];
    outlineConsumer.moveto(x,y);
  }

  public void hmoveto (double stack[], int stackDepth) {
    pathStarted = false;
    x += stack [stackDepth - 1];
    outlineConsumer.moveto(x,y);
  }

  public void vmoveto (double stack[], int stackDepth) {
    pathStarted = false;
    y += stack [stackDepth - 1];
    outlineConsumer.moveto(x,y);
  }

  public void moveto(double stack[], int stackDepth) {
    pathStarted = false;

    x = stack [stackDepth - 2];
    y = stack [stackDepth - 1];
    outlineConsumer.moveto(x,y);
  }


  public void rlineto (double stack[], int stackDepth) {  
    // Note that startPathIfNeeded is called just before calling
    // outlineConsumer.line, rather than outside the loop. The
    // difference of behaviour is when stackDepth is 0:
    // x y moveto rlineto x' y' moveto does not generate a 
    // a line call with our organization, and would with 
    // the alternative organization.
    // Also, we can more easily verify the correctness of our
    // code by checking that there is a startPathIfNeeded just
    // before any call on the consumer.
    for (int i = 0; i < stackDepth; i += 2) {
      startPathIfNeeded (x, y);
      x += stack [i];
      y += stack [i+1];
      outlineConsumer.lineto (x, y);  }
  }

  public void hlineto (double stack[], int stackDepth) {
    for (int i = 0; i < stackDepth; i++) {
      if (i % 2 == 0) {
        startPathIfNeeded (x, y);  
        x += stack [i];    
        outlineConsumer.lineto (x, y);  }
      else {
        startPathIfNeeded (x, y); 
        y += stack [i];     
        outlineConsumer.lineto (x, y); } }
  }

  public void vlineto (double stack[], int stackDepth) {
    for (int i = 0; i < stackDepth; i++) {
      if (i % 2 == 1) {
        startPathIfNeeded (x, y);   
        x += stack [i];   
        outlineConsumer.lineto (x, y); }
      else {
        startPathIfNeeded (x, y);   
        y += stack [i];   
        outlineConsumer.lineto (x, y); }}
  }

  public void rrcurveto (double stack[], int stackDepth) {
    double x2, y2, x3, y3;    
    for (int i = 0; i < stackDepth; i += 6) {
      startPathIfNeeded (x, y); 
      x += stack [i];
      y += stack [i+1];
      x2 = x;
      y2 = y;
      x += stack [i+2];
      y += stack [i+3];
      x3 = x;
      y3 = y;
      x += stack [i+4];
      y += stack [i+5];     
      outlineConsumer.curveto (x2, y2, x3, y3, x, y); }
  }

  public void hhcurveto (double stack[], int stackDepth) {
    double x1, y1, x2, y2, x3, y3;    
    int i = 0;
    x1 = x;
    y1 = y;
    if (stackDepth % 2 == 1) {
      y += stack [i++]; }
    while (i < stackDepth) {
      x += stack [i++];
      x2 = x;
      y2 = y;
      x += stack [i++];
      y += stack [i++];
      x3 = x;
      y3 = y;
      x += stack [i++];
      startPathIfNeeded (x1, y1);      
      outlineConsumer.curveto (x2, y2, x3, y3, x, y);
      x1 = x;
      y1 = y; }
  }

  public void hvcurveto (double stack[], int stackDepth) {
    double x1, y1, x2, y2, x3, y3;    
    int i = 0;
    x1 = x;
    y1 = y;

    if (stackDepth % 8 == 4 || stackDepth % 8 == 5) {
      x += stack [i++];
      x2 = x;
      y2 = y;
      x += stack [i++];
      y += stack [i++];
      x3 = x;
      y3 = y;
      y += stack [i++];
      if (i + 1 == stackDepth) {
        x += stack [i++]; }
      startPathIfNeeded (x1, y1);      
      outlineConsumer.curveto (x2, y2, x3, y3, x, y);
      x1 = x;
      y1 = y;

      while (i + 8 <= stackDepth) {
        y += stack [i++];
        x2 = x;
        y2 = y;
        x += stack [i++];
        y += stack [i++];
        x3 = x;
        y3 = y;
        x += stack [i++];
        startPathIfNeeded (x1, y1);      
        outlineConsumer.curveto (x2, y2, x3, y3, x, y);
        x1 = x;
        y1 = y;

        x += stack [i++];
        x2 = x;
        y2 = y;
        x += stack [i++];
        y += stack [i++];
        x3 = x;
        y3 = y;
        y += stack [i++];
        if (i + 1 == stackDepth) {
          x += stack [i++]; }
        startPathIfNeeded (x1, y1);      
        outlineConsumer.curveto (x2, y2, x3, y3, x, y);
        x1 = x;
        y1 = y;}}

    else {
      while (i + 8 <= stackDepth) {
        x += stack [i++];
        x2 = x;
        y2 = y;
        x += stack [i++];
        y += stack [i++];
        x3 = x;
        y3 = y;
        y += stack [i++];
        startPathIfNeeded (x1, y1);      
        outlineConsumer.curveto (x2, y2, x3, y3, x, y);
        x1 = x;
        y1 = y;

        y += stack [i++];
        x2 = x;
        y2 = y;
        x += stack [i++];
        y += stack [i++];
        x3 = x;
        y3 = y;
        x += stack [i++];
        if (i + 1 == stackDepth) {
          y += stack [i++]; }
        startPathIfNeeded (x1, y1);      
        outlineConsumer.curveto (x2, y2, x3, y3, x, y);
        x1 = x;
        y1 = y; }}
  }

  public void rcurveline (double stack[], int stackDepth) {
    double x1, y1, x2, y2, x3, y3;    
    int i = 0;

    x1 = x;
    y1 = y;

    while (i + 6 <= stackDepth) {
      x += stack [i++];
      y += stack [i++];
      x2 = x;
      y2 = y;
      x += stack [i++];
      y += stack [i++];
      x3 = x;
      y3 = y;
      x += stack [i++];
      y += stack [i++];
      startPathIfNeeded (x1, y1);      
      outlineConsumer.curveto (x2, y2, x3, y3, x, y);
      x1 = x;
      y1 = y; }

    x += stack [i++];
    y += stack [i++];
    startPathIfNeeded (x1, y1);      
    outlineConsumer.lineto (x, y);
  }

  public void rlinecurve (double stack[], int stackDepth) {
    double x1, y1, x2, y2, x3, y3;    
    int i = 0;

    x1 = x;
    y1 = y;
    while (i + 6 < stackDepth) {
      x += stack [i++];
      y += stack [i++];
      startPathIfNeeded (x1, y1);      
      outlineConsumer.lineto (x, y);
      x1 = x;
      y1 = y; }

    x += stack [i++];
    y += stack [i++];
    x2 = x;
    y2 = y;
    x += stack [i++];
    y += stack [i++];
    x3 = x;
    y3 = y;
    x += stack [i++];
    y += stack [i++];
    startPathIfNeeded (x1, y1);      
    outlineConsumer.curveto (x2, y2, x3, y3, x, y);
  }

  public void vvcurveto (double stack[], int stackDepth) {
    double x1, y1, x2, y2, x3, y3;    
    int i = 0;

    x1 = x;
    y1 = y;
    if (stackDepth % 2 == 1) {
      x += stack [i++]; }
    while (i < stackDepth) {
      y += stack [i++];
      x2 = x;
      y2 = y;
      x += stack [i++];
      y += stack [i++];
      x3 = x;
      y3 = y;
      y += stack [i++];
      startPathIfNeeded (x1, y1);      
      outlineConsumer.curveto (x2, y2, x3, y3, x, y);
      x1 = x;
      y1 = y; }
  }

  public void vhcurveto (double stack[], int stackDepth) {
    double x1, y1, x2, y2, x3, y3;    
    int i = 0;
    x1 = x;
    y1 = y;
    if (stackDepth % 8 == 4 || stackDepth % 8 == 5) {
      y += stack [i++];
      x2 = x;
      y2 = y;
      x += stack [i++];
      y += stack [i++];
      x3 = x;
      y3 = y;
      x += stack [i++];
      if (i + 1 == stackDepth) {
        y += stack [i++]; }
      startPathIfNeeded (x1, y1);      
      outlineConsumer.curveto (x2, y2, x3, y3, x, y);
      x1 = x;
      y1 = y;

      while (i + 8 <= stackDepth) {
        x += stack [i++];
        x2 = x;
        y2 = y;
        x += stack [i++];
        y += stack [i++];
        x3 = x;
        y3 = y;
        y += stack [i++];
        startPathIfNeeded (x1, y1);      
        outlineConsumer.curveto (x2, y2, x3, y3, x, y);
        x1 = x;
        y1 = y;

        y += stack [i++];
        x2 = x;
        y2 = y;
        x += stack [i++];
        y += stack [i++];
        x3 = x;
        y3 = y;
        x += stack [i++];
        if (i + 1 == stackDepth) {
          y += stack [i++]; }
        startPathIfNeeded (x1, y1);      
        outlineConsumer.curveto (x2, y2, x3, y3, x, y);
        x1 = x;
        y1 = y; }}

    else {
      while (i + 8 <= stackDepth) {
        y += stack [i++];
        x2 = x;
        y2 = y;
        x += stack [i++];
        y += stack [i++];
        x3 = x;
        y3 = y;
        x += stack [i++];
        startPathIfNeeded (x1, y1);      
        outlineConsumer.curveto (x2, y2, x3, y3, x, y);
        x1 = x;
        y1 = y;

        x += stack [i++];
        x2 = x;
        y2 = y;
        x += stack [i++];
        y += stack [i++];
        x3 = x;
        y3 = y;
        y += stack [i++];
        if (i + 1 == stackDepth) {
          x += stack [i++]; }
        startPathIfNeeded (x1, y1);      
        outlineConsumer.curveto (x2, y2, x3, y3, x, y);
        x1 = x;
        y1 = y; }}
  }

  public void hflex (double stack[], int stackDepth) {
    double x1, y1, x2, y2, x3, y3, starty;    
    int i = 0;

    x1 = x;
    starty = y1 = y;
    x += stack [i++];
    x2 = x;
    y2 = y;
    x += stack [i++];
    y += stack [i++];
    x3 = x;
    y3 = y;
    x += stack [i++];
    startPathIfNeeded (x1, y1);      
    outlineConsumer.curveto (x2, y2, x3, y3, x, y);
    x1 = x;
    y1 = y;

    x += stack [i++];
    x2 = x;
    y2 = y;
    x += stack [i++];
    x3 = x;
    y3 = starty;
    x += stack [i++];
    y = starty;
    startPathIfNeeded (x1, y1);      
    outlineConsumer.curveto (x2, y2, x3, y3, x, starty);
  }

  public void flex (double stack[], int stackDepth) {
    double x1, y1, x2, y2, x3, y3;    
    int i = 0;

    x1 = x;
    y1 = y;
    x += stack [i++];
    y += stack [i++];
    x2 = x;
    y2 = y;
    x += stack [i++];
    y += stack [i++];
    x3 = x;
    y3 = y;
    x += stack [i++];
    y += stack [i++];
    startPathIfNeeded (x1, y1);      
    outlineConsumer.curveto (x2, y2, x3, y3, x, y);
    x1 = x;
    y1 = y;

    x += stack [i++];
    y += stack [i++];
    x2 = x;
    y2 = y;
    x += stack [i++];
    y += stack [i++];
    x3 = x;
    y3 = y;
    x += stack [i++];
    y += stack [i++];
    startPathIfNeeded (x1, y1);      
    outlineConsumer.curveto (x2, y2, x3, y3, x, y);
  }

  public void hflex1 (double stack[], int stackDepth) {
    double x1, y1, x2, y2, x3, y3, startY;    
    int i = 0;

    x1 = x;
    startY = y1 = y;
    x += stack [i++];
    y += stack [i++];
    x2 = x;
    y2 = y;
    x += stack [i++];
    y += stack [i++];
    x3 = x;
    y3 = y;
    x += stack [i++];
    startPathIfNeeded (x1, y1);      
    outlineConsumer.curveto (x2, y2, x3, y3, x, y);
    x1 = x;
    y1 = y;

    x += stack [i++];
    x2 = x;
    y2 = y;
    x += stack [i++];
    y += stack [i++];
    x3 = x;
    y3 = y;
    x += stack [i++];
    y = startY;
    startPathIfNeeded (x1, y1);      
    outlineConsumer.curveto (x2, y2, x3, y3, x, y);
  }

  public void flex1 (double stack[], int stackDepth) {
    double x1, y1, x2, y2, x3, y3;    
    int i = 0;

    double startx = x;
    double starty = y;
    x1 = x;
    y1 = y;
    x += stack [i++];
    y += stack [i++];
    x2 = x;
    y2 = y;
    x += stack [i++];
    y += stack [i++];
    x3 = x;
    y3 = y;
    x += stack [i++];
    y += stack [i++];
    startPathIfNeeded (x1, y1);      
    outlineConsumer.curveto (x2, y2, x3, y3, x, y);
    x1 = x;
    y1 = y;

    x += stack [i++];
    y += stack [i++];
    x2 = x;
    y2 = y;
    x += stack [i++];
    y += stack [i++];
    x3 = x;
    y3 = y;
    if (Math.abs (x - startx) > Math.abs (y - starty)) {
      x += stack [i++]; 
      y = starty;
    }
    else {
      y += stack [i++]; 
      x = startx;
    }
    startPathIfNeeded (x1, y1);      
    outlineConsumer.curveto (x2, y2, x3, y3, x, y);
  }
}