/*
 *  File: Type1CStringParser
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 * 
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 */

package com.adobe.fontengine.font.type1;

import com.adobe.fontengine.font.HintedOutlineConsumer;
import com.adobe.fontengine.font.InvalidGlyphException;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.postscript.SeacPhase;
import com.adobe.fontengine.font.postscript.StandardEncoding;

/**
 * Parses Type1 charstrings
 * 
 * * A <code>Type1CStringParser</code> offers a single method, <code>parse</code> which is given
 * a glyphID, a type1 font, and a </code>HintedOutlineConsumer</code>. As the charString is parsed, the 
 * <code>HintedOutlineConsumer</code> is called for each operator that affects the path. The parser maintains the 
 * stack and storage area, interpreting the operators which affect them,
 * as well as interpreting subroutine calls and seac operators. The intent is that a Consumer 
 * can concentrate on dealing with the path and hints operators, and 
 * ignore the arithmetic, storage, conditional and subroutine 
 * operators. 
 * 
 * <p>The same parser can be used repeatedly to
 * parse multiple charstrings; those charstrings do not need to be
 * in the same font. The parser maintains
 * no state between invocations of <code>parse</code>.
 * 
 * <p>The consumer must not call <code>parse</code> on the parser that 
 * invoked it.
 * 
 * <h4>Synchronization</h4>
 * 
 * This class is not synchronized. Multiple instances can safely
 * coexist without threadsafety issues, but each must only be accessed 
 * from one thread (or must be guarded by the client).
 * The <code>parse</code> method invokes the
 * HintedOutlineConsumer on the thread on which it is called.
 */
final public class Type1CStringParser {

	private static final int CMD_RESERVED0 = 0;
    private static final int CMD_HSTEM = 1;
    private static final int CMD_RESERVED2 = 2;
    private static final int CMD_VSTEM = 3;
    private static final int CMD_VMOVETO = 4;
    private static final int CMD_RLINETO = 5;
    private static final int CMD_HLINETO = 6;
    private static final int CMD_VLINETO = 7;
    private static final int CMD_RRCURVETO = 8;
    private static final int CMD_CLOSEPATH = 9;
    private static final int CMD_CALLSUBR = 10;
    private static final int CMD_RETURN = 11;
    private static final int CMD_ESC = 12;
    private static final int CMD_HSBW = 13;
    private static final int CMD_ENDCHAR = 14;
    private static final int CMD_MOVETO = 15;
    private static final int CMD_BLEND = 16;
    private static final int CMD_RESERVED17 = 17;
    private static final int CMD_RESERVED18 = 18;
    private static final int CMD_RESERVED19 = 19;
    private static final int CMD_RESERVED20 = 20;
    private static final int CMD_RMOVETO = 21;
    private static final int CMD_HMOVETO = 22;
    private static final int CMD_RESERVED23 = 23;
    private static final int CMD_RESERVED24 = 24;
    private static final int CMD_RESERVED25 = 25;
    private static final int CMD_RESERVED26 = 26;
    private static final int CMD_RESERVED27 = 27;
    private static final int CMD_RESERVED28 = 28;
    private static final int CMD_RESERVED29 = 29;
    private static final int CMD_VHCURVETO = 30;
    private static final int CMD_HVCURVETO = 31;
    

    private static final int ESC_DOTSECTION = 0;
    private static final int ESC_VSTEM3 = 1;
    private static final int ESC_HSTEM3 = 2;
    private static final int ESC_AND = 3;
    private static final int ESC_OR = 4;
    private static final int ESC_NOT = 5;
    private static final int ESC_SEAC = 6;
    private static final int ESC_SBW = 7;
    private static final int ESC_STORE = 8;
    private static final int ESC_ABS = 9;
    private static final int ESC_ADD = 10;
    private static final int ESC_SUB = 11;
    private static final int ESC_DIV = 12;
    private static final int ESC_LOAD = 13;
    private static final int ESC_NEG = 14;
    private static final int ESC_EQ = 15;
    private static final int ESC_CALLOTHER = 16;
    private static final int ESC_POP = 17;
    private static final int ESC_DROP = 18;
    private static final int ESC_RESERVED19 = 19; 
    private static final int ESC_PUT = 20;
    private static final int ESC_GET = 21;
    private static final int ESC_IFELSE = 22;
    private static final int ESC_RANDOM = 23;
    private static final int ESC_MUL = 24;
    private static final int ESC_DIV2 = 25;
    private static final int ESC_SQRT = 26;
    private static final int ESC_DUP = 27;
    private static final int ESC_EXCH = 28;
    private static final int ESC_INDEX = 29;
    private static final int ESC_ROLL = 30;
    private static final int ESC_RESERVED31 = 31;
    private static final int ESC_RESERVED32 = 32;
    private static final int ESC_SETCURRENTPOINT = 33;
//    private static final int ESC_HFLEX = 34;
//    private static final int ESC_FLEX = 35;
//    private static final int ESC_HFLEX1 = 36;
//    private static final int ESC_FLEX1 = 37;
//    private static final int ESC_CNTRON = 38;
    
    private static final int OTHER_FLEX = 0;
    private static final int OTHER_PREFLEX1 = 1;
    private static final int OTHER_PREFLEX2 = 2;
    private static final int OTHER_HINTSUBS = 3;
    private static final int OTHER_RESERVED4 = 4;
    private static final int OTHER_GLOBALCOLORME = 6;
    private static final int OTHER_RESERVED7 = 7;
    private static final int OTHER_RESERVED8 = 8;
    private static final int OTHER_RESERVED9 = 9;
    private static final int OTHER_RESERVED10 = 10;
    private static final int OTHER_RESERVED11 = 11;
    private static final int OTHER_GLBCLR1 = 12;
    private static final int OTHER_GLBCLR2 = 13;	
    private static final int OTHER_BLEND1 = 14;
    private static final int OTHER_BLEND2 = 15;
    private static final int OTHER_BLEND3 = 16;
    private static final int OTHER_BLEND4 = 17;
    private static final int OTHER_BLEND6 = 18;
    private static final int OTHER_STOREWV = 19;
    private static final int OTHER_ADD = 20;
    private static final int OTHER_SUB = 21;
    private static final int OTHER_MUL = 22;
    private static final int OTHER_DIV = 23;
    private static final int OTHER_PUT = 24;
    private static final int OTHER_GET = 25;
    private static final int OTHER_PSPUT =  26;
    private static final int OTHER_IFELSE = 27;
    private static final int OTHER_RANDOM = 28;
    private static final int OTHER_DUP = 29;
    private static final int OTHER_EXCH = 30;
    
    private static final int MAX_STACK_SIZE = 24;
    private static final int MAX_STORAGE_SIZE = 32;
    private static final int MAX_FLEX = 17;
	
	private SeacPhase seacPhase;
	private double seac_adx;
	private double seac_ady;
	private double currentX;
	private double currentY;
	private double lsb_x;
	private double lsb_y;
	
    private int stackCount;
    private double[] stack = new double[MAX_STACK_SIZE];
    private double[] storage = new double[32];
    
    private int flexCount;
    private double[] flexArgs = new double[MAX_FLEX];
    
    private int counterCount;
    private double[] counters = new double[96 * 2 + 2];
    
    private boolean inFlex;
    private boolean pendingMove;
    private boolean endcharSeen;
    private boolean pendingHintSub; // with the next stem, a hint substitution needs to happen
    private boolean seenHintSub; // we have done (or are about to do) a hint substition
    private boolean movetoSeen;
    

    private void push(double arg)
    {
        stack[stackCount++] = arg;
    }
    
    private double pop()
    {
        return stack[--stackCount];
    }
    
    private double index(int i)
    {
        return stack[i];
    }
    
    private void clearStack()
    {
        stackCount = 0;
    }
    
    private void pushFlex(double arg)
    {
        flexArgs[flexCount++] = arg;
    }
    
    private void checkFlex(int cnt)
    throws InvalidGlyphException
    {
        if (flexCount + cnt > MAX_FLEX)
            throw new InvalidGlyphException("too many flex args");
    }
    
    private void checkUnderflow(int cnt)
    throws InvalidGlyphException
    {
        if (stackCount < cnt)
            throw new InvalidGlyphException("Stack underflow in charstring");
    }
    
    private void checkOverflow(int cnt)
    throws InvalidGlyphException
    {
        if (stackCount + cnt > MAX_STACK_SIZE)
            throw new InvalidGlyphException("Stack overflow in charstring");
    }
    
    private void doAnd()
    throws InvalidGlyphException
    {
        checkUnderflow(2);
        double a = pop();
        double b = pop();
        if (a != 0 && b != 0)
            push(1);
        else
            push(0);
    }
    
    private void doOr()
    throws InvalidGlyphException
    {
        checkUnderflow(2);
        double a = pop();
        double b = pop();
        if (a != 0 || b != 0)
            push(1);
        else
            push(0);
    }
    
    private void doNot()
    throws InvalidGlyphException
    {
        checkUnderflow(1);
        double a = pop();
        if (a == 0)
            push(1);
        else
            push(0);
    }
    
    private void doAbs()
    throws InvalidGlyphException
    {
        checkUnderflow(1);
        double a = pop();
        if (a < 0)
            push(-a);
        else
            push(a);
    }
    
    private void doAdd()
    throws InvalidGlyphException
    {
        checkUnderflow(2);
        double a = pop();
        double b = pop();
        push(a+b);
    }
    
    private void doSub()
    throws InvalidGlyphException
    {
        checkUnderflow(2);
        double a = pop();
        double b = pop();
        push(b - a);
    }
    
    private void doNeg()
    throws InvalidGlyphException
    {
        checkUnderflow(1);
        double a = pop();
        push(-a);
    }
    
    private void doEq()
    throws InvalidGlyphException
    {
        checkUnderflow(2);
        double a = pop();
        double b = pop();
        if (a == b)
            push(1);
        else
            push(0);
    }
    
    private void doIfElse()
    throws InvalidGlyphException
    {
        checkUnderflow(4);
        
        double c1 = pop();
        double c2 = pop();
        double s1 = pop();
        double s2 = pop();
        
        push((c2 <= c1) ? s2:s1);
    }
    
    private void doSqrt()
    throws InvalidGlyphException
    {
	    double a;
	    checkUnderflow(1);
		a = pop();
		if (a < 0.0)
			throw new InvalidGlyphException("Invalid sqrt argument");
		push(java.lang.Math.sqrt(a));
    }
    
    private void doPut()
    throws InvalidGlyphException
    {
        int index;
       
        checkUnderflow(2);
        index = (int)pop();
        if (index < 0 || index >= MAX_STORAGE_SIZE)
            throw new InvalidGlyphException("Invalid index");
        
        storage[index] = pop();
    }
    
    private void doGet()
    throws InvalidGlyphException
    {
        int index;
        
        checkUnderflow(1);
        index = (int)pop();
        if (index < 0 || index >= MAX_STORAGE_SIZE)
            throw new InvalidGlyphException("Invalid index");
        
        push(storage[index]);
    }
    
    private void doMul()
    throws InvalidGlyphException
    {
        double a, b;
        
        checkUnderflow(2);
        a = pop();
        b = pop();
        push(a*b);
    }
    
    private void doDiv()
    throws InvalidGlyphException
    {
        double a, b;
        checkUnderflow(2);
        a = pop();
        b = pop();
        push(b/a);
    }
    
    private void doDup()
    throws InvalidGlyphException
    {
        double a;
        checkUnderflow(1);
        a = pop();
        checkOverflow(2);
        push(a);
        push(a);
    }
    
    private void doExch()
    throws InvalidGlyphException
    {
        double a, b;
        
        checkUnderflow(2);
        a = pop();
        b = pop();
        push(a);
        push(b);
    }
    
    private void reverse(int i, int j)
	{
    	while (i < j)
    		{
    		double tmp = stack[i];
    		stack[i++] = stack[j];
    		stack[j--] = tmp;
    		}
   	}
    
    private double shiftStem(boolean verticalStem, double stem)
    {
        if (seacPhase == SeacPhase.seacAccentPostMove || seacPhase == SeacPhase.seacAccentPreMove)
        {
            stem += verticalStem ? seac_adx:seac_ady;
        }
        else
        {
            stem += verticalStem ? lsb_x: lsb_y;
        }
        
        return stem;
    }
    
    private void doStem(HintedOutlineConsumer consumer, boolean verticalStem, double stem1, double delta)
    {   
        if (!seenHintSub && movetoSeen)
        {
            seenHintSub = true;
            pendingHintSub = true;
        }
        
        stem1 = shiftStem(verticalStem, stem1);
        consumer.stem(stem1, stem1+delta, pendingHintSub, verticalStem, false);
        
        pendingHintSub = false;
    }
    
    
    private void doStem3(HintedOutlineConsumer consumer, boolean verticalStem)
    throws InvalidGlyphException
    {
        
        double s1, d1, s2, d2, s3, d3;
        
        if (!seenHintSub && movetoSeen)
        {
            seenHintSub = true;
            pendingHintSub = true;
        }
        
        checkUnderflow(6);
        s1 = index(0);
        d1 = index(1);
        s2 = index(2);
        d2 = index(3);
        s3 = index(4);
        d3 = index(5);
        
        s1 = shiftStem(verticalStem, s1);
        s2 = shiftStem(verticalStem, s2);
        s3 = shiftStem(verticalStem, s3);
        
        consumer.stem3(s1, s1 + d1, s2, s2 + d2, s3, s3 + d3, pendingHintSub, verticalStem);
        
        pendingHintSub = false;
    }
    
    
    private void doMoveto(HintedOutlineConsumer consumer, double dx, double dy)
    {
        pendingMove = false;
        movetoSeen = true;
        
        if (seacPhase == SeacPhase.seacAccentPreMove)
        {
            currentX = seac_adx + dx;
            currentY = seac_ady + dy;
            seacPhase = SeacPhase.seacAccentPostMove;
        }
        else
        {
            currentX += dx;
            currentY += dy;
        }
        
        consumer.moveto(currentX, currentY);
    }
    
    private void doLineto(HintedOutlineConsumer consumer, double dx, double dy)
    {
        if (pendingMove)
            doMoveto(consumer, 0,0);
        
        currentX += dx;
        currentY += dy;
        consumer.lineto(currentX, currentY);
    }
    
    private void doCurveto(HintedOutlineConsumer consumer, 
            double a1, double a2, 
            double a3, double a4, 
            double a5, double a6)
    {
        double x1, y1, x2, y2;
        
        if (pendingMove)
            doMoveto(consumer, 0,0);
        
    	x1 = currentX + a1;	
    	y1 = currentY + a2;
    	x2 = x1 + a3; 		
    	y2 = y1 + a4; 
    	currentX = x2 + a5; 
    	currentY = y2 + a6;
    	
    	consumer.curveto(x1, y1, x2, y2, currentX, currentY);	
    }
    
    private void doFlex(HintedOutlineConsumer consumer)
    {
    	
    	if (pendingMove)
    		doMoveto(consumer, 0, 0);	/* Insert missing move */

		double x1 = currentX + flexArgs[0] + flexArgs[2];  
		double y1 = currentY + flexArgs[1] + flexArgs[3];
		double x2 = x1 + flexArgs[4];    
		double y2 = y1 + flexArgs[5];
		double x3 = x2 + flexArgs[6];    
		double y3 = y2 + flexArgs[7];
		double x4 = x3 + flexArgs[8];    
		double y4 = y3 + flexArgs[9];
		double x5 = x4 + flexArgs[10];   	
		double y5 = y4 + flexArgs[11];
		double x6 = x5 + flexArgs[12];   	
		double y6 = y5 + flexArgs[13];

		currentX = x6; 				
		currentY = y6;
		
		consumer.flex(flexArgs[14],
				   x1, y1,
				   x2, y2,
				   x3, y3,
				   x4, y4,
				   x5, y5,
				   x6, y6);
		
		if (seacPhase == SeacPhase.seacNone || seacPhase == SeacPhase.seacBase)
		{
		    currentX = flexArgs[15]; 
		    currentY = flexArgs[16];
		}
    }
    
    private boolean doWidth(HintedOutlineConsumer consumer, double width)
    {
    	if (width < -32000.0f || width > 32000.0f)
    		/* This is an rare case where a number in the charstring was specified
    		   as a 5-byte number but was never followed by a "div" operator which
    		   would have reduced its magnitude. The value is erroneous and cannot
    		   be represented in Type 2 so we reduce to a safe range so as not to
    		   cause subsequent processing problems. */
    		width /= 65536.0;
    	
    	return consumer.width(width);
    }
    
    private int addCounterGroup(HintedOutlineConsumer consumer, int index, boolean isVertical, boolean newGroup)
    {
        double edge1 = 0;
        for (;;)
        {
            double edge0 = edge1 + counters[index--];
            double width = counters[index--];
            edge1 = edge0 + width;
            
            if (width < 0)
            {
                // end of group
                consumer.stem(edge1, edge0, newGroup, isVertical, true);
                break;
            }
            else
            {
                consumer.stem(edge0, edge1, newGroup, isVertical, true);
            }
            newGroup = false;
        }
        
        return index;
    }
    
    /**
     * More counters are on the stack. Add them to the current pile. If we
     * have seen all of the counters, send them to the consumer.
     * 
     * @param consumer The consumer that will receive the counters.
     * @param endOfCounters True iff we have seen all of the counters there are.
     */
    private void addCounters(HintedOutlineConsumer consumer, boolean endOfCounters)
    {
        int i, j, numHorizGroups, numVertGroups, startOfHoriz, startOfVert;
        
        for (i = stackCount - 1; i >= 0; i--)
        {
            counters[counterCount++] = stack[i];
        }
        
        if (!endOfCounters)
            return;

        j = counterCount-1;
        numHorizGroups = (int)counters[j];
        startOfHoriz = j - 1 ;
        for (i = 0; i < numHorizGroups; i++)
        {
            do // look for the end of this group
            {
                j -= 2;
                if (j < 1)
                    return; // invalid counter groups. skip them.
            } while (counters[j] >= 0);
        }
        startOfVert = --j;
        numVertGroups = (int)counters[startOfVert--];
        
        if (numVertGroups == 0 && numHorizGroups == 0)
            consumer.noCounters();
        else
        {
            while (numHorizGroups > 0 || numVertGroups > 0)
            {
                if (numHorizGroups-- > 0)
                {
                    startOfHoriz = addCounterGroup(consumer, startOfHoriz, false, true);
	                if (numVertGroups-- > 0)
	                    startOfVert = addCounterGroup(consumer, startOfVert, true, false);
                }
                else 
                {
                    numVertGroups--;
	                startOfVert = addCounterGroup(consumer, startOfVert, true, true);
                }
            }
        }
    }

    private void internalParse(Type1Font type1Font, byte[] charString, HintedOutlineConsumer consumer)
    throws InvalidGlyphException, UnsupportedFontException
    {
        int i;
		int next;
		
		
        for (i = 0; i < charString.length; i++)
        {
            int byte0 = (charString[i] & 0xff);
            switch(byte0)
            {
    		case CMD_RESERVED0:
    		case CMD_RESERVED2:
    		case CMD_BLEND:
    		case CMD_RESERVED17:
    		case CMD_RESERVED18:
    		case CMD_RESERVED19:
    		case CMD_RESERVED20:
    		case CMD_RESERVED23:
    		case CMD_RESERVED24:
    		case CMD_RESERVED25:
    		case CMD_RESERVED26:
    		case CMD_RESERVED27:
    		case CMD_RESERVED28:
    		case CMD_RESERVED29:
    		    throw new InvalidGlyphException("illegal charstring operator");
    			
    		case CMD_HSTEM:
    		{
    		    checkUnderflow(2);
    		    doStem(consumer, false, index(0), index(1));	    
    			break;
    		}
    			
    		case CMD_VSTEM:
    		{
    		    checkUnderflow(2);
    		    doStem(consumer, true, index(0), index(1));
    			break;
    		}
    			
    		case CMD_VMOVETO:
    		    checkUnderflow(1);
    		    
    			if (inFlex)
    			{
    			    checkFlex(2);
    			    pushFlex(0);
    			    pushFlex(index(0));
    			}
    			else
    			{
    				doMoveto(consumer, 0, index(0));
    			}
    				
    			break;
    		case CMD_RLINETO:
    		{
    		    checkUnderflow(2);
    		    doLineto(consumer, index(0), index(1));
    			break;
    		}
    		
    		case CMD_HLINETO:
    		    checkUnderflow(1);
    		    doLineto(consumer, index(0), 0);
    			break;
    			
    		case CMD_VLINETO:
    		    checkUnderflow(1);
    		    doLineto(consumer, 0, index(0));
    			break;
    			
    		case CMD_RRCURVETO:
    		    checkUnderflow(6);
    		    doCurveto(consumer, 
    		            index(0),index(1),
    		            index(2),index(3),
    		            index(4),index(5));
    		    break;
    		    
    		case CMD_CLOSEPATH:
    		    pendingMove = true;
    		    consumer.closepath();
    			break;
    			
    		case CMD_CALLSUBR:
    		    int subrNumber;
    		    byte[] subr;
    		    int currOffset;
    		    
    		    checkUnderflow(1);
    		    subrNumber = (int)pop();
    		    if (subrNumber < 0 || subrNumber > type1Font.getNumSubrs())
    		        throw new InvalidGlyphException("invalid subr number");
    		    
    		    subr = type1Font.getSubr(subrNumber);
    		    
    		    currOffset = i;
    		    internalParse(type1Font, subr, consumer);
    		    if (endcharSeen)
    		        return;
    		    
    		    i = currOffset;
    			continue;
    			
    		case CMD_RETURN:
    			return;
    			
    		case CMD_ESC:
    		{
    		    if (++i >= charString.length)
    		        throw new InvalidGlyphException("invalid esc command in charstring");
    		    
    		    byte0 = (charString[i] & 0xff);
    		    
    			switch (byte0)
    			{
    			case ESC_DOTSECTION:
    				break;
    				
    			case ESC_AND:
    				doAnd();
    				continue;
    				
    			case ESC_OR:
    				doOr();
    				continue;
    				
    			case ESC_NOT:
    			    doNot();
    				continue;
    				
    			case ESC_STORE:
    			    throw new UnsupportedFontException("mm not supported");
    			    
    			case ESC_ABS:
    			    doAbs();
    				continue;
    				
    			case ESC_ADD:	
    			    doAdd();
    				continue;
    				
    			case ESC_SUB:
    			    doSub();
    				continue;
    			
    			case ESC_LOAD:
    			    throw new UnsupportedFontException("mm not supported");

    			case ESC_NEG:
    			    doNeg();
    				continue;
    				
    			case ESC_EQ:
    			    doEq();
    				continue;
    				
    			case ESC_DROP:
    				checkUnderflow(1);
    				stackCount--;
    				continue;
    				
    			case ESC_PUT:
    			    doPut();
    				continue;
    				
    			case ESC_GET:
    			    doGet();
    				continue;
    				
    			case ESC_IFELSE:
    			    doIfElse();
    				continue;
    				
    			case ESC_RANDOM:
    			    /* XXX_lmb random */
    				continue;
    				
    			case ESC_MUL:
    			    doMul();
    				continue;
    				
    			case ESC_SQRT:
    			    doSqrt();
    				continue;
    				
    			case ESC_DUP:
    			    doDup();
    				continue;
    				
    			case ESC_EXCH:
    			    doExch();
    				continue;
    				
    			case ESC_INDEX:
    			{
    			    int index;
    				checkUnderflow(1);
    				
    				index = (int)pop();
    				if (index < 0)
    				    index = 0;	
    				if (index >= stackCount)
    					throw new InvalidGlyphException("invalid index");
    				push(stack[stackCount - 1 - index]);
    				continue;
    			}
    			
    			case ESC_ROLL:
    			{
    			    int j, n, top, bottom;
    			    
    				checkUnderflow(2);
    				j = (int)pop();
    				n = (int)pop();
    				top = stackCount - 1;
    				bottom = stackCount - n;

    				if (n < 0 || bottom < 0)
    					throw new InvalidGlyphException("invalid roll");

    				
    				if (j < 0)
    					j = n - (-j % n);
    				j %= n;

    				reverse(top - j + 1, top);
    				reverse(bottom, top - j);
    				reverse(bottom, top);
    				continue;
    			}
    			
    			case ESC_VSTEM3:
    			    doStem3(consumer, true);
    				break;
    				
    			case ESC_HSTEM3:
    			    doStem3(consumer, false);
    				break;
    				
    			case ESC_SEAC:
    			{
    			    byte[] nextcharstring;
			    
    			    checkUnderflow(5);
    			    String bChar = StandardEncoding.names[((int)index(3)) & 0xff];
    			    String aChar = StandardEncoding.names[((int)index(4)) & 0xff];
    			    
    			    seac_adx = lsb_x + index(1);
    			    seac_ady = lsb_y + index(2);
    			    
    			    
    			    
    			    seacPhase = SeacPhase.seacBase;
    			    nextcharstring = type1Font.getCharstring(type1Font.glyphName2gid(bChar));
    			    internalParse(type1Font, nextcharstring, consumer);
    			    endcharSeen = false;
    			    
    			    pendingHintSub = true;
    			    
    			    seacPhase = SeacPhase.seacAccentPreMove;
    			    nextcharstring = type1Font.getCharstring(type1Font.glyphName2gid(aChar));
    			    internalParse(type1Font, nextcharstring, consumer);
    				return; // seac must be the last thing in a charstring.
    			}
    			case ESC_SBW:
    			    if (seacPhase == SeacPhase.seacNone)
    				{
	    			    checkUnderflow(4);
	    			    currentY = lsb_y = index(1);
	    			    currentX = lsb_x = index(0);
	    			    if (!doWidth(consumer, index(2)))
	    			        return;
    				}
    				break;
    				
    			case ESC_DIV:
    			    doDiv();
    				continue;
    				
    			case ESC_CALLOTHER:
    			{
    			    int othersubr;

    				checkUnderflow(2);

    				othersubr = (int)pop();
    				/*argcnt = (int)*/ pop();
    				switch (othersubr)
    				{
    				case OTHER_FLEX:
    					checkUnderflow(3);
    					if (flexCount != MAX_FLEX - 3)
    					    throw new InvalidGlyphException("missing flex arguments");
    					pushFlex(index(0));
    					pushFlex(index(1));
    					pushFlex(index(2));
    					doFlex(consumer);
    					break;
    					
    				case OTHER_PREFLEX1:
    				    flexCount = 0;
    				    inFlex = true;
    					continue;
    					
    				case OTHER_PREFLEX2:
    					continue;	
    					
    				case OTHER_HINTSUBS:
    				    pendingHintSub = true;
    				    seenHintSub = true;
    					continue;
    					
    				case OTHER_GLOBALCOLORME:
    				    consumer.globalColorOn();
    					continue;
    					
    				case OTHER_GLBCLR1:
    				    addCounters(consumer, false);
    					break;
    					
    				case OTHER_GLBCLR2:
    				    addCounters(consumer, true);
    					break;
    					
    				case OTHER_BLEND1:
    				case OTHER_BLEND2:
    				case OTHER_BLEND3:
    				case OTHER_BLEND4:
    				case OTHER_BLEND6:
        			    throw new UnsupportedFontException("mm not supported");
    				    
    				    
    				case OTHER_ADD:
    				    doAdd();
    					continue;
    					
    				case OTHER_SUB:
    				    doSub();
    					continue;
    					
    				case OTHER_MUL:
    				    doMul();
    					continue;
    					
    				case OTHER_DIV:
    				    doDiv();
    					continue;
    					
    				case OTHER_PUT:
    					doPut();
    					continue;
    					
    				case OTHER_GET:
    				    doGet();
    					continue;
    					
    				case OTHER_IFELSE:
    				    doIfElse();
    					continue;
    					
    				case OTHER_RANDOM:
    				    /* XXX_lmb random*/
    					continue;
    				case OTHER_DUP:
    				    doDup();
    					continue;
    					
    				case OTHER_EXCH:
    				    doExch();
    					continue;
    					
    				case OTHER_STOREWV:
    					throw new UnsupportedFontException("mm not supported");
    					
    				case OTHER_RESERVED4:
    				case OTHER_RESERVED7:
    				case OTHER_RESERVED8:
    				case OTHER_RESERVED9:
    				case OTHER_RESERVED10:
    				case OTHER_RESERVED11:
    				case OTHER_PSPUT:
    				default:
    					throw new InvalidGlyphException("Invalid OtherSubr");
    				} 		
    			} 		
    			break;
    			
    			case ESC_POP:
    				continue;	
    				
    			case ESC_DIV2:
    			    doDiv();
    				continue;
    				
    			case ESC_SETCURRENTPOINT:
    			    if (inFlex)
    			    {
    			        inFlex = false;
    			    }
    			    else
    			    {
    			        checkUnderflow(2);
    			        currentX = index(0);
    			        currentY = index(1);
    			    }
    				break;
    				
    			case ESC_RESERVED19:
    			case ESC_RESERVED31:
    			case ESC_RESERVED32:
    			default:
    				throw new InvalidGlyphException("illegal operators in charstring");
    			}		
    			break;
    		} 
    		
    		case CMD_HSBW:
    		    if (seacPhase == SeacPhase.seacNone)
    		    {
    		        checkUnderflow(2);
    		        currentX = lsb_x = index(0);
    				currentY = lsb_y = 0;
    				if (!doWidth(consumer, index(1)))
    				    return;
    		    }
    			break;
    			
    		case CMD_ENDCHAR:
    		    pendingMove = true;
    		    endcharSeen = true;
    		    if (seacPhase != SeacPhase.seacBase)
    		        consumer.endchar();
    			return;
    			
    		case CMD_MOVETO:
    		{
    			double secondArg;
    			
    			checkUnderflow(2);
    			secondArg = pop();
    			push(pop() - currentX);
    			push(secondArg - currentY);
    			// turn non-relative moveto into relative and fallthrough...
    		}
    			
    		case CMD_RMOVETO:
    		    checkUnderflow(2);
    		    if (inFlex)
    		    {
    		        checkFlex(2);
    		        pushFlex(index(0));
    		        pushFlex(index(1));
    		    }
    		    else
    		    {
    		        doMoveto(consumer, index(0), index(1));
    		    }
    		    break;
    		
    		case CMD_HMOVETO:
    			checkUnderflow(1);
    			if (inFlex)
    		    {
    		        checkFlex(2);
    		        pushFlex(index(0));
    		        pushFlex(0);
    		    }
    		    else
    		    {
    		        doMoveto(consumer, index(0), 0);
    		    }
    			break;
    			
    		case CMD_VHCURVETO:
    			checkUnderflow(4);
    			
    			doCurveto(consumer,
						  0, index(0),
						  index(1), index(2),
						  index(3), 0);
    			break;
    			
    		case CMD_HVCURVETO:
    		{
    			checkUnderflow(4);
    			doCurveto(consumer,
						  index(0), 0,
						  index(1), index(2),
						  0, index(3));
    			break;
    		}
    			
    		default:		
    		    checkOverflow(1);
    			push(byte0 - 139);
    			continue;
    			
    		case 247: 
    		case 248: 
    		case 249: 
    		case 250:
    			
    			checkOverflow(1);
    			
    			if (++i >= charString.length)
    		        throw new InvalidGlyphException("invalid esc command in charstring");
    		    
    			next = (charString[i] & 0xff);
    			push(108 + 256*(byte0 - 247) + next);
    			continue;
    			
    		case 251: 
    		case 252: 
    		case 253: 
    		case 254:
    			
    		    checkOverflow(1);
    		    if (++i >= charString.length)
    		        throw new InvalidGlyphException("invalid esc command in charstring");
    		    
    			next = (charString[i] & 0xff);
    			push(-108 - 256*(byte0 - 251) - next);
    			continue;
    			
    		case 255:
    			
    		    checkOverflow(1);
    		    
    		    if (i+4 >= charString.length)
    		        throw new InvalidGlyphException("invalid esc command in charstring");
    		    
    			int value;
    			value = (charString[++i] & 0xff);
    			value = value<<8 | (charString[++i] & 0xff);
    			value = value<<8 | (charString[++i] & 0xff);
    			value = value<<8 | (charString[++i] & 0xff);

    			push(value);
    			continue;
    		} 				
    							
            clearStack();
    	}
        
        // we didn't see an endchar. add one now.
        consumer.endchar();
    }
    
    /**
     * Decode a type1 charstring
     * @param consumer the HintedOutlineConsumer to alert of parsing events
     * @param type1Font The Type1Font from which charstrings and subroutines should be pulled
     * @param glyphID The glyphID of the charstring to be interpretted
     * @throws InvalidGlyphException The charstring could not be interpretted
     * @throws UnsupportedFontException thrown when the font contains data indicating that this
     * class of fonts is not supported.
     */
    public void parse(HintedOutlineConsumer consumer, Type1Font type1Font, int glyphID)
    throws InvalidGlyphException, UnsupportedFontException
    {
        byte[] charstring = type1Font.getCharstring(glyphID);
        currentX = currentY = lsb_x = lsb_y = 0;
        seacPhase = SeacPhase.seacNone;
        stackCount = 0;
        counterCount = 0;
        inFlex = false;
        pendingMove = true;
        endcharSeen = false;
        pendingHintSub = false;
        seenHintSub = false;
        movetoSeen = false;
        
        internalParse(type1Font, charstring, consumer);
    }
}
