/*
*
*	File: Type1OutlineParser.java
*
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2004-2005 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
*/
package com.adobe.fontengine.font.type1;

import com.adobe.fontengine.font.OutlineConsumer;
import com.adobe.fontengine.font.InvalidGlyphException;
import com.adobe.fontengine.font.UnsupportedFontException;

/** Extract the unhinted bezier outlines from a Type1 charstring.
 * 
 * A <code>Type1OutlineParser</code> offers a single method, <code>parse</code> which is 
 * given a Type1Font, a glyphID and an <code>OutlineConsumer</code>. As the 
 * lines and curves of the charstring are parsed, 
 * the <code>OutlineConsumer</code> is called.
 * 
 * <p>The same <code>Type1OutlineParser</code> object can be used repeatedly
 * to extract outlines. The successive invocations do not need to be for
 * the same font, nor do they need to use
 * the same consumer. 
 * 
 * <p>The consumer must not call <code>parse</code> on the Type1OutlineParser 
 * that invoked it.
 * 
 * <h4>Synchronization</h4>
 * 
 * This class is not synchronized. Multiple instances can safely
 * coexist without threadsafety issues, but each must only be accessed 
 * from one thread (or must be guarded by the client).
 * 
 * The <code>parse</code> method invokes
 * the <code>OutlineConsumer</code> on the thread on which it is called.
 */
final public class Type1OutlineParser {
    private final Type1CStringParser parser = new Type1CStringParser();
    private OutlineConsumer c;
    private T1OutlineConsumer outlineconsumer = new T1OutlineConsumer();
    double cpx,cpy; // the currentpoint
    double pathStartX, pathStartY;
    boolean pathStarted;
    
    
    
    public void parse(Type1Font font, int glyphID, OutlineConsumer consumer)
    throws InvalidGlyphException, UnsupportedFontException
    {
        cpx = cpy = 0;
        pathStarted = false;
        c = consumer;
        
        consumer.setMatrix(font.getFontMatrix());
        parser.parse(outlineconsumer, font, glyphID);
    }
    
    private class T1OutlineConsumer extends Type1CStringConsumerDefault
    {    	
        private void startPath(double x, double y)
        {
            if (!pathStarted)
            {
                pathStarted = true;
                pathStartX = x;
                pathStartY = y;
            }
        }
        
        public void closepath()
        {
            if (pathStarted)
            {
                c.lineto(pathStartX, pathStartY);
                pathStarted = false;
            }
        }
        
        public void endchar()
        {
            if (pathStarted)
                c.lineto(pathStartX, pathStartY);
            c.endchar();
        }
        
        public void lineto(double x, double y)
        {
            startPath(cpx,cpy);
            cpx = x;
            cpy = y;
            c.lineto(x,y);
        }
        
        public void curveto(double x1, double y1, 
                			double x2, double y2, 
                			double x3, double y3)
        {
            startPath(cpx,cpy);
            cpx = x3; 
            cpy = y3;
            c.curveto(x1, y1, x2, y2, x3, y3);
        }
        
        public void moveto(double x, double y)
        {
            pathStarted = false;
            cpx = x;
            cpy = y;
            c.moveto(x,y);
        }
        
        public void flex(double depth,
                			double x1, double y1,
                			double x2, double y2, 
                			double x3, double y3,
                			double x4, double y4,
                			double x5, double y5,
                			double x6, double y6)
        {
            curveto(x1,y1,x2,y2,x3,y3);
            curveto(x4,y4,x5,y5,x6,y6);
        }
    }
}
