/*
 *
 *	File: PSNameFontDatabase.java
 *
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */

package com.adobe.fontengine.fontmanagement.postscript;

import java.util.HashMap;
import java.util.Iterator;
import java.util.TreeSet;

import com.adobe.fontengine.font.Font;
import com.adobe.fontengine.font.FontLoadingException;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.fontmanagement.FontResolutionPriority;
import com.adobe.fontengine.fontmanagement.IntelligentResolver;
import com.adobe.fontengine.inlineformatting.css20.FamilyNameNormalizer;
import com.adobe.fontengine.inlineformatting.css20.PassThroughFamilyNameNormalizer;

final public class PSNameFontDatabase implements PSNameResolver 
{
	static final long serialVersionUID = 1;

	private HashMap/*<String, Font>*/ psFonts;
	private FontResolutionPriority resolutionPriority = FontResolutionPriority.FIRST;
	private final FamilyNameNormalizer normalizer;

	public PSNameFontDatabase () 
	{
		this (new PassThroughFamilyNameNormalizer ());
	}

	public PSNameFontDatabase (FamilyNameNormalizer normalizer)
	{
		this.psFonts = new HashMap/*<PostscriptFontDescription,Font>*/();
		if (normalizer == null)
		{
			normalizer = new PassThroughFamilyNameNormalizer();
		}
		this.normalizer = normalizer;
	}    

	public PSNameFontDatabase(PSNameFontDatabase original)
	{
		this.psFonts = (HashMap/*<PostscriptFontDescription,Font>*/) original.psFonts.clone();
		this.normalizer = original.normalizer;
	}

	public void addFont(Font font) throws UnsupportedFontException, InvalidFontException, FontLoadingException
	{
		PostscriptFontDescription[] fontDesc = font.getPostscriptFontDescription();

		for (int i = 0; i < fontDesc.length; i++) 
		{
			this.addFont (fontDesc[i], font); 
		}
	}

	public void addFont(PostscriptFontDescription psDesc, Font font)
	throws UnsupportedFontException, InvalidFontException, FontLoadingException 
	{
		String name = normalizer.normalize(psDesc.getPSName ());
		Font oldFont = (Font) psFonts.get(name);
		if (oldFont != null) 
		{
			if (this.resolutionPriority == FontResolutionPriority.FIRST) 
			{
				// don't add as the old one has priority
				return; 
			}
			else if (this.resolutionPriority == FontResolutionPriority.INTELLIGENT_LAST
					|| this.resolutionPriority == FontResolutionPriority.INTELLIGENT_FIRST) 
			{
				Font preferredFont = IntelligentResolver.choosePreferredFont(oldFont, font,
						this.resolutionPriority == FontResolutionPriority.INTELLIGENT_FIRST);
				if (preferredFont == oldFont) 
				{
					return; 
				}
			}
		}
		psFonts.put (name, font);
	}

	public boolean isEmpty()
	{
		return psFonts.isEmpty();
	}

	/** @deprecated This method is deprecated. Use {@link #findFont(String)} instead. */
	public Font findFont(PostscriptFontDescription psDesc)
	{
		return findFont(psDesc.getPSName ());
	}

	public Font findFont(String psName)
	{
		return (Font) psFonts.get (normalizer.normalize (psName));
	}

	public FontResolutionPriority setResolutionPriority(FontResolutionPriority priority)
	{
		FontResolutionPriority oldPriority = this.resolutionPriority;

		this.resolutionPriority = priority;
		return oldPriority;
	}

	public boolean equals(Object obj)
	{
		if (obj == null) 
		{
			return false; 
		}

		if (this == obj) 
		{
			return true; 
		}

		return this.psFonts.equals(((PSNameFontDatabase)obj).psFonts);
	}

	public int hashCode()
	{
		return this.psFonts.hashCode();
	}

	public String toString()
	{
		TreeSet tempSet = new TreeSet();
		for (Iterator it = psFonts.keySet().iterator(); it.hasNext(); ) {
			tempSet.add(it.next());
		}
		StringBuffer sb = new StringBuffer();
		sb.append("priority = ");
		sb.append(resolutionPriority.toString());
		sb.append("; PS names = ");
		String prefix = "";
		for (Iterator it = tempSet.iterator(); it.hasNext(); ) {
			sb.append(prefix);
			sb.append((String) it.next());
			prefix = ", ";
		}
		return sb.toString();
	}
}
