/*
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe
 *  All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 */

package com.adobe.granite.contexthub.api;

import java.util.EnumSet;
import java.util.List;
import java.util.Set;

import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.SlingHttpServletResponse;
import org.apache.sling.api.resource.Resource;
import org.osgi.annotation.versioning.ProviderType;

@ProviderType
public interface ContextHub {
    /**
     * Defines possible states based on which stores can be filtered.
     */
    public static enum StoreStatus {
        ENABLED, DISABLED
    }

    /**
     * Defines possible states based on which modules can be filtered.
     */
    public static enum ModuleStatus {
        ENABLED, DISABLED
    }

    /**
     * Defines possible states based on which modes can be filtered.
     */
    public static enum ModeStatus {
        ENABLED, DISABLED
    }

    /**
     * Returns Path to ContextHub
     *
     * @param request request
     * @return URL to contexthub
     */
    public String getPath(SlingHttpServletRequest request);

    /**
     * Returns Path to ContextHub
     *
     * @param request request
     * @param checkRequestResource indicates whether request resource should be used to resolve ContextHub path
     * @return URL to contexthub
     */
    public String getPath(SlingHttpServletRequest request, boolean checkRequestResource);

    /**
     * Returns instance's run modes.
     *
     * @return list of run modes
     */
    public Set<String> getRunModes();

    /**
     * Given a ContextHub configuration resource, return the corresponding {@link Store}s as list
     *
     * @param contextHub the ContextHub configuration resource
     * @return list of stores
     */
    public List<Store> findStores(Resource contextHub);

    /**
     * Given a ContextHub configuration resource, return the corresponding {@link Module}s as list
     *
     * @param contextHub the ContextHub configuration resource
     * @return list of modules
     */
    public List<Module> findModules(Resource contextHub);

    /**
     * Given a ContextHub configuration resource, return the corresponding {@link Mode}s as list
     *
     * @param contextHub the ContextHub configuration resource
     * @return list of modes
     */
    public List<Mode> findModes(Resource contextHub);

    /**
     * Returns all stores matching to requested filter.
     *
     * @param contextHub the ContextHub configuration resource
     * @param flags filter ({@code EnumSet} containing desired status of the store)
     * @return stores matching to requested filter
     */
    public List<Store> filterStores(Resource contextHub, EnumSet<StoreStatus> flags);

    /**
     * Returns all modules matching to requested filter.
     *
     * @param contextHub the ContextHub configuration resource
     * @param flags filter ({@code EnumSet} containing desired status of the store)
     * @return modules matching to requested filter
     */
    public List<Module> filterModules(Resource contextHub, EnumSet<ModuleStatus> flags);

    /**
     * Returns all enabled ContextHub's stores.
     *
     * @param contextHub the ContextHub configuration resource
     * @return enabled stores
     */
    public List<Store> getEnabledStores(Resource contextHub);

    /**
     * Returns all visible ContextHub modules.
     *
     * @param contextHub the ContextHub configuration resource
     * @return enabled modules
     */
    public List<Module> getEnabledModules(Resource contextHub);

    /**
     * Returns all visible ContextHub modes.
     *
     * @param contextHub the ContextHub configuration resource
     * @return enabled modes
     */
    public List<Mode> getEnabledModes(Resource contextHub);


    /**
     * Returns all disabled ContextHub's stores.
     *
     * @param contextHub the ContextHub configuration resource
     * @return disabled stores
     */
    public List<Store> getDisabledStores(Resource contextHub);

    /**
     * Returns all hidden ContextHub modules.
     *
     * @param contextHub the ContextHub configuration resource
     * @return hidden modules
     */
    public List<Module> getDisabledModules(Resource contextHub);

    /**
     * Returns all hidden ContextHub modes.
     *
     * @param contextHub the ContextHub configuration resource
     * @return hidden modes
     */
    public List<Mode> getDisabledModes(Resource contextHub);


    /**
     * Returns ContextHub's UI theme name.
     *
     * @param contextHub the ContextHub configuration resource
     * @return UI theme name
     */
    public String getTheme(Resource contextHub);

    /**
     * Determines if ContextHub's instance is ready to use (it's {@code false} when ContextHub's resource wasn't found).
     *
     * @param contextHub the ContextHub configuration resource
     * @return {@code true} if ContextHub was properly initialized
     */
    public boolean isConfigured(Resource contextHub);

    /**
     * Determines if ContextHub is configured for this request.
     *
     * @param request request
     * @return {@code true} if ContextHub was properly initialized
     */
    public boolean isConfigured(SlingHttpServletRequest request);

    /**
     * Returns URL to ContextHub's kernel.js
     *
     * @param request request
     * @return URL to kernel.js
     */
    public String getKernelCodeURL(SlingHttpServletRequest request);

    /**
     * Returns the URL to ContextHub's ui.js
     *
     * @param request request
     * @return URL to ui.js
     */
    public String getUICodeURL(SlingHttpServletRequest request);

    /**
     * Returns URL to ContextHub's styles.css (containing ui source code)
     *
     * @param request request
     * @return URL to styles.css
     */
    public String getStylesURL(SlingHttpServletRequest request);

    /**
     * Returns URL to ContextHub's UI page (which is "static", only the code is "dynamic")
     *
     * @param request request
     * @return URL to ui.html
     */
    public String getUIPageURL(SlingHttpServletRequest request);

    /**
     * Returns a list of kernel client library categories (including javascript part of ui libraries if includeUI is
     * set to {@code true}).
     *
     * @param request request
     * @return list of categories required to build code.js
     */
    public List<String> getKernelCategories(SlingHttpServletRequest request);

    /**
     * Returns a list of ui (type: js) client library categories.
     *
     * @param request request
     * @return list of categories required to build styles.css
     */
    public List<String> getUICategories(SlingHttpServletRequest request);

    /**
     * Returns a list of ui (type: css) client library categories.
     *
     * @param request request
     * @return list of categories required to build styles.css
     */
    public List<String> getStylesCategories(SlingHttpServletRequest request);

    /**
     * Returns generated output of kernel.js.
     *
     * Output is built based on categories returned by {@link #getKernelCategories(org.apache.sling.api.SlingHttpServletRequest)} and
     * minimized if not set to debug.
     *
     * @param request request
     * @param response response
     * @return ContextHub's kernel JavaScript code
     */
    public String getKernelCode(SlingHttpServletRequest request, SlingHttpServletResponse response);

    /**
     * Returns generated output of ui.js. Output is build basing on categories returned by
     * {@link #getUICategories(org.apache.sling.api.SlingHttpServletRequest)} and minimized if not set to debug.
     *
     * @param request request
     * @param response response
     * @return output of ui.js
     */
    public String getUICode(SlingHttpServletRequest request, SlingHttpServletResponse response);

    /**
     * Returns generated output of styles.css. Output is build basing on categories returned by
     * {@link #getUICategories(org.apache.sling.api.SlingHttpServletRequest)} and minimized if not set to debug.
     *
     * @param request request
     * @return output of styles.css
     */
    public String getStyles(SlingHttpServletRequest request);

    /**
     * Returns whether to include the ContextHub UI or not.
     *
     * @return {@code true} if ContextHub UI is to be included
     */
    public boolean showUi();

    /**
     * Returns whether ContextHub should run in silent mode (takes priority over debug mode).
     *
     * @return {@code true} if debug is enabled
     */
    public boolean isSilentMode();

    /**
     * Returns whether debug is enabled for a given ContextHub instance.
     *
     * @return {@code true} if debug is enabled
     */
    public boolean isDebug(SlingHttpServletRequest request);

    /**
     * Returns whether a given ContextHub instance is disabled.
     *
     * @param request - request
     * @return {@code true} if a given ContextHub instance is disabled.
     */
    public boolean isDisabled(SlingHttpServletRequest request);

    /**
     * Returns home path of anonymous user.
     *
     * @return {String|null}
     */
    public String getAnonymousPath();
}
