/*************************************************************************
* ADOBE CONFIDENTIAL
* ___________________
*
* Copyright 2014 Adobe
* All Rights Reserved.
*
* NOTICE: All information contained herein is, and remains
* the property of Adobe and its suppliers, if any. The intellectual
* and technical concepts contained herein are proprietary to Adobe
* and its suppliers and are protected by all applicable intellectual
* property laws, including trade secret and copyright laws.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe.
**************************************************************************/
package com.adobe.granite.ui.components;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Set;

import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import javax.servlet.http.Cookie;

import org.apache.sling.api.SlingHttpServletRequest;

/**
 * A key-value map of client-side state. A client may set a state such that the
 * server can retrieve it to do something specific about this client based on
 * the state.
 */
public class State {
    private SlingHttpServletRequest request;
    private Set<String> names;

    public State(@Nonnull SlingHttpServletRequest request) {
        this.request = request;
    }

    /**
     * Returns the item value of the given name. Returns {@code null} if the
     * item is not found.
     * @param name the name of the item
     * @return the item value as a string
     */
    @CheckForNull
    public String get(@Nonnull String name) {
        Item i = getItem(name);
        return i == null ? null : i.getString();
    }

    /**
     * Returns the item value of the given name. Returns the given defaultValue if
     * the item is not found.
     * @param name the name of the item
     * @param defaultValue the default value
     * @return the item value as a string or the default value if the item is not found
     */
    @Nonnull
    public String get(@Nonnull String name, @Nonnull String defaultValue) {
        Item i = getItem(name);
        return i == null ? defaultValue : i.getString();
    }

    /**
     * Returns the item value of the given name. Returns the given defaultValue if
     * the item is not found, or the value is not a boolean string ("true" or "false").
     * @param name the name of the item
     * @param defaultValue the default value
     * @return the item value as a boolean or the default value if the value is not a
     * boolean string
     */
    public boolean get(@Nonnull String name, boolean defaultValue) {
        Item i = getItem(name);

        if (i == null || (!"true".equals(i.getString()) && !"false".equals(i.getString()))) {
            return defaultValue;
        }

        return i.getBoolean();
    }

    /**
     * Returns the item value of the given name. Returns the given defaultValue if
     * the item is not found, or the value is not an integer string.
     * @param name the name of the item
     * @param defaultValue the default value
     * @return the item value as an int or the default value if the value is not an
     * int
     */
    public int get(@Nonnull String name, int defaultValue) {
        try {
            Item i = getItem(name);
            return i == null ? defaultValue : i.getInt();
        } catch (NumberFormatException e) {
            return defaultValue;
        }
    }

    /**
     * Returns the available names.
     * @return the available names
     */
    @SuppressWarnings("null")
    @Nonnull
    public Iterator<String> names() {
        if (names == null) {
            names = new LinkedHashSet<String>();

            Cookie[] cookies = request.getCookies();

            if (cookies != null) {
                for (Cookie c : cookies) {
                    if (!c.isHttpOnly()) {
                        names.add(c.getName());
                    }
                }
            }
        }

        return names.iterator();
    }

    /**
     * Returns the state item with the given name. Returns {@code null} if the
     * item is not found.
     * @param name the name of the item
     * @return the state item
     */
    @SuppressWarnings("null")
    @CheckForNull
    public Item getItem(@Nonnull String name) {
        Cookie c = request.getCookie(name);

        if (c == null || c.isHttpOnly()) {
            return null;
        }

        try {
            return new ItemImpl(name, URLDecoder.decode(c.getValue(), "utf-8"));
        } catch (UnsupportedEncodingException impossible) {
            throw new RuntimeException(impossible);
        }
    }

    /**
     * An item in the state.
     */
    public interface Item {
        /**
         * Returns the name of the item.
         * @return the name of the item
         */
        @Nonnull
        String getName();

        /**
         * Returns the value as string.
         * @return the value as string
         */
        @Nonnull
        String getString();

        /**
         * Returns the value as boolean.
         * The conversion is following {@link Boolean#parseBoolean(String)} semantic.
         * @return the value as boolean
         */
        boolean getBoolean();

        /**
         * Returns the value as int.
         * The conversion is following {@link Integer#parseInt(String)} semantic.
         * @return the value as int
         */
        int getInt() throws NumberFormatException;
    }

    private class ItemImpl implements Item {
        @Nonnull
        private String name;

        @Nonnull
        private String value;

        public ItemImpl(@Nonnull String name, @Nonnull String value) {
            this.name = name;
            this.value = value;
        }

        @Override
        @Nonnull
        public String getName() {
            return name;
        }

        @Override
        @Nonnull
        public String getString() {
            return value;
        }

        @Override
        public boolean getBoolean() {
            return Boolean.parseBoolean(value);
        }

        @Override
        public int getInt() throws NumberFormatException {
            return Integer.parseInt(value);
        }
    }
}
