/* ****************************************************************************
 *
 *	File: CosContainer.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.cos;

import java.io.IOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;

import com.adobe.internal.io.stream.InputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.types.ASName;

/**
 * Base class for all COS container object (e.g. COSDictionary, CosArray).
 */
public abstract class CosContainer extends CosObject
{
	// If this container is direct, this is set to parent container's object number.
	// This is used in encryption of strings since they are direct and use their
	// parent's object number.  If this is direct, it will have to use its parent's
	// object number for its strings.
	protected CosContainer mParentObj;
	private static final ASName k_endobj = ASName.create("endobj");

	/**
	 * @see com.adobe.internal.pdftoolkit.core.cos.CosObject#CosObject(CosDocument, CosObjectInfo)
	 */
	CosContainer(CosDocument doc, CosObjectInfo info)
	{
		super(doc, info);
	}

	/**
	 *
	 * Set this value if this container is direct and is being added to a container.
	 *
	 * @param obj - CosObjectInfo
	 */
	abstract void setParentObj(CosContainer obj);

	/**
	 * Get parent's object info.  The value is initialized to null (the free
	 * root object) so if this value is not null it has been set.
	 *
	 * @return CosObjectInfo parent's object info.
	 */
	CosContainer getParentObj()
	{
		return mParentObj;
	}

	/**
	 * Mark object as not dirty if it was dirty.
	 * Returns false if not dirty.
	 */
	@Override
	public boolean markNotDirty()
	{
		if (isIndirect())
			return super.markNotDirty();
		if (mParentObj != null)
			return mParentObj.markNotDirty();
		return false;
	}

	/**
	 * Get the position of specific keys in container objects.
	 */
	public Map findObjPos(Map keys)
		throws PDFIOException
	{
		try {
			Map retVal = new HashMap();
			keys.put(k_endobj, Integer.valueOf(1));
			long curPos = getObjPos();
			InputByteStream fileStream = getDocument().getStreamRaw();
			long endPos = fileStream.length();
			fileStream.seek(curPos);
			CosParseBuf pBuf = new CosParseBuf(fileStream, 4096);
			int[] indexes = new int[keys.size()];			// Count of number of instance to search
			byte[][] byteKeys = new byte[keys.size()][];		// Bytes of ASName to be searched
			ASName[] names = new ASName[keys.size()];		// ASNames
			int[] locIndex = new int[keys.size()];			// Partial match index of ASName
	
			// Initialize all values
			int k = 0;
			for (Iterator it = keys.keySet().iterator(); it.hasNext(); k++) {
				names[k] = (ASName)it.next();
				byteKeys[k] = names[k].getBytes();
				indexes[k] = ((Integer)keys.get(names[k])).intValue();
				Long[] location = new Long[((Integer)keys.get(names[k])).intValue()];
				for (int i = 0; i < location.length; i++)
					location[i] = Long.valueOf(0);		// Dummy values
				retVal.put(names[k], location);
			}
			while (curPos < endPos) {
				int c = pBuf.read();
				curPos++;
				boolean done = true;				// Is true when all keys are retrieved
				for (k = 0; k < byteKeys.length; k++) {
					if (indexes[k] > 0) {			// Check if num instances have been found
						done = false;			// Not all ASName matches are done
						// Partial Match - Match of char with the byte index
						if ((byte)c == byteKeys[k][locIndex[k]])
							locIndex[k]++;		// Increment for checking next byte
						else
							locIndex[k] = 0;	// No match, reset
						// Got a full match
						if (locIndex[k] == byteKeys[k].length) {
							// Update the return value
							Long[] location = (Long[])retVal.get(names[k]);
							location[((Integer)keys.get(names[k])).intValue() - indexes[k]] = Long.valueOf(curPos);
							indexes[k]--;
							locIndex[k] = 0;	// Reset for next instance, if any
							if (names[k] == k_endobj)
								return retVal;
						}
					}
				}
				if (done)
					break;
			}
			return retVal;
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}
	
	/**
	 * returns 0 if pair already exists with CosObject of smaller object number as key and other one in the 
	 * corresponding value set.
	 * returns 1 if CosObject with smaller object number exists in Map as key 
	 * but it doesn't have another object in it's value set.
	 * returns 2 if CosObject with smaller object number doesn't exist in Map as key.
	 * @param objectNums array of object numbers of CosObjects to be searched in list.
	 * @param alreadyComparedCosObjectPairsList
	 * @return int 0 or 1 or 2
	 */
	protected int cosObjectPairAlreadyInList(Integer objectNums[], HashMap<Integer, HashSet<Integer>> alreadyComparedCosObjectPairsList){
		///sorting two Object numbers in ascending order.
		if(objectNums[0] > objectNums[1]){ 
			int temp = objectNums[0];
			objectNums[0] = objectNums[1];
			objectNums[1] = temp;
		}
		
		////checking if these two indirect CosObjects have already been compared.
			HashSet<Integer> matchedCosObjectsSet = alreadyComparedCosObjectPairsList.get(objectNums[0]);
			if((matchedCosObjectsSet != null)){
				if(matchedCosObjectsSet.contains(objectNums[1]))
					return 0;
				return 1;
			}
		return 2;
	}
	
	
	/**
	 * Checks if CosObject pair ("this" and "CosObject with object number objNum") is already in Map.
	 * If not, then adds it and returns true otherwise returns false.
	 * @param objNum
	 * @param alreadyComparedCosObjectPairsList
	 * @return boolean
	 */
	protected boolean addCosObjectPair(int objNum, HashMap<Integer, HashSet<Integer>> alreadyComparedCosObjectPairsList){
		Integer objectNums[] = new Integer[]{this.getObjNum(),objNum};
		
		/// This "if" will ensure that we are not inserting direct CosObjects in Map. because 
		///Gibson returns object number as 0 in case of direct CosObjects.
		if(!(objectNums[0] > 0 && objectNums[1] > 0))
			return true;
		
		////checks if pair already exists.
		int pairExistenceStatus = cosObjectPairAlreadyInList(objectNums, alreadyComparedCosObjectPairsList);
			
		
		/// Pair is always being added with smaller CosObject number as key. 
		///This will avoid duplicate entries and also double checking for pair in list.
		switch(pairExistenceStatus){
			case 0:
				return false;/// pair is already in list.
			case 1:
				/// CosObject with smaller object number exists in Map as key 
				/// but it doesn't have another object in it's value set.
				alreadyComparedCosObjectPairsList.get(objectNums[0]).add(objectNums[1]);
				break;
			case 2:
				/// CosObject with smaller object number doesn't exist in Map as key.
				HashSet<Integer> matchedCosObjectsSet = new HashSet<Integer>();
				matchedCosObjectsSet.add(objectNums[1]);
				alreadyComparedCosObjectPairsList.put(objectNums[0], matchedCosObjectsSet);
		}
		return true;
	}
	abstract public CosContainerValuesIterator getValuesIterator();
	abstract boolean safeEquals(CosObject value , HashMap<Integer, HashSet<Integer>> alreadyComparedCosObjectPairsList);
}
