/*
 * File: TIFFEngine.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.filter;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Arrays;

import com.adobe.internal.pdftoolkit.core.exceptions.PDFUnsupportedFeatureException;

/*--------- TIFF pre-compression/post-decompression predictors --------
  LZW and Flate use these predictors.
  NOTE: The predictors are all very slow.  Hor. differencing, for example,
  is 100% slower compressing and 600% slower expanding than no predictor.
  Predictors:   0      (no prediction)
  1      DEFAULT (no prediction)
  Component sizes 1, 2, 4, and 8:
  2      hor. differencing (see TIFF 6.0 spec)
  Component size 8 only:
  3..9   unused
  10     PNG code 0: None
  11     PNG code 1: Sub
  12     PNG code 2: Up
  13     PNG code 3: Average
  14     PNG code 4: Paeth
  15     PNG optimum (best efforts)
  ----------------------------------------------------------------------*/

class TIFFEngine
{
	// Fields
	private boolean	encode;
	private TIFFInputStream rdr;
	private OutputStream wtr;
	private boolean	init;
	private long	totalOut;

	private boolean	eofRead;
	private int		predictor;
	private int		samplesPerRow;
	private int		componentsPerSample;
	private int		componentsPerRow;
	private int		bitsPerComponent;
	private int		bytesPerSample;
	private int		rowPredictor;				/* -1 means not yet known */
	private int		nRows;
	private byte	row[][];					/* PNG filter buffers */
	private int		predictedComponent[];		/* for differential predictors */

	/* Fields used during filter operations */
	private int		curSample;
	private int		curComponent;
	private int num_pixels_processed =0;
	private short currComp=0;

	// Constructors
	private TIFFEngine(boolean encode)
	{
		this.encode = encode;
		init = false;
		predictor = 1;
		samplesPerRow = 1;
		componentsPerSample = 1;
		bitsPerComponent = 8;
	}

	public TIFFEngine(InputStream  in, FilterParams p)
	{
		this(false);
		rdr = (TIFFInputStream)in;
		engineInit(p);
	}

	public TIFFEngine(InputStream in)
	{
		this(in, null);
	}

	public TIFFEngine(OutputStream w, FilterParams p)
	{
		this(true);
		wtr = w;
		engineInit(p);
	}

	public TIFFEngine(OutputStream w)
	{
		this(w, null);
	}

	private void engineInit(FilterParams p)
	{
		if (p != null) {
			if (p.containsKey(FilterParams.Predictor_K))
				predictor = ((Integer)p.get(FilterParams.Predictor_K)).intValue();

			if (p.containsKey(FilterParams.Columns_K))
				samplesPerRow = ((Integer)p.get(FilterParams.Columns_K)).intValue();

			if (p.containsKey(FilterParams.Components_K))
				componentsPerSample = ((Integer)p.get(FilterParams.Components_K)).intValue();

			if (p.containsKey(FilterParams.BitsPerComponent_K))
				bitsPerComponent = ((Integer)p.get(FilterParams.BitsPerComponent_K)).intValue();
		}
	}

	// Methods
	private void initTIFFPredictor()
	{
		int	bitsPerSample;

		/* Validate parameters -- attempt to clean up bad params */
		if (samplesPerRow < 1 || componentsPerSample < 1 || bitsPerComponent < 1)
			predictor = 1;
		nRows = 0;

		switch (predictor) {
		case 2:
			if (bitsPerComponent != 1 && bitsPerComponent != 2 && bitsPerComponent != 4 && bitsPerComponent != 8 && bitsPerComponent != 16){
	               throw new PDFUnsupportedFeatureException("Predictor value 2 (Horizontal Differencing) is not supported for" +  bitsPerComponent +  " bis per component");				
			}
			break;
		case 10:
		case 11:
		case 12:
		case 13:
		case 14:
		case 15:
			nRows = 2;
			break;

		default:
			predictor = 1;
		}

		bitsPerSample = componentsPerSample * bitsPerComponent;
		bytesPerSample = (bitsPerSample + 7) / 8;
		int bytesPerRow = ((samplesPerRow * bitsPerSample) + 7) / 8;
		componentsPerRow = bytesPerRow;// PNG predictor works on bytes not on pixels.
		rowPredictor = -1;

		predictedComponent = new int[componentsPerSample];
		for (int j = 0; j < componentsPerSample; j++)
			predictedComponent[j] = 0;

		row = new byte[nRows][];
		for (int i = 0; i < nRows; i++) {
			row[i] = new byte[bytesPerRow + bytesPerSample];
			for (int j = 0; j < bytesPerRow + bytesPerSample; j++)
				row[i][j] = 0;
		}

		curSample = 0;
		curComponent = 0;
		eofRead = false;
		init = true;
	}

	/* Do the prediction for one byte of input or output */
	public int applyTIFFPredictor(int b)
		throws IOException
	{
		int result = 0, pred =0;

		if (!init)
			initTIFFPredictor();

		if (b < 0 && bitsPerComponent!=16)
			eofRead = true;
		if (eofRead)
			return -1;

		switch (predictor) {
		case 1:		/* TIFF predictor 1: Identity */
			result = b & 0x0ff;
			break;

		case 2:		/* TIFF Predictor 2: Horizontal differencing */
			if (bitsPerComponent == 8) {	/* Fast case */
				if ( encode ) {
					int nextPrediction = b & 0x0ff;
					result = (nextPrediction - predictedComponent[curComponent]) & 0x0ff;
					predictedComponent[curComponent] = nextPrediction;
				}
				else {
					result = (b + predictedComponent[curComponent]) & 0x0ff;
					predictedComponent[curComponent] = result;
				}

				if (++curComponent == componentsPerSample) {
					curComponent = 0;
					if (++curSample == samplesPerRow) {
						curSample = 0;
						for (int j = 0; j < componentsPerSample; j++)
							predictedComponent[j] = 0;
					}
				}
			}else if(bitsPerComponent==16){
				// when the whole row has been processed we initialize the past to 0
				if(num_pixels_processed == samplesPerRow){
					num_pixels_processed = 0;
		            currComp =0;
		            Arrays.fill(predictedComponent, 0);
				}
				if(encode){
			        result = (short) (b - predictedComponent[currComp] );
			        predictedComponent[currComp] = b;
				}else{
	                result = (short) (predictedComponent[currComp] + b);
	                predictedComponent[currComp] = result;
				}
				currComp ++;
	            if(currComp == componentsPerSample){
	            	currComp = 0;
	            	num_pixels_processed ++;
	            }
			}else {
				/* Slow but compact.  Rows are byte-aligned,
				   and any leftover bits required for
				   byte-alignment are set to zero (this is different
				   from the "C" implementation).
				*/
				int		mask, baseMask, shift;

				baseMask = (1 << bitsPerComponent) - 1;
				shift = 8 - bitsPerComponent;
				mask = baseMask << shift;
				result = 0;

				while (mask != 0) {		/* Do for all components in a byte */
					if ( encode ) {
						int nextPrediction = (b & mask) >>> shift;
						result += ((nextPrediction - predictedComponent[curComponent]) & baseMask) << shift;
						predictedComponent[curComponent] = nextPrediction;
					}
					else {
						int nextPrediction = (b & mask) >>> shift;
						result = ((nextPrediction + predictedComponent[curComponent]) & baseMask) << shift;
						predictedComponent[curComponent] = nextPrediction;
					}

					if (++curComponent == componentsPerSample) {
						curComponent = 0;
						if (++curSample == samplesPerRow) {
							curSample = 0;
							for (int j = 0; j < componentsPerSample; j++)
								predictedComponent[j] = 0;
							break;	/* out of the while loop, leaving the end of the byte zero */
						}
					}

					mask >>>= bitsPerComponent;
					shift -= bitsPerComponent;
				}
			}
			break;

		default:	/* one of the PNG predictors */
			if (rowPredictor == -1) {	/* Start of row */
				if (encode) {
					rowPredictor = (predictor < 15) ? (predictor - 10)
						: 1 ;	/* "Optimum" is simply "Sub" for now */
					wtr.write(rowPredictor);		/* prefix row with the predictor value */
				}
				else {
					if (b > 4) {
						eofRead = true;
						return -1;		    /* Bad data -- return EOF */
					}
					rowPredictor = b;
					b = rdr.read1();		/* Read the first real data byte */
					if (b < 0) {
						eofRead = true;
						return -1;
					}
				}
			}

			switch (rowPredictor) {
			default:
			case 0: /* None */
				pred = 0;
				break;

			case 1:  /* Sub */
				pred = left() & 0x0ff;
				break
					;
			case 2:  /* Up */
				pred = above() & 0x0ff;
				break;

			case 3:  /* Average */
				pred = ( (left() & 0x0ff)
						 +(above() & 0x0ff) ) >> 1;
				break;

			case 4:  /* Paeth */
				{
					int a = left() & 0x0ff;					/* left */
					int d = above() & 0x0ff;	/* above */
					int c = aboveLeft() & 0x0ff;					/* upper left */
					int p = a + d - c;

					int pa = Math.abs(p - a);
					int pb = Math.abs(p - d);
					int pc = Math.abs(p - c);
					pred = ((pa <= pb) && (pa <= pc)) ? a : ((pb <= pc) ? d : c);
				}
				break;
			}

			if ( encode ) {
				result = (b - pred) & 0x0ff;
				row[0][curComponent] = (byte)b;
			}
			else {
				result = (b + pred) & 0x0ff;
				row[0][curComponent] = (byte)result;
			}

			curComponent++;
			if (curComponent == componentsPerRow) {
				/* Cycle the rows */
				byte t[] = row[nRows-1];
				for (int i = nRows-1; i > 0; i-- )
					row[i] = row[i - 1];
				row[0] = t;				/* We don't have to zero this */
				curComponent = 0;
				rowPredictor = -1;
			}
			break;
		}

		return result;
	}

	/**
	 * Returns left byte of current byte being processed.
	 * @return byte
	 */
	private byte left(){
		if (curComponent >= bytesPerSample)
			return row[0][curComponent-bytesPerSample];
		return 0;
	}
	
	/**
	 * Returns upper byte of current byte being processed.
	 * @return byte
	 */
	private byte above(){
		return row[1][curComponent];
	}
	
	/**
	 * Returns upper left byte of current byte being processed.
	 * @return byte
	 */
	private byte aboveLeft(){
		if (curComponent >= bytesPerSample)
			return row[1][curComponent-bytesPerSample];
		return 0;
	}
	// The way we interface to the engine
	protected void put(int b)
		throws IOException
	{
		int result = applyTIFFPredictor(b);
		if(bitsPerComponent!=16){
			wtr.write(result);
			++totalOut;
		}else{
	        wtr.write((byte) ((result & 0xFF00) >> 8));
	        wtr.write((byte) (result & 0x00FF));
		}
	}

	
	/**
	 * Counts the number of bytes written by this filter.
	 * @return	actual number of bytes written
	 */
	public long getTotalOut()
	{
		return totalOut;
	}
	
	/**
	 * Returns the bits per components in the image.
	 * @return int
	 */
	int getBitsPerComponent(){
		return bitsPerComponent;
	}

}