/* ****************************************************************************
*
*	File: PermissionProviderStandard.java
*
* ****************************************************************************
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2005 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
* ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.permissionprovider;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import com.adobe.internal.pdftoolkit.core.exceptions.PDFUnableToCompleteOperationException;
import com.adobe.internal.util.BitwiseExpression;
import com.adobe.internal.util.BitwiseOperandEvaluator;
import com.adobe.internal.util.InvalidOperandException;
import com.adobe.internal.util.MalformedExpressionException;

/**
 * 
 * This class implements the Encryption PermissionProvider. 
 *         
 * @author speri
 *
*
*/
public class PermissionProviderStandard implements PermissionProvider, BitwiseOperandEvaluator
{
	private int m_Perms = 0;    // Everything is disallowed
	
	// following variables represents the PermissionParams for the Encryption.
	static final String  OPEN = "Open";
	static final String  SECURE = "Secure";
	static final String  PRINT_LOW = "PrintLow";
	static final String  EDIT = "Modfiy/Edit";
	static final String  COPY = "Copy";
	static final String  EDIT_NOTES = "EditNotes";
	static final String  EXT = "Ext";
	static final String  SAVE_AS = "SaveAs";
	static final String  FILL_SIGNIN = "FillAndSign";
	static final String  ACCESSIBLE = "Accessible";
	static final String  DOC_ASSEMBLY = "DocAssembly";
	static final String  HIGH_PRINT = "HighPrint";
	
	private  static final Map m_PermissionData = initPermissionData()  ;
	private  static final Map m_ReversePermissionData = initReversePermissionData()  ;
	private static final Map m_BitFlags = initBitFlags();
//	private static final int CLEAR_BITS = -1 ^ 
//											((1 << ((Integer) m_BitFlags.get(OPEN)).intValue()) |
//											(1 << ((Integer) m_BitFlags.get(SECURE)).intValue()));
	/**
	 * 
    constructor for PermissionProviderStandard. It loads the permissions data that we got 
    from the Acrobat Permissions document. 
    * 
    */
	
	public PermissionProviderStandard(int perms)
	{
		m_Perms = perms;
	}
	
	/**
	 * Constructs a new instance of {@link PermissionProviderStandard} with the same permissions
	 * allowed in passed provider here.
	 */
	public static PermissionProviderStandard newInstance(PermissionProvider provider)
	throws PDFUnableToCompleteOperationException
	{
		if (provider == null)
		{
			return null;
		}
		if (provider instanceof PermissionProviderStandard)
		{
			return new PermissionProviderStandard(((PermissionProviderStandard) provider).getPermissionBits());
		}
		int perms = 0;
		Iterator operationsIter = m_ReversePermissionData.keySet().iterator();
		while (operationsIter.hasNext())
		{
			ObjectOperations operation = (ObjectOperations) operationsIter.next();
			if (provider.isPermitted(operation))
			{
				String permission = (String)m_ReversePermissionData.get(operation);
				Integer bitFlag = (Integer) m_BitFlags.get(permission);
				perms |= 1 << bitFlag.intValue();
			}
		}
		return new PermissionProviderStandard(perms);
	}
	
	/**
	 * This method returns an integer whose bits represent different permissions allowed
	 * or disallowed.
	 */
	public int getPermissionBits()
	{
		return m_Perms;
	}

	private static Map initPermissionData()
	{
		Map permissionData = new HashMap();
		permissionData.put(ObjectOperations.DOC_MODIFY , EDIT ); 
		permissionData.put(ObjectOperations.DOC_COPY , COPY ); 
		permissionData.put(ObjectOperations. DOC_ACCESSIBLE , COPY +"|"+ACCESSIBLE ); 
		permissionData.put(ObjectOperations.DOC_OPEN , OPEN ); 
		permissionData.put(ObjectOperations.DOC_SECURE , SECURE ); 
		permissionData.put(ObjectOperations.DOC_PRINT_HIGH , HIGH_PRINT ); 
		permissionData.put(ObjectOperations.DOC_PRINT_LOW , PRINT_LOW ); 
		permissionData.put(ObjectOperations.DOC_FULLSAVE , EDIT +"|" + EDIT_NOTES +"|"+ FILL_SIGNIN +"|"+ DOC_ASSEMBLY); 
		permissionData.put(ObjectOperations.DOC_IMPORT , EDIT ); 
		permissionData.put(ObjectOperations.DOC_EXPORT , COPY ); 

		permissionData.put(ObjectOperations.PAGE_CREATE , EDIT ); 
		permissionData.put(ObjectOperations.PAGE_DELETE , EDIT +"|"+DOC_ASSEMBLY ); 
		permissionData.put(ObjectOperations.PAGE_MODIFY , EDIT); 
		permissionData.put(ObjectOperations.PAGE_COPY ,EDIT ); 
		permissionData.put(ObjectOperations.PAGE_ROTATE ,EDIT +"|"+DOC_ASSEMBLY );
		permissionData.put(ObjectOperations.PAGE_CROP ,EDIT );
		permissionData.put(ObjectOperations.PAGE_INSERT ,EDIT +"|"+DOC_ASSEMBLY );
		permissionData.put(ObjectOperations.PAGE_REPLACE ,EDIT);
		permissionData.put(ObjectOperations.PAGE_REORDER ,EDIT );
		permissionData.put(ObjectOperations.PAGE_EXPORT ,SECURE);
		
		permissionData.put(ObjectOperations.LINK_CREATE ,EDIT );
		permissionData.put(ObjectOperations.LINK_DELETE,EDIT );
		permissionData.put(ObjectOperations.LINK_MODIFY ,EDIT );
		permissionData.put(ObjectOperations.LINK_IMPORT ,EDIT );
		permissionData.put(ObjectOperations.LINK_EXPORT ,EDIT +"|" + COPY);
		
		permissionData.put(ObjectOperations.BOOKMARK_CREATE ,EDIT +"|"+DOC_ASSEMBLY );
		permissionData.put(ObjectOperations.BOOKMARK_DELETE ,EDIT +"|"+DOC_ASSEMBLY  );
		permissionData.put(ObjectOperations.BOOKMARK_MODIFY ,EDIT +"|"+DOC_ASSEMBLY );
		permissionData.put(ObjectOperations.BOOKMARK_EXPORT ,EDIT +"|"+DOC_ASSEMBLY );
		
		permissionData.put(ObjectOperations.THUMBNAIL_CREATE ,EDIT +"|"+DOC_ASSEMBLY);
		permissionData.put(ObjectOperations.THUMBNAIL_DELETE ,EDIT +"|"+DOC_ASSEMBLY);
		permissionData.put(ObjectOperations.THUMBNAIL_EXPORT ,EDIT +"|"+DOC_ASSEMBLY);
		
		permissionData.put(ObjectOperations.ANNOT_CREATE ,EDIT_NOTES );
		permissionData.put(ObjectOperations.ANNOT_DELETE ,EDIT_NOTES );
		permissionData.put(ObjectOperations.ANNOT_MODIFY ,EDIT_NOTES );
		permissionData.put(ObjectOperations.ANNOT_COPY ,EDIT_NOTES );
		permissionData.put(ObjectOperations.ANNOT_SUMMARIZE ,EDIT_NOTES );
		permissionData.put(ObjectOperations.ANNOT_IMPORT ,EDIT_NOTES );
		permissionData.put(ObjectOperations.ANNOT_EXPORT ,COPY +"|"+EDIT_NOTES );
		    
		permissionData.put(ObjectOperations.FORM_CREATE ,EDIT +"&" + EDIT_NOTES);
		permissionData.put(ObjectOperations.FORM_DELETE ,EDIT +"&" + EDIT_NOTES );
		permissionData.put(ObjectOperations.FORM_MODIFY ,EDIT +"&" + EDIT_NOTES );
		permissionData.put(ObjectOperations.FORM_FILLIN ,EDIT +"|" + EDIT_NOTES +"|" + FILL_SIGNIN);
		permissionData.put(ObjectOperations.FORM_IMPORT ,EDIT +"|" + EDIT_NOTES +"|" + FILL_SIGNIN);
		permissionData.put(ObjectOperations.FORM_EXPORT ,COPY +"| (" + EDIT +"&" + EDIT_NOTES + ")" );
		permissionData.put(ObjectOperations.FORM_SPAWN_TEMPLATE ,EDIT +"|"+FILL_SIGNIN );
	
		permissionData.put(ObjectOperations.SIGNATURE_CREATE ,EDIT +"&" + EDIT_NOTES );
		permissionData.put(ObjectOperations.SIGNATURE_DELETE ,EDIT +"&" + EDIT_NOTES );
		permissionData.put(ObjectOperations.SIGNATURE_MODIFY ,EDIT +"&" + EDIT_NOTES);
		permissionData.put(ObjectOperations.SIGNATURE_FILLIN ,EDIT +"|" + EDIT_NOTES +"|" + FILL_SIGNIN );

		// ref to table4 in acrobat permissions doc , note b and c . 
	    permissionData.put(ObjectOperations.EF_CREATE, EDIT +"|" + EDIT_NOTES );
		permissionData.put(ObjectOperations.EF_DELETE ,EDIT +"|" + EDIT_NOTES );
		permissionData.put(ObjectOperations.EF_MODIFY ,EDIT +"|" + EDIT_NOTES +"|" + FILL_SIGNIN);
		permissionData.put(ObjectOperations.EF_IMPORT ,EDIT +"|" + EDIT_NOTES );
		permissionData.put(ObjectOperations.EF_EXPORT ,COPY );

		return permissionData;
	}

	private static Map initReversePermissionData()
	{
		Map permissionData = new HashMap();
		permissionData.put(ObjectOperations.DOC_MODIFY , EDIT ); 
		permissionData.put(ObjectOperations.DOC_COPY , COPY ); 
		permissionData.put(ObjectOperations.DOC_PRINT_HIGH , HIGH_PRINT ); 
		permissionData.put(ObjectOperations.DOC_PRINT_LOW , PRINT_LOW ); 
		permissionData.put(ObjectOperations.DOC_IMPORT , EDIT ); 
		permissionData.put(ObjectOperations.DOC_EXPORT , COPY ); 
		permissionData.put(ObjectOperations. DOC_ACCESSIBLE , ACCESSIBLE ); 

		permissionData.put(ObjectOperations.PAGE_CREATE , EDIT ); 
		permissionData.put(ObjectOperations.PAGE_MODIFY , EDIT); 
		permissionData.put(ObjectOperations.PAGE_COPY ,EDIT ); 
		permissionData.put(ObjectOperations.PAGE_CROP ,EDIT );
		permissionData.put(ObjectOperations.PAGE_REPLACE ,EDIT);
		permissionData.put(ObjectOperations.PAGE_REORDER ,EDIT );
		permissionData.put(ObjectOperations.PAGE_DELETE , DOC_ASSEMBLY ); 
		permissionData.put(ObjectOperations.PAGE_ROTATE ,DOC_ASSEMBLY );
		permissionData.put(ObjectOperations.PAGE_INSERT ,DOC_ASSEMBLY );
		
		permissionData.put(ObjectOperations.LINK_CREATE ,EDIT );
		permissionData.put(ObjectOperations.LINK_DELETE,EDIT );
		permissionData.put(ObjectOperations.LINK_MODIFY ,EDIT );
		permissionData.put(ObjectOperations.LINK_IMPORT ,EDIT );
				
		
		permissionData.put(ObjectOperations.ANNOT_CREATE ,EDIT_NOTES );
		permissionData.put(ObjectOperations.ANNOT_DELETE ,EDIT_NOTES );
		permissionData.put(ObjectOperations.ANNOT_MODIFY ,EDIT_NOTES );
		permissionData.put(ObjectOperations.ANNOT_COPY ,EDIT_NOTES );
		permissionData.put(ObjectOperations.ANNOT_SUMMARIZE ,EDIT_NOTES );
		permissionData.put(ObjectOperations.ANNOT_IMPORT ,EDIT_NOTES );
		    
		permissionData.put(ObjectOperations.BOOKMARK_CREATE ,DOC_ASSEMBLY );
		permissionData.put(ObjectOperations.BOOKMARK_DELETE ,DOC_ASSEMBLY  );
		permissionData.put(ObjectOperations.BOOKMARK_MODIFY ,DOC_ASSEMBLY );
		permissionData.put(ObjectOperations.BOOKMARK_EXPORT ,DOC_ASSEMBLY );
		
		permissionData.put(ObjectOperations.THUMBNAIL_CREATE ,DOC_ASSEMBLY);
		permissionData.put(ObjectOperations.THUMBNAIL_DELETE ,DOC_ASSEMBLY);
		permissionData.put(ObjectOperations.THUMBNAIL_EXPORT ,DOC_ASSEMBLY);

		permissionData.put(ObjectOperations.FORM_FILLIN ,FILL_SIGNIN);
		permissionData.put(ObjectOperations.FORM_IMPORT ,FILL_SIGNIN);
		permissionData.put(ObjectOperations.FORM_SPAWN_TEMPLATE ,FILL_SIGNIN );

		permissionData.put(ObjectOperations.SIGNATURE_FILLIN ,FILL_SIGNIN );

		return permissionData;
	}

	private static Map initBitFlags()
	{
		Map bitFlags = new HashMap();
		bitFlags.put(OPEN, Integer.valueOf(0));
		bitFlags.put(SECURE, Integer.valueOf(1));
		bitFlags.put(PRINT_LOW, Integer.valueOf(2));
		bitFlags.put(EDIT, Integer.valueOf(3));
		bitFlags.put(COPY, Integer.valueOf(4));
		bitFlags.put(EDIT_NOTES, Integer.valueOf(5));
		bitFlags.put(FILL_SIGNIN, Integer.valueOf(8));
		bitFlags.put(ACCESSIBLE, Integer.valueOf(9));
		bitFlags.put(DOC_ASSEMBLY, Integer.valueOf(10));
		bitFlags.put(HIGH_PRINT, Integer.valueOf(11));
		return bitFlags;
	}
	
	/**	Determines whether the given object operation combination is permitted on a given document. 
	  @return	boolean

		Only the valid object operations are allowed as input paramater.
	*/
	public boolean isPermitted(ObjectOperations objOpr)
	{	
		String permValue =  (String) m_PermissionData.get(objOpr);
		
		if(permValue == null ) return true;
			
		BitwiseExpression expObj = new BitwiseExpression(this);
		boolean bRet = true;
		try
		{
			bRet = expObj.evaluateExpression(permValue);
		}
		catch (MalformedExpressionException e) 
		{
			throw new RuntimeException("Gibson Internal error",e);
		}
		catch (InvalidOperandException e)
		{
			throw new RuntimeException("Gibson Internal error",e);
		}
		return bRet;
	}

	/** 
	 * This is the callback from the BitwiseExpressionEvaluator which is called for each 
	 * Pemission Param.
	 * 
	 * @see com.adobe.internal.util.BitwiseOperandEvaluator#evaluate(java.lang.String)
	 * 
	 */
	public boolean evaluate(String operandVar) throws InvalidOperandException 
	{
		Integer bitVal = (Integer) m_BitFlags.get(operandVar);
		if (bitVal == null)
		{
			throw new InvalidOperandException(operandVar + " is not valid Encryption permission paramter");
		}
		return (m_Perms & (1 << bitVal.intValue())) != 0;
/*		
		if(operandVar.equals(OPEN))
			return (m_Perms & 1) != 0; // 1<<0
		else if(operandVar.equals(SECURE))
			return (m_Perms & 2) != 0; // 1<<1
		else if(operandVar.equals(PRINT_LOW))
			return (m_Perms & 4) != 0; // 1<<2
		else if(operandVar.equals(EDIT))
			return (m_Perms & 8) != 0; // 1<<3
		else if(operandVar.equals(COPY))
			return (m_Perms & 16) != 0; // 1<<4
		else if(operandVar.equals(EDIT_NOTES))
			return (m_Perms & 32) != 0; // 1<<5
		else if(operandVar.equals(SAVE_AS))
			return (m_Perms & 64) != 0; // 1<< 6
		else if(operandVar.equals(FILL_SIGNIN))
			return (m_Perms & 256) != 0; // 1<<8
		else if(operandVar.equals(ACCESSIBLE))
			return (m_Perms & 512) != 0; // 1<<9
		else if(operandVar.equals(DOC_ASSEMBLY))
			return (m_Perms & 1024) != 0; // 1<<10
		else if(operandVar.equals(HIGH_PRINT))
			return (m_Perms & 2048) != 0; // 1<<11
		else
			throw new InvalidOperandException(operandVar + " is not valid Encryption permission paramter");
*/
	}
}
