/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa;

import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.MsgFormatPos;
import com.adobe.xfa.ut.ResId;
import com.adobe.xfa.ut.StringUtils;

/**
 * A convenience class that returns the frequently-used EnumAttr
 * types BOOL_TRUE and BOOL_FALSE. This will save applications
 * from needing to use "new" to reference these values.
 *
 * @exclude from published api -- Mike Tardif, May 2006.
 */

public final class Bool {

	/**
	 * Static function for returning Off/false/0
	 */
	public static EnumAttr falseValue() {
		return EnumAttr.getEnum(EnumAttr.BOOL_FALSE);
	}


	/**
	 * Construct an instance from a boolean
	 * @param bValue the input boolean
	 */
	public static EnumAttr getBool(boolean bValue) {
		if (bValue)
			return trueValue();
		else
			return falseValue();
	}


	/**
	 * Get the boolean instance from a string<br>
	 * @param sValue "1" for true, "0" or "" for false
	 * @return The EnumAttr attribute representing this boolean value.
	 * @exception InvalidPropertyValueException
	 */
	public static EnumAttr getBool(String sValue) {
		if (StringUtils.isEmpty(sValue) || sValue.equals("0"))
			return falseValue();
		else if (sValue.equals("1"))
			return trueValue();
		else {
			MsgFormatPos msg
				= new MsgFormatPos(ResId.InvalidPropertyValueException, sValue);
			msg.format(XFA.BOOLEAN);
			throw new ExFull(msg);
		}
	}


	/**
	 * Return the boolean value of an enumerated value
	 * @param e the enumerated value
	 * @return true or false
	 */
	public static boolean getValue(EnumAttr e) {
		return e.getInt() == EnumAttr.BOOL_TRUE;
	}


	/**
	 * Return the boolean value of an enumerated attribute
	 * @param e the enumerated attribute
	 * @return true or false
	 */
	public static boolean getValue(EnumValue e) {
		return e.getInt() == EnumAttr.BOOL_TRUE;
	}


	/**
	 * Convenience method to return a string for a boolean value.
	 * @param b the boolean value
	 * @return the String representation
	 */
	public static String toString(boolean b) {
		return b ? "1" : "0";
	}


	/**
	 * Static function for returning On/true/1
	 */
	public static EnumAttr trueValue() {
		return EnumAttr.getEnum(EnumAttr.BOOL_TRUE);
	}

}
