/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */

package com.adobe.xfa.ut.trace;

import com.adobe.xfa.ut.MsgFormatPos;

/**
 * This auto registering class is used to define trace areas and issue messages.
 * The implementer defines an instances of the jfTrace class in the file scope
 * to ensure all trace areas are available before run-time. These instances then
 * are registered with the jfTraceHandler during its construction
 * 
 * <pre>
 *  
 *  example:
 * 
 * 		Trace oDataTrace = new Trace(&quot;data&quot;, ResId.DataTraceHelp);
 * 		Trace oStringTrace new Trace(&quot;string&quot; ResId.StringTraceHelp);
 * 		main(...)
 * 		{
 * 			TraceHandler oTraceHandler = new TraceHandler();
 * 
 * 			TraceHandler.registerTraceHandler(oTraceHandler);
 * 
 * 			// two ways to activate 
 * 			oTraceHandler.activate(&quot;data&quot;, 2);
 * 			oStringTrace.activate(3);
 * 
 * 			...
 * 			oTraceHandler.trace(oDataTrace, 2, 27, &quot;data msg&quot;); // issue message
 * 			...
 * 			oDataTrace.trace(2, 33, &quot;string msg&quot;); // issue message
 * 			...
 * 		}
 * 
 *  
 * </pre>
 *
 * @exclude from published api -- Mike Tardif, May 2006.
 */

public final class Trace {
	
	private final String msName;	

	private volatile boolean mbEnabled;	
	private volatile int mnHelpId;	
	private volatile boolean mbIsPlaceHolder;
	

	// This constructor doesn't add this trace to the GlobalTraceStore.
	// It is a private constructor that only GlobalTraceStore itself uses.
	Trace(String sName) {
		msName = sName;
		mbIsPlaceHolder = true;
	}

	/**
	 * Constructor
	 * Registers the trace with a global storage area.
	 * @param sName the name of the trace section for which the message will be issued
	 * @param nHelpId the message Id associated with a one-line help message for this trace section.
	 */
	public Trace(String sName, int nHelpId) {
		msName = sName;
		mnHelpId = nHelpId;
		
		GlobalTraceStore.getStore().addTrace(this);
	}

	
	/**
	 * Check if this trace section is enabled on the current thread
	 * @param sName - the name of the trace section for which the message will be issued
	 * @param nLevel - the detail level of the messages allowed,
	 * @return true if this trace section is enabled on the current thread, else false
	 */
	public static boolean isEnabled(String sName, int nLevel) {
		final GlobalTraceStore globalTraceStore = GlobalTraceStore.getStore();
		if (globalTraceStore.isEnabled()) {
			return TraceHandler.getTraceHandler().isActive(sName, nLevel);
		}
		
		return false;
	}
	/**
	 * Issues a trace message on the current thread.
	 * 
	 * @param sName the name of the trace section for which the message will be issued
	 * @param nLevel  the detail level of the message, 1 for high level information,
	 *            3 for detailed information.
	 * @param nId the message Id.
	 * @param sMsg the message text.
	 */
	public static void trace(String sName, int nLevel, int nId, String sMsg) {
		
		final GlobalTraceStore globalTraceStore = GlobalTraceStore.getStore();
		if (globalTraceStore.isEnabled()) {
			
			TraceHandler.getTraceHandler().trace(sName, nLevel, nId, sMsg);
		}
	}
	/**
	 * Issues a trace message.
	 * @param sName the name of the trace section for which the message will be issued
	 * @param nLevel the detail level of the message, 1 for high level information, 3 for detailed information.
	 * @param oMsg the message for the trace.
	 */
	public static void trace(String sName, int nLevel, MsgFormatPos oMsg) {
		final GlobalTraceStore globalTraceStore = GlobalTraceStore.getStore();
		if (globalTraceStore.isEnabled()) {
			TraceHandler.getTraceHandler().trace(sName, nLevel, oMsg);
		}
	}

	
	/**
	 * Activates this trace section for the current thread
	 * @param nLevel the detail level of the messages allowed, 1 for high level information, 3 for detailed information.
	 */
	public void activate(int nLevel) {
		TraceHandler.getTraceHandler().activate(this, nLevel);
	}

	/**
	 * Deactivates this trace section for the current thread
	 */
	public void deactivate() {
		if (mbEnabled) {
			TraceHandler.getTraceHandler().deactivate(msName);
		}
	}

	/**
	 * set that this trace section is enabled on some thread
	 */
	protected void enable() {
		mbEnabled = true;
	}
	
	public int getHelpId() {
		return mnHelpId;
	}
	
	void setHelpId(int nHelpId) {
		mnHelpId = nHelpId;
	}
	
	boolean isPlaceHolder() {
		return mbIsPlaceHolder;
	}
	
	void setPlaceHolder(boolean bIsPlaceHolder) {
		mbIsPlaceHolder = bIsPlaceHolder;
	}

	/**
	 * Returns the name of this trace section
	 * 
	 * @return the name of this trace section
	 */
	public String getName() {
		return msName;
	}
	
	/**
	 * Check if this trace section is enabled on the current thread.
	 * 
	 * @param nLevel the detail level of the messages allowed,
	 * @return true if this trace section is enabled on the current thread, else false
	 */
	public boolean isEnabled(int nLevel) {
		if (mbEnabled) {
			return TraceHandler.getTraceHandler().isActive(msName, nLevel);
		}
		
		return false;
	}

	/**
	 * Issues a trace message.
	 * @param nLevel the detail level of the message, 1 for high level information, 3 for detailed information.
	 * @param nId the message Id.
	 * @param sMsg the message text.
	 */
	public void trace(int nLevel, int nId, String sMsg) {
		if (mbEnabled) {
			TraceHandler.getTraceHandler().trace(msName, nLevel, nId, sMsg);
		}
	}

	/**
	 * Issues a trace message.
	 * @param nLevel the detail level of the message, 1 for high level information, 3 for detailed information.
	 * @param oMsg the message for the trace.
	 */
	public void trace(int nLevel, MsgFormatPos oMsg) {
		if (mbEnabled) {
			TraceHandler.getTraceHandler().trace(msName, nLevel, oMsg);
		}
	}
}