/*
 * Copyright 1997-2010 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.commons;

import com.adobe.granite.toggle.api.ToggleRouter;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleContext;
import org.osgi.framework.FrameworkUtil;
import org.osgi.framework.ServiceReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import java.util.Locale;

/**
 * Implements a wrapper around a locale that supports different codes for
 * country and language. This is mostly used as workaround for language
 * trees that don't follow the same format for locales in java.util.Locale.
 */
public class Language {
    /**
     * default logger
     */
    private static final Logger log = LoggerFactory.getLogger(Language.class);

    private final Locale locale;

    private final String language;

    private final String country;

    private String script = "";
    
    private static ToggleRouter toggleRouter;

    public Language(Locale locale) {
        if (toggleRouter != null && !toggleRouter.isEnabled(LanguageUtil.SCRIPT_TOGGLE)) {
            this.locale = locale;
            language = locale.getLanguage();
            country = locale.getCountry();
        } else {
            this.locale = locale;
            language = locale.getLanguage();
            country = locale.getCountry();
            script = locale.getScript();
        }
    }
    
    static {
        Bundle bundle = FrameworkUtil.getBundle(Language.class);
        if (bundle != null) {
            BundleContext bundleContext = bundle.getBundleContext();
            if (bundleContext != null) {
                ServiceReference serviceReference = bundleContext.getServiceReference(ToggleRouter.class.getName());
                if (serviceReference != null) {
                    toggleRouter = (ToggleRouter) bundleContext.getService(serviceReference);
                }
            }
        }
    }

    public Language(String code) {
        // support both "-" and "_" as delimiters
        code = code.replaceAll("-", "_");

        int idx = code.indexOf('_');
        if (toggleRouter != null && !toggleRouter.isEnabled(LanguageUtil.SCRIPT_TOGGLE)) {
            if (idx < 0) {
                language = code;
                country = "";
                locale = new Locale(language);
            } else {
                language = code.substring(0, idx);
                country = code.substring(idx + 1);
                locale = new Locale(language, country);
            }
        } else {
            if (idx < 0) {
                this.language = code;
                this.country = "";
                this.script = "";
                this.locale = new Locale(language);
            } else {
                this.language = code.substring(0, idx);
                String secondHalf = code.substring(idx + 1);
                idx = secondHalf.indexOf('_');
                String tempScript = "";
                String tempCountry = "";
                Locale tempLocale = null;
                if (idx < 0) {
                    if (LanguageUtil.isScriptTag(secondHalf) && LanguageUtil.isLanguageCode(this.language)) {
                        tempScript = secondHalf;
                        try {
                            tempLocale = new Locale.Builder().setLanguage(this.language).setScript(secondHalf).build();
                        } catch (IllegalArgumentException e) { // fallback to old implementation
                            log.debug("Failed to create Locale with code: {}", code, e);
                            tempScript = "";
                            tempCountry = secondHalf;
                            tempLocale = new Locale(this.language, tempCountry);
                        }
                    } else {
                        tempCountry = secondHalf;
                        tempLocale = new Locale(this.language, tempCountry);
                    }
                    this.script = tempScript;
                    this.country = tempCountry;
                    this.locale = tempLocale;
                } else {
                    String middlePart = secondHalf.substring(0, idx);
                    String lastPart = secondHalf.substring(idx + 1);
                    if (LanguageUtil.isScriptTag(middlePart) && LanguageUtil.isLanguageCode(this.language) && LanguageUtil.isCountryCode(lastPart)) {
                        tempScript = middlePart;
                        tempCountry = lastPart;
                        try {
                            tempLocale = new Locale.Builder().setLanguage(this.language).setRegion(tempCountry).setScript(tempScript).build();
                        } catch (IllegalArgumentException e) { // fallback to old implementation
                            log.debug("Failed to create Locale with code: {}", code, e);
                            tempScript = "";
                            tempCountry = secondHalf;
                            tempLocale = new Locale(this.language, tempCountry);
                        }
                    } else {
                        tempCountry = secondHalf;
                        tempLocale = new Locale(this.language, tempCountry);
                    }
                    this.script = tempScript;
                    this.country = tempCountry;
                    this.locale = tempLocale;
                }
            }
        }
    }

    public Language(Locale locale, String language, String country) {
        if (toggleRouter != null && !toggleRouter.isEnabled(LanguageUtil.SCRIPT_TOGGLE)) {
            this.locale = locale;
            this.language = language;
            this.country = country;
        } else {
            this.locale = locale;
            this.language = language;
            this.country = country;
            this.script = "";
        }
    }

    public Language(String language, String country) {
        if (toggleRouter != null && !toggleRouter.isEnabled(LanguageUtil.SCRIPT_TOGGLE)) {
            this.language = language;
            this.country = country;
            this.locale = new Locale(language, country);
        } else {
            this.language = language;
            this.country = country;
            this.script = "";
            this.locale = new Locale(language, country);
        }
    }

    public Locale getLocale() {
        return locale;
    }

    public String getLanguageCode() {
        return language;
    }

    public String getCountryCode() {
        return country;
    }
    String getScriptCode() {
        return script;
    }

    public String toString() {
        final StringBuilder sb = new StringBuilder(language);
        if(script != null && script.length() > 0) {
            sb.append("_").append(this.script);
        }
        if (country.length() > 0) {
            sb.append("_").append(country);
        }
        return sb.toString();
    }

    /**
     * Parses a language code such as "de", "de_CH", "zh_Hans", "zh_Hans_CN", "de_DE_EURO" or "de-CH" and
     * returns the proper {@link Locale} object.
     * @param code language code
     * @return locale object
     */
    public static Locale getLocale(String code) {
        if (code == null) {
            return null;
        }
        // support both "-" and "_" as delimiters
        code = code.replaceAll("-", "_");
    
        // check for existence of country (de_CH)
        int idx = code.indexOf('_');
        if (toggleRouter != null && !toggleRouter.isEnabled(LanguageUtil.SCRIPT_TOGGLE)) {
            if (idx < 0) {
                return new Locale(code);
            } else {
                String language = code.substring(0, idx);
                String country = code.substring(idx + 1);
                // check for existence of variant (de_DE_EURO)
                idx = country.indexOf('_');
                if (idx < 0) {
                    return new Locale(language, country);
                } else {
                    return new Locale(language, country.substring(0, idx), country.substring(idx + 1));
                }
            }
        } else {
            if (idx < 0) {
                return new Locale(code);
            } else {
                String language = code.substring(0, idx);
                String secondHalf = code.substring(idx + 1);
                String country = "";
                String script = "";
                String variant = "";
                idx = secondHalf.indexOf('_');
                if (idx < 0) {
                    if (LanguageUtil.isScriptTag(secondHalf) && LanguageUtil.isLanguageCode(language)) {
                        script = secondHalf;
                        try {
                            return new Locale.Builder().setLanguage(language).setRegion(country).setVariant(variant).setScript(script).build();
                        } catch (IllegalArgumentException e) { // fallback to old implementation
                            log.debug("Failed to create Locale with code: {}", code, e);
                        }
                    }
                    country = secondHalf;
                    return new Locale(language, country);
                } else {
                    String middlePart = secondHalf.substring(0, idx);
                    String lastPart = secondHalf.substring(idx + 1);
                    idx = lastPart.indexOf('_');
                    if (idx < 0) {
                        if (LanguageUtil.isScriptTag(middlePart) && LanguageUtil.isLanguageCode(language) && LanguageUtil.isCountryCode(lastPart)) {
                            script = middlePart;
                            country = lastPart;
                            try {
                                return new Locale.Builder().setLanguage(language).setRegion(country).setVariant(variant).setScript(script).build();
                            } catch (IllegalArgumentException e) { // fallback to old implementation
                                log.debug("Failed to create Locale with code: {}", code, e);
                            }
                        }
                        country = middlePart;
                        variant = lastPart;
                        return new Locale(language, country, variant);
                    } else {
                        script = middlePart;
                        country = lastPart.substring(0, idx);
                        variant = lastPart.substring(idx + 1);
                        if (LanguageUtil.isScriptTag(middlePart) && LanguageUtil.isLanguageCode(language) && LanguageUtil.isCountryCode(country)) {
                            try {
                                return new Locale.Builder().setLanguage(language).setRegion(country).setVariant(variant).setScript(script).build();
                            } catch (IllegalArgumentException e) { // fallback to old implementation
                                log.debug("Failed to create Locale with code: {}", code, e);
                            }
                        }
                        country = middlePart;
                        variant = lastPart;
                        return new Locale(language, country, variant);
                    }
                }
            }
        }
    }
}