/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.personalization;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * The {@code Segment} class is a server-side representation of a segment which may be assigned to a user.
 * 
 * <p>
 * Since AEM segments are resolved client-side, this class makes a best-effort approach to represent the segments on the
 * server side, but some precision may be lost. For instance, script segments will not have their scripts loaded by this
 * segment, but will rather be referenced as {@link Kind#ClientOnly} segments.
 * </p>
 * 
 * <p>
 * This class is intended to be used, but not extended by consumers.
 * </p>
 * 
 */
public class Segment {

    public static final String NAME_PERCENTILE = "percentile";

    public enum Kind {

        /**
         * Composite segment kind which holds references to child segments in the {@code children} property.
         * 
         * <p>
         * The top-level segment is evaluated to true only if all child segments evaluate to true.
         * </p>
         */
        And,

        /**
         * Composite segment kind which holds references to child segments in the {@code children} property.
         * 
         * <p>
         * The top-level segment is evaluated to true only if any of its child segments evaluate to true.
         * </p>
         */
        Or,

        /**
         * Segment kind which should be directly evaluated based on its name/operator/value combination.
         */
        Direct,

        /**
         * Segment kind which is a reference to a segment defined in an external system
         */
        ExternalReference,

        /**
         * Segment kind which signals that the segment could not be properly represented on the server side.
         * 
         * <p>
         * Instead of being defined by its name/operator/value combination, this segment contains its JCR repository
         * path in the {@code value} property. Typical example of this kind of segment is a segment which contains a
         * script trait.
         * </p>
         */
        ClientOnly
    }
	
    public static Segment newLogicSegment(Kind kind) {

        if (kind == Kind.Direct)
            throw new IllegalArgumentException("Segment kind " + kind + " is invalid for a logic segment.");

        return new Segment(null, null, Collections.<String> emptyList(), kind);
    }

    public static Segment newDirectSegment(String name, String operator, String value) {

        return new Segment(name, operator, Collections.singletonList(value), Kind.Direct);
    }

    public static Segment newDirectSegment(String name, String operator, List<String> values) {

        return new Segment(name, operator, values, Kind.Direct);
    }

    public static Segment newExternalReferenceSegment(String name, String operator, String externalId) {
        return new Segment(name, operator, Collections.singletonList(externalId), Kind.ExternalReference);
    }

    /**
     * Creates a new {@code ClientOnly} segment instance
     * 
     * @param path the repository location of the segment
     * @return a new segment instance
     */
    public static Segment newByPathSegment(String path) {
        return new Segment(null, null, Collections.singletonList(path), Kind.ClientOnly);
    }

	private final String name;
    private final String operator;
    private final List<String> values;
	private final Kind kind;
	
	private final List<Segment> children = new ArrayList<Segment>();
	
    private Segment(String name, String operator, List<String> values, Kind kind) {

        this.name = name;
        this.operator = operator;
        this.values = values;
        this.kind = kind;
    }

	public Kind getKind() {

		return kind;
	}
	
	public String getName() {

		return name;
	}
	
	public String getOperator() {

		return operator;
	}
	
    public List<String> getValue() {

        return values;
	}
	
	public List<Segment> getChildren() {
		
		return Collections.unmodifiableList(children);
	}

	public void addChild(Segment resolvedSegment) {
		
		children.add(resolvedSegment);
	}

	@Override
	public String toString() {
		
		return "[" + Segment.class.getSimpleName() + "# name : " + name +", operator: " + operator +", " +
		        "values: " + values + " kind: " + kind + ", children.size(): " + children.size() + "]";
	}

}
