/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.search;

import java.io.IOException;

import javax.jcr.RepositoryException;
import javax.jcr.Session;

import com.day.cq.search.eval.PredicateEvaluator;
import com.day.cq.search.facets.Facet;

/**
 * <code>QueryBuilder</code> is a service for building {@link Query Queries}
 * searching the Java Content Repository and which are easily extensible. The
 * individual constraints of a {@link Query} are expressed as {@link PredicateEvaluator
 * Predicates}, which are pluggable through OSGi mechanisms, ie. one can easily
 * define a new {@link PredicateEvaluator} that maps some higher-level search constraint
 * to an explicit JCR XPath expression, can filter the search result for
 * constraints that cannot be expressed via XPath and automatically provide
 * {@link Facet Facets} based on the search results.
 * 
 * <p>
 * This service allows to create queries with several convenience methods, for
 * example directly from a request using a fixed convention for the structure of
 * request parameters that make up the predicates. In addition, it allows to
 * store queries in the repository and load them again.
 * 
 * <h3>Example usages</h3>
 * 
 * <i>All examples assume that <code>queryBuilder</code> is available as an OSGi
 * SCR reference.</i>
 * 
 * <p>
 * <dl>
 * <dt>From HTTP Form Post Request:</dt>
 * <dd><pre>
 * Session session = request.getResourceResolver().adaptTo(Session.class);
 * Query query = queryBuilder.createQuery(PredicateGroup.create(request.getParameterMap()), session);
 * SearchResult result = query.getResult();
 * ...
 * </pre>
 * </dd>
 * <dt>From key/value map:</dt>
 * <dd><pre>
 * Map<String, String> map = new HashMap<String, String>();
 * map.put("path", "/content");
 * map.put("type", "nt:file");
 * Query query = builder.createQuery(PredicateGroup.create(map), session);
 * SearchResult result = query.getResult();
 * ...
 * </pre>
 * </dd>
 * <dt>From predicates:</dt>
 * <dd><pre>
 * PredicateGroup group = new PredicateGroup();
 * group.add(new Predicate("mypath", "path").set("path", "/content"));
 * group.add(new Predicate("mytype", "type").set("type", "nt:file"));
 * Query query = builder.createQuery(group, session);
 * SearchResult result = query.getResult();
 * ...
 * </pre>
 * </dd>
 * </dl>
 * 
 * @since 5.2
 */
public interface QueryBuilder {

    /**
     * Creates an "empty" query, ie. a query without any constraints. Use
     * {@link Query#getPredicates()} to get the root predicate group in order to
     * add predicates via {@link PredicateGroup#add(Predicate)}.
     * 
     * @param session
     *            the JCR session for the query
     * @return a query without any predicates
     */
    Query createQuery(Session session);

    /**
     * Creates a query with the given predicate group as root group.
     * 
     * @param predicates
     *            a predicate group used as the root group
     * @param session
     *            the JCR session for the query
     * @return a query to execute
     */
    Query createQuery(PredicateGroup predicates, Session session);

    /**
     * Loads a saved query from the given repository path. The path can either
     * point to a String property or to an <code>nt:file</code> node that
     * contains the query as a text file (ie. in the
     * <code>jcr:content/jcr:data</code> property). In both cases, the query
     * definition is expected as a multi-line string in the java properties file
     * format, based on the map reader
     * {@link PredicateConverter#createPredicates(java.util.Map)}.
     * 
     * <p>
     * To store a query, use
     * {@link #storeQuery(Query, String, boolean, Session)}.
     * 
     * @param path
     *            location of the saved query
     * @param session
     *            the JCR session to access the repository
     * @return a query loaded from the repository path, ready to execute, or
     *         <code>null</code> if the query could not be found
     * @throws IOException
     *             if reading from the binary stream failed
     * @throws RepositoryException
     *             if something went wrong in the repository
     * 
     * @since 5.3
     */
    Query loadQuery(String path, Session session) throws RepositoryException, IOException;

    /**
     * Saves a query at the given repository path. This will either create a
     * simple string property or an <code>nt:file</code> containing the query as
     * a text file, depending on the 'createFile' flag. In both cases, the query
     * definition will be stored as a multi-line string in the java properties
     * file format, based on the map created from
     * {@link PredicateConverter#createMap(PredicateGroup)}. If the full path to
     * that property does not exist yet, it will be created using
     * <code>nt:unstructured</code> nodes.
     * 
     * <p>
     * To load this query again, use {@link #loadQuery(String, Session)}.
     * 
     * @param query
     *            the query to save
     * @param path
     *            target location
     * @param createFile
     *            if <code>true</code>, an <code>nt:file</code> will be created
     *            under that path with the query as file contents; otherwise, if
     *            the value is <code>false</code>, a simple string property will
     *            be created at the given location
     * @param session
     *            the JCR session to access the repository
     * @throws IOException
     *             if writing to the binary stream failed
     * @throws RepositoryException
     *             if something went wrong in the repository
     * 
     * @since 5.3
     */
    void storeQuery(Query query, String path, boolean createFile, Session session) throws RepositoryException, IOException;

    /**
     * Invalidates the facet cache, removing all currently
     * cached facets and forcing a new facet extraction on
     * future queries.
     *
     * @since 6.0
     */
    public void clearFacetCache();
}
