/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2015 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 *
 **************************************************************************/
package com.day.cq.wcm.api;

import java.util.List;
import java.util.function.Predicate;

import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ValueMap;
import org.osgi.annotation.versioning.ProviderType;

import com.day.cq.wcm.api.components.ComponentContext;

/**
 * Template Manger API.
 *
 * API for managing 'authored' templates and related components.
 */
@ProviderType
public interface TemplateManager {

    /**
     * Creates a new template
     * @param parentPath the parent path
     * @param templateName the template name
     * @param templateProperties properties of template
     * @param templateType type of template
     * @return Template
     * @throws WCMException is thrown if the template is still used on pages.
     */
    public Template createTemplate(String parentPath, String templateType, String templateName, ValueMap templateProperties) throws WCMException;

    /**
     * Creates a new template
     * @param parentPath the parent path
     * @param templateName the template name
     * @param templateProperties properties of template
     * @param templateType type of template
     * @param autoCommit if false, the template is not created, but the method returns the template that would be created
     * @return Template
     * @throws WCMException is thrown if the template is still used on pages.
     */
    public Template createTemplate(String parentPath, String templateType, String templateName, ValueMap templateProperties, boolean autoCommit) throws WCMException;

    /**
     * @param template the template
     *
     * @throws com.day.cq.wcm.api.WCMException is thrown if the template is still used on pages.
     */
    public void deleteTemplate(Template template) throws WCMException;

    /**
     * @param template the template
     *
     * @return Containing Pages
     */
    public List<Page> getContainingPages(Template template);


    /**
     * @param componentContext the component context
     *
     * @return list of structureResources
     */
    public List<Resource> getStructureResources(ComponentContext componentContext);

    /**
     * @param resource the resource that contains template
     *
     * @return Template
     */
    public Template getContainingTemplate(Resource resource);

    /**
     * @return List of All Templates
     */
    public List<Template> getAllTemplates();


    /**
     * @param filter filter templates
     *
     * @return Get Filtered Templates
     */
    public List<Template> getTemplates(Predicate filter);

    /**
     * @param filter filter templates
     *
     * @return Get Filtered Templates
     * @deprecated use {@link #getTemplates(Predicate filter)} instead
     */
    @Deprecated
    public List<Template> getTemplates(org.apache.commons.collections.Predicate filter);

    /**
     * @param templatePath the path for the template
     *
     * @return the template at the templatePath
     */
    public Template getTemplate(String templatePath);

    /**
     * Get template types
     *  @param parentPath The parent path for the templates
     * @return Template type for each template
     */
    public List<Template> getTemplateTypes(String parentPath);

    /**
     * Verifies if there is any Page using the given Template. If the parameter is null, false is returned.
     *
     * @param template the {@code Template} to test
     * @return {@code true} if there is any Page using the Template
     * @since 1.29
     */
    public boolean isTemplateUsed(Template template);

}
