/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.wcm.designimporter.parser.taghandlers;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

import com.day.cq.wcm.designimporter.ErrorCodes;
import com.day.cq.wcm.designimporter.util.TagUtils;
import org.apache.sling.api.resource.ValueMap;
import org.apache.sling.api.wrappers.ValueMapDecorator;
import org.xml.sax.Attributes;

import com.day.cq.dam.indd.PageComponent;
import com.day.cq.wcm.designimporter.DesignImportException;
import com.day.cq.wcm.designimporter.UnsupportedTagContentException;
import com.day.cq.wcm.designimporter.api.ContainerComponentProvider;
import com.day.cq.wcm.designimporter.parser.HTMLContentType;

/**
 * The parsys component tag handler is used to translate a parsys component div
 * into the parsys component. Since parsys is a container component, its div is only
 * allowed to contain nested component divs.
 *
 * <p>
 * This tag handler is registered against the markup of type < div data-cq-component="parsys" >. That implies
 * this tag handler will come into action whenever the design importer framework
 * encounters an HTML tag matching the rule this handler is defined against.
 * </p>
 *
 * @see com.day.cq.wcm.designimporter.parser.taghandlers.factory.ParsysComponentTagHandlerFactory
 */
public class ParsysComponentTagHandler extends AbstractTagHandler implements ContainerComponentProvider {

    private String suffix;

    private String resourceType;

    @Override
    public void startElement(String uri, String localName, String qName, Attributes atts) throws DesignImportException,
            UnsupportedTagContentException {
        super.startElement(uri, localName, qName, atts);

        if (delegate instanceof DefaultTagHandler) {
            throw new UnsupportedTagContentException(
                    ErrorCodes.PARSYS_NESTED_TAG);
        }
    }

    @Override
    public void characters(char[] ch, int start, int length) throws DesignImportException {
        super.characters(ch, start, length);
        if (delegate == null && new String(ch, start, length).trim().length() > 0) // Only WS characters allowed. Any text is not tolerable.
            throw new UnsupportedTagContentException(
                    ErrorCodes.PARSYS_NESTED_TAG);
    }

    @Override
    public void endHandling(String uri, String localName, String qName) throws DesignImportException {
        super.endHandling(uri, localName, qName);
        suffix = designImporterContext.componentSuffixGenerator.getSuffix("par");
        Map<String, Object> base = new HashMap<String, Object>();
        ValueMap properties = new ValueMapDecorator(base);
        PageComponent parsys = pageBuilder.createComponent(resourceType, properties, getNameHint());
        parsys.getChildComponents().addAll(getPageComponents());
        pageComponents = new ArrayList<PageComponent>();
        pageComponents.add(parsys);
    }

    private String getNameHint() {
        return "par" + suffix;
    }

    @Override
    public boolean supportsContent(HTMLContentType htmlContentType) {
        if (htmlContentType == HTMLContentType.MARKUP)
            return true;
        else
            return false;
    }

    @Override
    public Object getContent(HTMLContentType htmlContentType) {
        if (htmlContentType == HTMLContentType.MARKUP) {
            String cqIncludeJspTag = "<sling:include path=" + "\"" + getNameHint() + "\"" + "/>";
            return componentDivStartTag + cqIncludeJspTag + TagUtils.getMatchingEndTag(componentDivStartTag);
        }
        return null;
    }

    public void setResourceType(String resourceType) {
        this.resourceType = resourceType;
    }
}
