/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.workflow.compatibility;

import com.adobe.granite.workflow.WorkflowException;
import com.adobe.granite.workflow.WorkflowSession;
import com.adobe.granite.workflow.exec.WorkItem;
import com.adobe.granite.workflow.exec.WorkflowExternalProcessProxy;
import com.adobe.granite.workflow.metadata.MetaDataMap;
import com.day.cq.workflow.WorkflowService;
import com.day.cq.workflow.exec.WorkflowExternalProcess;
import com.day.cq.workflow.exec.WorkflowProcess;
import com.day.cq.workflow.impl.CQWorkflowSessionWrapper;
import com.day.cq.workflow.impl.exec.CQWorkItemWrapper;
import com.day.cq.workflow.impl.metadata.CQMetaDataMap;
import org.apache.felix.scr.annotations.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.Serializable;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

@Component(metatype = false, label = "%cq.workflow.compat.extproxy.name", description = "%cq.workflow.compat.extproxy.description")
@Property(name = "service.description", value="%cq.workflow.compat.extproxy.description")
@Reference(name = "WorkflowExternalProcess", cardinality = ReferenceCardinality.OPTIONAL_MULTIPLE, referenceInterface = WorkflowExternalProcess.class, policy = ReferencePolicy.DYNAMIC)
@Service
public class CQWorkflowExtProcessProxy implements WorkflowExternalProcessProxy {
    protected final Logger log = LoggerFactory.getLogger(getClass());

    @Reference
    private WorkflowService cqWorkflowService;

    protected List<WorkflowExternalProcess> externalProcesses = new CopyOnWriteArrayList<WorkflowExternalProcess>();

    public void bindWorkflowExternalProcess(WorkflowExternalProcess process) {
        externalProcesses.add(process);
    }

    public void unbindWorkflowExternalProcess(WorkflowExternalProcess process) {
        externalProcesses.remove(process);
    }

    public Serializable execute(String externalProcessName, WorkItem workItem, WorkflowSession workflowSession, MetaDataMap metaDataMap) throws WorkflowException {
        WorkflowExternalProcess extProc = getExternalProcess(externalProcessName);
        if (extProc!=null) {
            CQWorkflowSessionWrapper sessionWrapper = new CQWorkflowSessionWrapper(cqWorkflowService, workflowSession);
            CQWorkItemWrapper workItemWrapper = new CQWorkItemWrapper(workItem);
            CQMetaDataMap cqMetaDataMap = new CQMetaDataMap(metaDataMap);
            try {
                return extProc.execute(workItemWrapper, sessionWrapper, cqMetaDataMap);
            } catch (com.day.cq.workflow.WorkflowException e) {
                throw new WorkflowException(e.getMessage(), e.getCause());
            }
        } else {
            log.info("no process found in proxy for external process {}",externalProcessName);
        }
        return null;
    }

    public boolean hasFinished(String externalProcessName, Serializable serializable, WorkItem workItem, WorkflowSession workflowSession, MetaDataMap metaDataMap) {
        WorkflowExternalProcess extProc = getExternalProcess(externalProcessName);
        if (extProc!=null) {
            CQWorkflowSessionWrapper sessionWrapper = new CQWorkflowSessionWrapper(cqWorkflowService, workflowSession);
            CQWorkItemWrapper workItemWrapper = new CQWorkItemWrapper(workItem);
            CQMetaDataMap cqMetaDataMap = new CQMetaDataMap(metaDataMap);
            return extProc.hasFinished(serializable, workItemWrapper, sessionWrapper, cqMetaDataMap);
        } else {
            log.info("no process found in proxy for external process {}",externalProcessName);
        }
        // if there is no process fall through return true to continue the workflow ...
        return true;
    }

    public void handleResult(String externalProcessName, Serializable serializable, WorkItem workItem, WorkflowSession workflowSession, MetaDataMap metaDataMap) throws WorkflowException {
        WorkflowExternalProcess extProc = getExternalProcess(externalProcessName);
        if (extProc!=null) {
            CQWorkflowSessionWrapper sessionWrapper = new CQWorkflowSessionWrapper(cqWorkflowService, workflowSession);
            CQWorkItemWrapper workItemWrapper = new CQWorkItemWrapper(workItem);
            CQMetaDataMap cqMetaDataMap = new CQMetaDataMap(metaDataMap);
            try {
                extProc.handleResult(serializable, workItemWrapper, sessionWrapper, cqMetaDataMap);
            } catch (com.day.cq.workflow.WorkflowException e) {
                throw new WorkflowException(e.getMessage(), e.getCause());
            }
        } else {
            log.info("no process found in proxy for external process {}", externalProcessName);
        }
    }

    public boolean canHandle(String externalProcessName) {
        return getExternalProcess(externalProcessName)!=null;
    }

    private WorkflowExternalProcess getExternalProcess(String externalProcessName) {
        if (externalProcessName!=null) {
            for (WorkflowExternalProcess workflowProcess : externalProcesses) {
                if (externalProcessName.equals(workflowProcess.getClass().getName())) {
                    return workflowProcess;
                }
            }
        }
        return null;
    }
}
