/*************************************************************************
*
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 1997 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/
package com.day.crx.statistics.result;

import com.day.crx.statistics.Entry;
import com.day.crx.statistics.PathBuilder;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Item;
import javax.jcr.PathNotFoundException;
import javax.jcr.Value;
import java.util.Calendar;
import java.util.List;
import java.util.ArrayList;

/**
 * <code>ResultSelected</code>...
 *
 * @author mreutegg
 */
public class ResultSelected extends Entry {

    /**
     * Name of the property that contains the count.
     */
    public static final String COUNT = "count";

    /**
     * Name of the property that contains the rolling week count.
     */
    public static final String ROLLING_WEEK_COUNT = "rollingWeekCount";

    /**
     * Name of the property that contains the rolling month count.
     */
    public static final String ROLLING_MONTH_COUNT = "rollingMonthCount";

    /**
     * Name of the property that contains the average position of a selected
     * result.
     */
    public static final String AVG_POSITION = "avgPosition";

    /**
     * Name of the property that contains the recent queries that lead to this
     * result.
     */
    public static final String QUERIES = "queries";

    /**
     * Maximum number of recent queries to remember.
     */
    private static final int MAX_REMEMBERED_QUERIES = 10;

    /**
     * The path of the selected result.
     */
    private final String path;

    /**
     * The position of the selected result within the query result (1-based).
     */
    private final long position;

    /**
     * The query that lead to this result.
     */
    private final String query;

    /**
     * Creates a new <code>ResultSelected</code> instance.
     *
     * @param pathPrefix the location where the query information will be
     *                   stored.
     * @param path       the path of the selected result.
     * @param position   the position of the result.
     * @param query      the query that lead to this result.
     */
    public ResultSelected(String pathPrefix, String path, long position, String query) {
        super(pathPrefix);
        this.path = path;
        this.position = position;
        this.query = query.toLowerCase();
    }

    /**
     * {@inheritDoc}
     */
    protected PathBuilder getPathBuilder() {
        return new ResultSelectedPathBuilder();
    }

    /**
     * Writes the statistics to the passed node.
     *
     * @param node the node where to write the statistics.
     * @throws RepositoryException if an error occurs while writing.
     */
    public void write(Node node) throws RepositoryException {
        long count = updateCount(node);
        Node month = node.getParent();
        updateCount(month);
        Node year = month.getParent();
        updateCount(year);
        updateAveragePosition(node, count);
        updateCumulativeCount(node, ROLLING_WEEK_COUNT, 6);
        updateCumulativeCount(node, ROLLING_MONTH_COUNT, 29);
        updateQuery(node);
        updateQuery(month);
        updateQuery(year);
    }

    /**
     * @return the path of the result node.
     */
    public String getResultPath() {
        return path;
    }

    /**
     * Updates the recent queries property on the given <code>node</code>.
     *
     * @param node the node where to store the recent queries.
     * @throws RepositoryException if an error occurs while writing to the
     *                             node.
     */
    private void updateQuery(Node node) throws RepositoryException {
        if (query == null) {
            return;
        }
        Value q = node.getSession().getValueFactory().createValue(query);
        List queries = new ArrayList();
        queries.add(q);
        if (node.hasProperty(QUERIES)) {
            Value[] values = node.getProperty(QUERIES).getValues();
            for (int i = 0; i < values.length && queries.size() < MAX_REMEMBERED_QUERIES; i++) {
                if (!q.getString().equals(values[i].getString())) {
                    queries.add(values[i]);
                }
            }
        }
        node.setProperty(QUERIES, (Value[]) queries.toArray(new Value[queries.size()]));
    }

    /**
     * Increments the count on the given <code>node</code>.
     *
     * @param node the node where to increment the count.
     * @return the new value of the count.
     * @throws RepositoryException if an error occurs while writing to the
     *                             node.
     */
    private long updateCount(Node node) throws RepositoryException {
        long count = 0;
        if (node.hasProperty(COUNT)) {
            count = node.getProperty(COUNT).getLong();
        }
        node.setProperty(COUNT, ++count);
        return count;
    }

    /**
     * Updates the average position on the given <code>node</code>.
     *
     * @param node  the node where to update the average position.
     * @param count the number of times this result had been selected.
     * @return the updated average position value.
     * @throws RepositoryException if an error occurs while writing to the
     *                             node.
     */
    private double updateAveragePosition(Node node,
                                         long count)
            throws RepositoryException {
        double avgPosition = position;
        if (node.hasProperty(AVG_POSITION)) {
            avgPosition = node.getProperty(AVG_POSITION).getDouble();
        }
        avgPosition = (avgPosition * (count - 1) + position) / count;
        node.setProperty(AVG_POSITION, avgPosition);
        return avgPosition;
    }

    /**
     * Updates a cumulative count on the <code>node</code>.
     *
     * @param node         the node where to update the cumulative count.
     * @param propertyName the name of the count property.
     * @param numDays      the number of days back in time that are cumulated.
     * @return the updated count.
     * @throws RepositoryException if an error occurs while reading or
     *                             updating.
     */
    private long updateCumulativeCount(Node node,
                                       String propertyName,
                                       int numDays) throws RepositoryException {
        long count;
        if (node.hasProperty(propertyName)) {
            count = node.getProperty(propertyName).getLong();
        } else {
            count = 0;
            Session session = node.getSession();
            PathBuilder builder = getPathBuilder();
            Calendar date = Calendar.getInstance();
            date.setTimeInMillis(getTimestamp());
            ResultSelected rs = new ResultSelected(
                    getPathPrefix(), getResultPath(), 1, "");
            StringBuffer buffer = new StringBuffer();
            for (int i = 0; i < numDays; i++) {
                // re-use buffer
                buffer.setLength(0);
                // go back one day
                date.add(Calendar.DAY_OF_MONTH, -1);
                rs.setTimestamp(date.getTimeInMillis());
                builder.formatPath(rs, buffer);
                String path = buffer.toString();
                try {
                    Item item = session.getItem(path);
                    if (item.isNode()) {
                        Node n = (Node) item;
                        if (n.hasProperty(COUNT)) {
                            count += n.getProperty(COUNT).getLong();
                        }
                    }
                } catch (PathNotFoundException e) {
                    // no statistics found for that day
                }
            }
        }
        node.setProperty(propertyName, ++count);
        return count;
    }
}
