/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.blob;

import com.google.common.base.Charsets;
import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.base.StandardSystemProperty;
import com.google.common.base.Stopwatch;
import com.google.common.collect.AbstractIterator;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterators;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.PeekingIterator;
import com.google.common.io.Closeables;
import com.google.common.io.Files;
import com.google.common.util.concurrent.ListenableFutureTask;
import java.io.BufferedWriter;
import java.io.ByteArrayInputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import javax.annotation.Nullable;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.LineIterator;
import org.apache.jackrabbit.core.data.DataRecord;
import org.apache.jackrabbit.core.data.DataStoreException;
import org.apache.jackrabbit.oak.commons.IOUtils;
import org.apache.jackrabbit.oak.plugins.blob.BlobGarbageCollector;
import org.apache.jackrabbit.oak.plugins.blob.BlobReferenceRetriever;
import org.apache.jackrabbit.oak.plugins.blob.GarbageCollectionRepoStats;
import org.apache.jackrabbit.oak.plugins.blob.GarbageCollectorFileState;
import org.apache.jackrabbit.oak.plugins.blob.ReferenceCollector;
import org.apache.jackrabbit.oak.plugins.blob.SharedDataStore;
import org.apache.jackrabbit.oak.plugins.blob.datastore.SharedDataStoreUtils;
import org.apache.jackrabbit.oak.spi.blob.GarbageCollectableBlobStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MarkSweepGarbageCollector
implements BlobGarbageCollector {
    public static final Logger LOG = LoggerFactory.getLogger(MarkSweepGarbageCollector.class);
    public static final String NEWLINE = StandardSystemProperty.LINE_SEPARATOR.value();
    public static final String TEMP_DIR = StandardSystemProperty.JAVA_IO_TMPDIR.value();
    public static final int DEFAULT_BATCH_COUNT = 2048;
    public static final String DELIM = ",";
    private final long maxLastModifiedInterval;
    private final GarbageCollectableBlobStore blobStore;
    private final BlobReferenceRetriever marker;
    private final Executor executor;
    private final int batchCount;
    private final String repoId;
    private final String root;

    public MarkSweepGarbageCollector(BlobReferenceRetriever marker, GarbageCollectableBlobStore blobStore, Executor executor, String root, int batchCount, long maxLastModifiedInterval, @Nullable String repositoryId) throws IOException {
        this.executor = executor;
        this.blobStore = blobStore;
        this.marker = marker;
        this.batchCount = batchCount;
        this.maxLastModifiedInterval = maxLastModifiedInterval;
        this.repoId = repositoryId;
        this.root = root;
    }

    public MarkSweepGarbageCollector(BlobReferenceRetriever marker, GarbageCollectableBlobStore blobStore, Executor executor, @Nullable String repositoryId) throws IOException {
        this(marker, blobStore, executor, TEMP_DIR, 2048, TimeUnit.HOURS.toMillis(24L), repositoryId);
    }

    public MarkSweepGarbageCollector(BlobReferenceRetriever marker, GarbageCollectableBlobStore blobStore, Executor executor, long maxLastModifiedInterval, @Nullable String repositoryId) throws IOException {
        this(marker, blobStore, executor, TEMP_DIR, 2048, maxLastModifiedInterval, repositoryId);
    }

    @Override
    public void collectGarbage(boolean markOnly) throws Exception {
        this.markAndSweep(markOnly);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public List<GarbageCollectionRepoStats> getStats() throws Exception {
        ArrayList<GarbageCollectionRepoStats> stats = Lists.newArrayList();
        if (SharedDataStoreUtils.isShared(this.blobStore)) {
            List<DataRecord> refFiles = ((SharedDataStore)((Object)this.blobStore)).getAllMetadataRecords(SharedDataStoreUtils.SharedStoreRecordType.REFERENCES.getType());
            ImmutableMap<String, DataRecord> references = Maps.uniqueIndex(refFiles, new Function<DataRecord, String>(){

                @Override
                public String apply(DataRecord input) {
                    return SharedDataStoreUtils.SharedStoreRecordType.REFERENCES.getIdFromName(input.getIdentifier().toString());
                }
            });
            List<DataRecord> markerFiles = ((SharedDataStore)((Object)this.blobStore)).getAllMetadataRecords(SharedDataStoreUtils.SharedStoreRecordType.MARKED_START_MARKER.getType());
            ImmutableMap<String, DataRecord> markers = Maps.uniqueIndex(markerFiles, new Function<DataRecord, String>(){

                @Override
                public String apply(DataRecord input) {
                    return SharedDataStoreUtils.SharedStoreRecordType.MARKED_START_MARKER.getIdFromName(input.getIdentifier().toString());
                }
            });
            List<DataRecord> repoFiles = ((SharedDataStore)((Object)this.blobStore)).getAllMetadataRecords(SharedDataStoreUtils.SharedStoreRecordType.REPOSITORY.getType());
            for (DataRecord repoRec : repoFiles) {
                String id = SharedDataStoreUtils.SharedStoreRecordType.REFERENCES.getIdFromName(repoRec.getIdentifier().toString());
                GarbageCollectionRepoStats stat = new GarbageCollectionRepoStats();
                stat.setRepositoryId(id);
                if (id != null && id.equals(this.repoId)) {
                    stat.setLocal(true);
                }
                if (references.containsKey(id)) {
                    DataRecord refRec = (DataRecord)references.get(id);
                    stat.setEndTime(refRec.getLastModified());
                    stat.setLength(refRec.getLength());
                    if (markers.containsKey(id)) {
                        stat.setStartTime(((DataRecord)markers.get(id)).getLastModified());
                    }
                    LineNumberReader reader = null;
                    try {
                        reader = new LineNumberReader(new InputStreamReader(refRec.getStream()));
                        while (reader.readLine() != null) {
                        }
                        stat.setNumLines(reader.getLineNumber());
                    }
                    catch (Throwable throwable) {
                        Closeables.close(reader, true);
                        throw throwable;
                    }
                    Closeables.close(reader, true);
                }
                stats.add(stat);
            }
        }
        return stats;
    }

    protected void markAndSweep(boolean markOnly) throws Exception {
        boolean threw = true;
        GarbageCollectorFileState fs = new GarbageCollectorFileState(this.root);
        try {
            Stopwatch sw = Stopwatch.createStarted();
            LOG.info("Starting Blob garbage collection with markOnly [{}]", (Object)markOnly);
            long markStart = System.currentTimeMillis();
            this.mark(fs);
            if (!markOnly) {
                long deleteCount = this.sweep(fs, markStart);
                threw = false;
                long maxTime = this.getLastMaxModifiedTime(markStart) > 0L ? this.getLastMaxModifiedTime(markStart) : markStart;
                LOG.info("Blob garbage collection completed in {}. Number of blobs deleted [{}] with max modification time of [{}]", new Object[]{sw.toString(), deleteCount, MarkSweepGarbageCollector.timestampToString(maxTime)});
            }
        }
        catch (Exception e) {
            LOG.error("Blob garbage collection error", (Throwable)e);
            throw e;
        }
        finally {
            if (!LOG.isTraceEnabled()) {
                Closeables.close(fs, threw);
            }
        }
    }

    protected void mark(GarbageCollectorFileState fs) throws IOException, DataStoreException {
        LOG.debug("Starting mark phase of the garbage collector");
        GarbageCollectionType.get(this.blobStore).addMarkedStartMarker(this.blobStore, this.repoId);
        this.iterateNodeTree(fs);
        GarbageCollectionType.get(this.blobStore).addMarked(this.blobStore, fs, this.repoId);
        LOG.debug("Ending mark phase of the garbage collector");
    }

    private void difference(GarbageCollectorFileState fs) throws IOException {
        LOG.debug("Starting difference phase of the garbage collector");
        FileLineDifferenceIterator iter = new FileLineDifferenceIterator(fs.getMarkedRefs(), fs.getAvailableRefs());
        this.calculateDifference(fs, iter);
        LOG.debug("Ending difference phase of the garbage collector");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private long calculateDifference(GarbageCollectorFileState fs, FileLineDifferenceIterator iter) throws IOException {
        long numCandidates = 0L;
        BufferedWriter bufferWriter = null;
        try {
            bufferWriter = Files.newWriter(fs.getGcCandidates(), Charsets.UTF_8);
            ArrayList<String> expiredSet = Lists.newArrayList();
            while (iter.hasNext()) {
                expiredSet.add((String)iter.next());
                if (expiredSet.size() <= this.getBatchCount()) continue;
                numCandidates += (long)expiredSet.size();
                MarkSweepGarbageCollector.saveBatchToFile(expiredSet, bufferWriter);
            }
            if (!expiredSet.isEmpty()) {
                numCandidates += (long)expiredSet.size();
                MarkSweepGarbageCollector.saveBatchToFile(expiredSet, bufferWriter);
            }
            LOG.debug("Found candidates - " + numCandidates);
        }
        catch (Throwable throwable) {
            IOUtils.closeQuietly(bufferWriter);
            IOUtils.closeQuietly(iter);
            throw throwable;
        }
        IOUtils.closeQuietly(bufferWriter);
        IOUtils.closeQuietly(iter);
        return numCandidates;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected long sweep(GarbageCollectorFileState fs, long markStart) throws Exception {
        long earliestRefAvailTime;
        try {
            earliestRefAvailTime = GarbageCollectionType.get(this.blobStore).mergeAllMarkedReferences(this.blobStore, fs);
            LOG.debug("Earliest reference available for timestamp [{}]", (Object)earliestRefAvailTime);
            earliestRefAvailTime = earliestRefAvailTime < markStart ? earliestRefAvailTime : markStart;
        }
        catch (Exception e) {
            return 0L;
        }
        new BlobIdRetriever(fs).call();
        this.difference(fs);
        long count = 0L;
        long deleted = 0L;
        long lastMaxModifiedTime = this.getLastMaxModifiedTime(earliestRefAvailTime);
        LOG.debug("Starting sweep phase of the garbage collector");
        LOG.debug("Sweeping blobs with modified time > than the configured max deleted time ({}). ", (Object)MarkSweepGarbageCollector.timestampToString(lastMaxModifiedTime));
        ConcurrentLinkedQueue<String> exceptionQueue = new ConcurrentLinkedQueue<String>();
        LineIterator iterator = FileUtils.lineIterator((File)fs.getGcCandidates(), (String)Charsets.UTF_8.name());
        ArrayList<String> ids = Lists.newArrayList();
        while (iterator.hasNext()) {
            ids.add(iterator.next());
            if (ids.size() < this.getBatchCount()) continue;
            count += (long)ids.size();
            deleted += this.sweepInternal(ids, exceptionQueue, lastMaxModifiedTime);
            ids = Lists.newArrayList();
        }
        if (!ids.isEmpty()) {
            count += (long)ids.size();
            deleted += this.sweepInternal(ids, exceptionQueue, lastMaxModifiedTime);
        }
        BufferedWriter writer = null;
        try {
            if (!exceptionQueue.isEmpty()) {
                writer = Files.newWriter(fs.getGarbage(), Charsets.UTF_8);
                MarkSweepGarbageCollector.saveBatchToFile(Lists.newArrayList(exceptionQueue), writer);
            }
        }
        finally {
            LineIterator.closeQuietly((LineIterator)iterator);
            IOUtils.closeQuietly(writer);
        }
        if (!exceptionQueue.isEmpty()) {
            LOG.warn("Unable to delete some blobs entries from the blob store. Details around such blob entries can be found in [{}]", (Object)fs.getGarbage().getAbsolutePath());
        }
        if (count != deleted) {
            LOG.warn("Deleted only [{}] blobs entries from the [{}] candidates identified. This may happen if blob modified time is > than the max deleted time ({})", new Object[]{deleted, count, MarkSweepGarbageCollector.timestampToString(lastMaxModifiedTime)});
        }
        GarbageCollectionType.get(this.blobStore).removeAllMarkedReferences(this.blobStore);
        LOG.debug("Ending sweep phase of the garbage collector");
        return deleted;
    }

    private int getBatchCount() {
        return this.batchCount;
    }

    private long getLastMaxModifiedTime(long maxModified) {
        return this.maxLastModifiedInterval > 0L ? (maxModified <= 0L ? System.currentTimeMillis() : maxModified) - this.maxLastModifiedInterval : 0L;
    }

    static void saveBatchToFile(List<String> ids, BufferedWriter writer) throws IOException {
        writer.append(Joiner.on(NEWLINE).join(ids));
        writer.append(NEWLINE);
        ids.clear();
        writer.flush();
    }

    private long sweepInternal(List<String> ids, ConcurrentLinkedQueue<String> exceptionQueue, long maxModified) {
        long deleted = 0L;
        try {
            LOG.trace("Blob ids to be deleted {}", ids);
            deleted = this.blobStore.countDeleteChunks(ids, maxModified);
            if (deleted != (long)ids.size()) {
                LOG.debug("Some [{}] blobs were not deleted from the batch : [{}]", (Object)((long)ids.size() - deleted), ids);
            }
        }
        catch (Exception e) {
            LOG.warn("Error occurred while deleting blob with ids [{}]", ids, (Object)e);
            exceptionQueue.addAll(ids);
        }
        return deleted;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void iterateNodeTree(GarbageCollectorFileState fs) throws IOException {
        final BufferedWriter writer = Files.newWriter(fs.getMarkedRefs(), Charsets.UTF_8);
        final AtomicInteger count = new AtomicInteger();
        try {
            this.marker.collectReferences(new ReferenceCollector(){
                private final List<String> idBatch;
                private final boolean debugMode;
                {
                    this.idBatch = Lists.newArrayListWithCapacity(MarkSweepGarbageCollector.this.getBatchCount());
                    this.debugMode = LOG.isTraceEnabled();
                }

                @Override
                public void addReference(String blobId, String nodeId) {
                    if (this.debugMode) {
                        LOG.trace("BlobId : {}, NodeId : {}", (Object)blobId, (Object)nodeId);
                    }
                    try {
                        Iterator<String> idIter = MarkSweepGarbageCollector.this.blobStore.resolveChunks(blobId);
                        Joiner delimJoiner = Joiner.on(MarkSweepGarbageCollector.DELIM).skipNulls();
                        while (idIter.hasNext()) {
                            String id = idIter.next();
                            this.idBatch.add(delimJoiner.join(id, nodeId, new Object[0]));
                            if (this.idBatch.size() >= MarkSweepGarbageCollector.this.getBatchCount()) {
                                MarkSweepGarbageCollector.saveBatchToFile(this.idBatch, writer);
                                this.idBatch.clear();
                            }
                            if (this.debugMode) {
                                LOG.trace("chunkId : {}", (Object)id);
                            }
                            count.getAndIncrement();
                        }
                        if (!this.idBatch.isEmpty()) {
                            MarkSweepGarbageCollector.saveBatchToFile(this.idBatch, writer);
                            this.idBatch.clear();
                        }
                        if (count.get() % MarkSweepGarbageCollector.this.getBatchCount() == 0) {
                            LOG.info("Collected ({}) blob references", (Object)count.get());
                        }
                    }
                    catch (Exception e) {
                        throw new RuntimeException("Error in retrieving references", e);
                    }
                }
            });
            LOG.info("Number of valid blob references marked under mark phase of Blob garbage collection [{}]", (Object)count.get());
            GarbageCollectorFileState.sort(fs.getMarkedRefs(), new Comparator<String>(){

                @Override
                public int compare(String s1, String s2) {
                    return s1.split(MarkSweepGarbageCollector.DELIM)[0].compareTo(s2.split(MarkSweepGarbageCollector.DELIM)[0]);
                }
            });
        }
        finally {
            IOUtils.closeQuietly(writer);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public long checkConsistency() throws Exception {
        boolean threw = true;
        GarbageCollectorFileState fs = new GarbageCollectorFileState(this.root);
        long candidates = 0L;
        try {
            Stopwatch sw = Stopwatch.createStarted();
            LOG.info("Starting blob consistency check");
            ListenableFutureTask<Integer> blobIdRetriever = ListenableFutureTask.create(new BlobIdRetriever(fs));
            this.executor.execute(blobIdRetriever);
            this.iterateNodeTree(fs);
            try {
                blobIdRetriever.get();
            }
            catch (ExecutionException e) {
                LOG.warn("Error occurred while fetching all the blobIds from the BlobStore");
                threw = false;
                throw e;
            }
            LOG.trace("Starting difference phase of the consistency check");
            FileLineDifferenceIterator iter = new FileLineDifferenceIterator(fs.getAvailableRefs(), fs.getMarkedRefs());
            candidates = this.calculateDifference(fs, iter);
            LOG.trace("Ending difference phase of the consistency check");
            LOG.info("Consistency check found [{}] missing blobs", (Object)candidates);
            if (candidates > 0L) {
                LOG.warn("Consistency check failure in the the blob store : {}, check missing candidates in file {}", (Object)this.blobStore, (Object)fs.getGcCandidates().getAbsolutePath());
            }
        }
        finally {
            if (!LOG.isTraceEnabled() && candidates == 0L) {
                Closeables.close(fs, threw);
            }
        }
        return candidates;
    }

    private static String timestampToString(long timestamp) {
        return (new Timestamp(timestamp) + "00").substring(0, 23);
    }

    static enum GarbageCollectionType {
        SHARED{

            @Override
            void removeAllMarkedReferences(GarbageCollectableBlobStore blobStore) {
                ((SharedDataStore)((Object)blobStore)).deleteAllMetadataRecords(SharedDataStoreUtils.SharedStoreRecordType.REFERENCES.getType());
                ((SharedDataStore)((Object)blobStore)).deleteAllMetadataRecords(SharedDataStoreUtils.SharedStoreRecordType.MARKED_START_MARKER.getType());
            }

            @Override
            long mergeAllMarkedReferences(GarbageCollectableBlobStore blobStore, GarbageCollectorFileState fs) throws IOException, DataStoreException {
                List<DataRecord> refFiles = ((SharedDataStore)((Object)blobStore)).getAllMetadataRecords(SharedDataStoreUtils.SharedStoreRecordType.REFERENCES.getType());
                List<DataRecord> repoFiles = ((SharedDataStore)((Object)blobStore)).getAllMetadataRecords(SharedDataStoreUtils.SharedStoreRecordType.REPOSITORY.getType());
                Set<String> unAvailRepos = SharedDataStoreUtils.refsNotAvailableFromRepos(repoFiles, refFiles);
                if (unAvailRepos.isEmpty()) {
                    ArrayList<File> files = Lists.newArrayList();
                    for (DataRecord refFile : refFiles) {
                        File file = GarbageCollectorFileState.copy(refFile.getStream());
                        files.add(file);
                    }
                    GarbageCollectorFileState.merge(files, fs.getMarkedRefs());
                    List<DataRecord> markerFiles = ((SharedDataStore)((Object)blobStore)).getAllMetadataRecords(SharedDataStoreUtils.SharedStoreRecordType.MARKED_START_MARKER.getType());
                    long earliestMarker = SharedDataStoreUtils.getEarliestRecord(markerFiles).getLastModified();
                    LOG.trace("Earliest marker timestamp {}", (Object)earliestMarker);
                    long earliestRef = SharedDataStoreUtils.getEarliestRecord(refFiles).getLastModified();
                    LOG.trace("Earliest ref timestamp {}", (Object)earliestRef);
                    return earliestMarker < earliestRef ? earliestMarker : earliestRef;
                }
                LOG.error("Not all repositories have marked references available : {}", unAvailRepos);
                throw new IOException("Not all repositories have marked references available");
            }

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            void addMarked(GarbageCollectableBlobStore blobStore, GarbageCollectorFileState fs, String repoId) throws DataStoreException, IOException {
                FileInputStream is = new FileInputStream(fs.getMarkedRefs());
                try {
                    ((SharedDataStore)((Object)blobStore)).addMetadataRecord(is, SharedDataStoreUtils.SharedStoreRecordType.REFERENCES.getNameFromId(repoId));
                }
                finally {
                    Closeables.close(is, false);
                }
            }

            @Override
            public void addMarkedStartMarker(GarbageCollectableBlobStore blobStore, String repoId) {
                try {
                    ((SharedDataStore)((Object)blobStore)).addMetadataRecord(new ByteArrayInputStream(new byte[0]), SharedDataStoreUtils.SharedStoreRecordType.MARKED_START_MARKER.getNameFromId(repoId));
                }
                catch (DataStoreException e) {
                    LOG.debug("Error creating marked time marker for repo : {}", (Object)repoId);
                }
            }
        }
        ,
        DEFAULT;


        void removeAllMarkedReferences(GarbageCollectableBlobStore blobStore) {
        }

        void addMarked(GarbageCollectableBlobStore blobStore, GarbageCollectorFileState fs, String repoId) throws DataStoreException, IOException {
        }

        long mergeAllMarkedReferences(GarbageCollectableBlobStore blobStore, GarbageCollectorFileState fs) throws IOException, DataStoreException {
            if (!fs.getMarkedRefs().exists() || fs.getMarkedRefs().length() == 0L) {
                throw new IOException("Marked references not available");
            }
            return fs.getMarkedRefs().lastModified();
        }

        public static GarbageCollectionType get(GarbageCollectableBlobStore blobStore) {
            if (SharedDataStoreUtils.isShared(blobStore)) {
                return SHARED;
            }
            return DEFAULT;
        }

        public void addMarkedStartMarker(GarbageCollectableBlobStore blobStore, String repoId) {
        }
    }

    static class FileLineDifferenceIterator
    extends AbstractIterator<String>
    implements Closeable {
        private final PeekingIterator<String> peekMarked;
        private final LineIterator marked;
        private final LineIterator all;

        public FileLineDifferenceIterator(File marked, File available) throws IOException {
            this(FileUtils.lineIterator((File)marked), FileUtils.lineIterator((File)available));
        }

        public FileLineDifferenceIterator(LineIterator marked, LineIterator available) throws IOException {
            this.marked = marked;
            this.peekMarked = Iterators.peekingIterator(marked);
            this.all = available;
        }

        @Override
        protected String computeNext() {
            String diff = this.computeNextDiff();
            if (diff == null) {
                this.close();
                return (String)this.endOfData();
            }
            return diff;
        }

        @Override
        public void close() {
            LineIterator.closeQuietly((LineIterator)this.marked);
            LineIterator.closeQuietly((LineIterator)this.all);
        }

        private String getKey(String row) {
            return row.split(MarkSweepGarbageCollector.DELIM)[0];
        }

        private String computeNextDiff() {
            if (!this.all.hasNext()) {
                return null;
            }
            if (!this.peekMarked.hasNext()) {
                return this.all.next();
            }
            String diff = null;
            block0: while (this.all.hasNext() && diff == null) {
                diff = this.all.next();
                while (this.peekMarked.hasNext()) {
                    String marked = this.peekMarked.peek();
                    int comparisonResult = this.getKey(diff).compareTo(this.getKey(marked));
                    if (comparisonResult > 0) {
                        this.peekMarked.next();
                        continue;
                    }
                    if (comparisonResult == 0) {
                        this.peekMarked.next();
                        diff = null;
                        continue block0;
                    }
                    return diff;
                }
            }
            return diff;
        }
    }

    private class BlobIdRetriever
    implements Callable<Integer> {
        private final GarbageCollectorFileState fs;

        public BlobIdRetriever(GarbageCollectorFileState fs) {
            this.fs = fs;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public Integer call() throws Exception {
            LOG.debug("Starting retrieve of all blobs");
            BufferedWriter bufferWriter = null;
            int blobsCount = 0;
            try {
                bufferWriter = new BufferedWriter(new FileWriter(this.fs.getAvailableRefs()));
                Iterator<String> idsIter = MarkSweepGarbageCollector.this.blobStore.getAllChunkIds(0L);
                ArrayList<String> ids = Lists.newArrayList();
                while (idsIter.hasNext()) {
                    ids.add(idsIter.next());
                    if (ids.size() <= MarkSweepGarbageCollector.this.getBatchCount()) continue;
                    MarkSweepGarbageCollector.saveBatchToFile(ids, bufferWriter);
                    LOG.info("Retrieved ({}) blobs", (Object)(blobsCount += ids.size()));
                }
                if (!ids.isEmpty()) {
                    MarkSweepGarbageCollector.saveBatchToFile(ids, bufferWriter);
                    LOG.info("Retrieved ({}) blobs", (Object)(blobsCount += ids.size()));
                }
                GarbageCollectorFileState.sort(this.fs.getAvailableRefs());
                LOG.info("Number of blobs present in BlobStore : [{}] ", (Object)blobsCount);
            }
            catch (Throwable throwable) {
                IOUtils.closeQuietly(bufferWriter);
                throw throwable;
            }
            IOUtils.closeQuietly(bufferWriter);
            return blobsCount;
        }
    }
}

