/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.index.lucene;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.io.ByteStreams;
import com.google.common.primitives.Ints;
import java.io.ByteArrayInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.SequenceInputStream;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import org.apache.jackrabbit.oak.api.Blob;
import org.apache.jackrabbit.oak.api.PropertyState;
import org.apache.jackrabbit.oak.api.Type;
import org.apache.jackrabbit.oak.commons.StringUtils;
import org.apache.jackrabbit.oak.plugins.index.lucene.IndexDefinition;
import org.apache.jackrabbit.oak.plugins.memory.PropertyStates;
import org.apache.jackrabbit.oak.spi.state.NodeBuilder;
import org.apache.jackrabbit.oak.util.PerfLogger;
import org.apache.lucene.store.AlreadyClosedException;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.store.IndexInput;
import org.apache.lucene.store.IndexOutput;
import org.apache.lucene.store.Lock;
import org.apache.lucene.store.LockFactory;
import org.apache.lucene.store.NoLockFactory;
import org.apache.lucene.util.WeakIdentityMap;
import org.slf4j.LoggerFactory;

class OakDirectory
extends Directory {
    static final PerfLogger PERF_LOGGER = new PerfLogger(LoggerFactory.getLogger((String)(OakDirectory.class.getName() + ".perf")));
    static final String PROP_DIR_LISTING = "dirListing";
    static final String PROP_BLOB_SIZE = "blobSize";
    static final String PROP_UNIQUE_KEY = "uniqueKey";
    static final int UNIQUE_KEY_SIZE = 16;
    private static final SecureRandom secureRandom = new SecureRandom();
    protected final NodeBuilder builder;
    protected final NodeBuilder directoryBuilder;
    private final IndexDefinition definition;
    private LockFactory lockFactory;
    private final boolean readOnly;
    private final Set<String> fileNames = Sets.newConcurrentHashSet();
    private final boolean activeDeleteEnabled;
    private final String indexName;
    static final int DEFAULT_BLOB_SIZE = 32768;

    public OakDirectory(NodeBuilder builder, IndexDefinition definition, boolean readOnly) {
        this(builder, ":data", definition, readOnly);
    }

    public OakDirectory(NodeBuilder builder, String dataNodeName, IndexDefinition definition, boolean readOnly) {
        this.lockFactory = NoLockFactory.getNoLockFactory();
        this.builder = builder;
        this.directoryBuilder = readOnly ? builder.getChildNode(dataNodeName) : builder.child(dataNodeName);
        this.definition = definition;
        this.readOnly = readOnly;
        this.fileNames.addAll(this.getListing());
        this.activeDeleteEnabled = definition.getActiveDeleteEnabled();
        this.indexName = definition.getIndexName();
    }

    @Override
    public String[] listAll() throws IOException {
        return this.fileNames.toArray(new String[this.fileNames.size()]);
    }

    @Override
    public boolean fileExists(String name) throws IOException {
        return this.fileNames.contains(name);
    }

    @Override
    public void deleteFile(String name) throws IOException {
        Preconditions.checkArgument(!this.readOnly, "Read only directory");
        this.fileNames.remove(name);
        NodeBuilder f = this.directoryBuilder.getChildNode(name);
        if (this.activeDeleteEnabled) {
            PropertyState property = f.getProperty("jcr:data");
            ArrayList<Object> data = property != null && property.getType() == Type.BINARIES ? Lists.newArrayList(property.getValue(Type.BINARIES)) : Lists.newArrayList();
            NodeBuilder trash = this.builder.child(":trash");
            long index = !trash.hasProperty("index") ? 1L : trash.getProperty("index").getValue(Type.LONG) + 1L;
            trash.setProperty("index", index);
            NodeBuilder trashEntry = trash.child("run_" + index);
            trashEntry.setProperty("time", System.currentTimeMillis());
            trashEntry.setProperty("name", name);
            trashEntry.setProperty("jcr:data", data, Type.BINARIES);
        }
        f.remove();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public long fileLength(String name) throws IOException {
        NodeBuilder file = this.directoryBuilder.getChildNode(name);
        OakIndexInput input = new OakIndexInput(name, file, this.indexName);
        try {
            long l = input.length();
            return l;
        }
        finally {
            input.close();
        }
    }

    @Override
    public IndexOutput createOutput(String name, IOContext context) throws IOException {
        NodeBuilder file;
        Preconditions.checkArgument(!this.readOnly, "Read only directory");
        if (!this.directoryBuilder.hasChildNode(name)) {
            file = this.directoryBuilder.child(name);
            byte[] uniqueKey = new byte[16];
            secureRandom.nextBytes(uniqueKey);
            String key = StringUtils.convertBytesToHex(uniqueKey);
            file.setProperty(PROP_UNIQUE_KEY, key);
            file.setProperty(PROP_BLOB_SIZE, this.definition.getBlobSize());
        } else {
            file = this.directoryBuilder.child(name);
        }
        this.fileNames.add(name);
        return new OakIndexOutput(name, file, this.indexName);
    }

    @Override
    public IndexInput openInput(String name, IOContext context) throws IOException {
        NodeBuilder file = this.directoryBuilder.getChildNode(name);
        if (file.exists()) {
            return new OakIndexInput(name, file, this.indexName);
        }
        String msg = String.format("[%s] %s", this.indexName, name);
        throw new FileNotFoundException(msg);
    }

    @Override
    public Lock makeLock(String name) {
        return this.lockFactory.makeLock(name);
    }

    @Override
    public void clearLock(String name) throws IOException {
        this.lockFactory.clearLock(name);
    }

    @Override
    public void sync(Collection<String> names) throws IOException {
    }

    @Override
    public void close() throws IOException {
        if (!this.readOnly && this.definition.saveDirListing()) {
            this.directoryBuilder.setProperty(PropertyStates.createProperty(PROP_DIR_LISTING, this.fileNames, Type.STRINGS));
        }
    }

    @Override
    public void setLockFactory(LockFactory lockFactory) throws IOException {
        this.lockFactory = lockFactory;
    }

    @Override
    public LockFactory getLockFactory() {
        return this.lockFactory;
    }

    @Override
    public String toString() {
        return "Directory for " + this.definition.getIndexName();
    }

    private Set<String> getListing() {
        PropertyState listing;
        long start = PERF_LOGGER.start();
        Iterable<String> fileNames = null;
        if (this.definition.saveDirListing() && (listing = this.directoryBuilder.getProperty(PROP_DIR_LISTING)) != null) {
            fileNames = listing.getValue(Type.STRINGS);
        }
        if (fileNames == null) {
            fileNames = this.directoryBuilder.getChildNodeNames();
        }
        ImmutableSet<String> result = ImmutableSet.copyOf(fileNames);
        PERF_LOGGER.end(start, 100L, "Directory listing performed. Total {} files", (Object)result.size());
        return result;
    }

    private final class OakIndexOutput
    extends IndexOutput {
        private final String dirDetails;
        private final OakIndexFile file;

        public OakIndexOutput(String name, NodeBuilder file, String dirDetails) throws IOException {
            this.dirDetails = dirDetails;
            this.file = new OakIndexFile(name, file, dirDetails);
        }

        @Override
        public long length() {
            return this.file.length;
        }

        @Override
        public long getFilePointer() {
            return this.file.position;
        }

        @Override
        public void seek(long pos) throws IOException {
            this.file.seek(pos);
        }

        @Override
        public void writeBytes(byte[] b, int offset, int length) throws IOException {
            try {
                this.file.writeBytes(b, offset, length);
            }
            catch (IOException e) {
                throw this.wrapWithDetails(e);
            }
        }

        @Override
        public void writeByte(byte b) throws IOException {
            this.writeBytes(new byte[]{b}, 0, 1);
        }

        @Override
        public void flush() throws IOException {
            try {
                this.file.flush();
            }
            catch (IOException e) {
                throw this.wrapWithDetails(e);
            }
        }

        @Override
        public void close() throws IOException {
            this.flush();
            OakIndexFile.access$302(this.file, null);
            this.file.data = null;
        }

        private IOException wrapWithDetails(IOException e) {
            String msg = String.format("Error occurred while writing to blob [%s][%s]", this.dirDetails, this.file.getName());
            return new IOException(msg, e);
        }
    }

    private static class OakIndexInput
    extends IndexInput {
        private final OakIndexFile file;
        private boolean isClone = false;
        private final WeakIdentityMap<OakIndexInput, Boolean> clones;
        private final String dirDetails;

        public OakIndexInput(String name, NodeBuilder file, String dirDetails) {
            super(name);
            this.dirDetails = dirDetails;
            this.file = new OakIndexFile(name, file, dirDetails);
            this.clones = WeakIdentityMap.newConcurrentHashMap();
        }

        private OakIndexInput(OakIndexInput that) {
            super(that.toString());
            this.file = new OakIndexFile(that.file);
            this.clones = null;
            this.dirDetails = that.dirDetails;
        }

        @Override
        public OakIndexInput clone() {
            OakIndexInput clonedIndexInput = new OakIndexInput(this);
            clonedIndexInput.isClone = true;
            if (this.clones != null) {
                this.clones.put(clonedIndexInput, Boolean.TRUE);
            }
            return clonedIndexInput;
        }

        @Override
        public void readBytes(byte[] b, int o, int n) throws IOException {
            this.checkNotClosed();
            this.file.readBytes(b, o, n);
        }

        @Override
        public byte readByte() throws IOException {
            this.checkNotClosed();
            byte[] b = new byte[1];
            this.readBytes(b, 0, 1);
            return b[0];
        }

        @Override
        public void seek(long pos) throws IOException {
            this.checkNotClosed();
            this.file.seek(pos);
        }

        @Override
        public long length() {
            this.checkNotClosed();
            return this.file.length;
        }

        @Override
        public long getFilePointer() {
            this.checkNotClosed();
            return this.file.position;
        }

        @Override
        public void close() {
            OakIndexFile.access$302(this.file, null);
            this.file.data = null;
            if (this.clones != null) {
                Iterator<OakIndexInput> it = this.clones.keyIterator();
                while (it.hasNext()) {
                    OakIndexInput clone = it.next();
                    assert (clone.isClone);
                    clone.close();
                }
            }
        }

        private void checkNotClosed() {
            if (this.file.blob == null && this.file.data == null) {
                throw new AlreadyClosedException("Already closed: [" + this.dirDetails + "] " + this);
            }
        }
    }

    private static class OakIndexFile {
        private final String name;
        private final NodeBuilder file;
        private final int blobSize;
        private long position = 0L;
        private long length;
        private List<Blob> data;
        private boolean dataModified = false;
        private int index = -1;
        private byte[] blob;
        private final byte[] uniqueKey;
        private boolean blobModified = false;
        private final String dirDetails;

        public OakIndexFile(String name, NodeBuilder file, String dirDetails) {
            this.name = name;
            this.file = file;
            this.dirDetails = dirDetails;
            this.blobSize = OakIndexFile.determineBlobSize(file);
            this.uniqueKey = OakIndexFile.readUniqueKey(file);
            this.blob = new byte[this.blobSize];
            PropertyState property = file.getProperty("jcr:data");
            this.data = property != null && property.getType() == Type.BINARIES ? Lists.newArrayList(property.getValue(Type.BINARIES)) : Lists.newArrayList();
            this.length = (long)this.data.size() * (long)this.blobSize;
            if (!this.data.isEmpty()) {
                Blob last = this.data.get(this.data.size() - 1);
                this.length -= (long)this.blobSize - last.length();
                if (this.uniqueKey != null) {
                    this.length -= (long)this.uniqueKey.length;
                }
            }
        }

        private OakIndexFile(OakIndexFile that) {
            this.name = that.name;
            this.file = that.file;
            this.dirDetails = that.dirDetails;
            this.blobSize = that.blobSize;
            this.uniqueKey = that.uniqueKey;
            this.blob = new byte[this.blobSize];
            this.position = that.position;
            this.length = that.length;
            this.data = Lists.newArrayList(that.data);
            this.dataModified = that.dataModified;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void loadBlob(int i) throws IOException {
            Preconditions.checkElementIndex(i, this.data.size());
            if (this.index != i) {
                this.flushBlob();
                Preconditions.checkState(!this.blobModified);
                int n = (int)Math.min((long)this.blobSize, this.length - (long)i * (long)this.blobSize);
                InputStream stream = this.data.get(i).getNewStream();
                try {
                    ByteStreams.readFully(stream, this.blob, 0, n);
                }
                finally {
                    stream.close();
                }
                this.index = i;
            }
        }

        private void flushBlob() throws IOException {
            if (this.blobModified) {
                int n = (int)Math.min((long)this.blobSize, this.length - (long)this.index * (long)this.blobSize);
                InputStream in = new ByteArrayInputStream(this.blob, 0, n);
                if (this.uniqueKey != null) {
                    in = new SequenceInputStream(in, new ByteArrayInputStream(this.uniqueKey));
                }
                Blob b = this.file.createBlob(in);
                if (this.index < this.data.size()) {
                    this.data.set(this.index, b);
                } else {
                    Preconditions.checkState(this.index == this.data.size());
                    this.data.add(b);
                }
                this.dataModified = true;
                this.blobModified = false;
            }
        }

        public void seek(long pos) throws IOException {
            if (pos < 0L || pos > this.length) {
                String msg = String.format("Invalid seek request for [%s][%s], position: %d, file length: %d", this.dirDetails, this.name, pos, this.length);
                throw new IOException(msg);
            }
            this.position = pos;
        }

        public void readBytes(byte[] b, int offset, int len) throws IOException {
            Preconditions.checkPositionIndexes(offset, offset + len, Preconditions.checkNotNull(b).length);
            if (len < 0 || this.position + (long)len > this.length) {
                String msg = String.format("Invalid byte range request for [%s][%s], position: %d, file length: %d, len: %d", this.dirDetails, this.name, this.position, this.length, len);
                throw new IOException(msg);
            }
            int i = (int)(this.position / (long)this.blobSize);
            int o = (int)(this.position % (long)this.blobSize);
            while (len > 0) {
                this.loadBlob(i);
                int l = Math.min(len, this.blobSize - o);
                System.arraycopy(this.blob, o, b, offset, l);
                offset += l;
                len -= l;
                this.position += (long)l;
                ++i;
                o = 0;
            }
        }

        public void writeBytes(byte[] b, int offset, int len) throws IOException {
            int i = (int)(this.position / (long)this.blobSize);
            int o = (int)(this.position % (long)this.blobSize);
            while (len > 0) {
                int l = Math.min(len, this.blobSize - o);
                if (this.index != i) {
                    if (o > 0 || l < this.blobSize && this.position + (long)l < this.length) {
                        this.loadBlob(i);
                    } else {
                        this.flushBlob();
                        this.index = i;
                    }
                }
                System.arraycopy(b, offset, this.blob, o, l);
                this.blobModified = true;
                offset += l;
                len -= l;
                this.position += (long)l;
                this.length = Math.max(this.length, this.position);
                ++i;
                o = 0;
            }
        }

        private static int determineBlobSize(NodeBuilder file) {
            if (file.hasProperty(OakDirectory.PROP_BLOB_SIZE)) {
                return Ints.checkedCast(file.getProperty(OakDirectory.PROP_BLOB_SIZE).getValue(Type.LONG));
            }
            return 32768;
        }

        private static byte[] readUniqueKey(NodeBuilder file) {
            if (file.hasProperty(OakDirectory.PROP_UNIQUE_KEY)) {
                String key = file.getString(OakDirectory.PROP_UNIQUE_KEY);
                return StringUtils.convertHexToBytes(key);
            }
            return null;
        }

        public void flush() throws IOException {
            this.flushBlob();
            if (this.dataModified) {
                this.file.setProperty("jcr:lastModified", System.currentTimeMillis());
                this.file.setProperty("jcr:data", this.data, Type.BINARIES);
                this.dataModified = false;
            }
        }

        public String toString() {
            return this.name;
        }

        public String getName() {
            return this.name;
        }

        static /* synthetic */ byte[] access$302(OakIndexFile x0, byte[] x1) {
            x0.blob = x1;
            return x1;
        }
    }
}

