/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.segment.file;

import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.io.Closeable;
import java.io.File;
import java.io.FileDescriptor;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.UUID;
import java.util.zip.CRC32;
import org.apache.jackrabbit.oak.plugins.segment.SegmentId;
import org.apache.jackrabbit.oak.plugins.segment.file.FileStoreMonitor;
import org.apache.jackrabbit.oak.plugins.segment.file.TarEntry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class TarWriter
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(TarWriter.class);
    static final int INDEX_MAGIC = 170937098;
    static final int GRAPH_MAGIC = 170936074;
    static final int BLOCK_SIZE = 512;
    private static final byte[] ZERO_BYTES = new byte[512];
    private final File file;
    private final FileStoreMonitor monitor;
    private RandomAccessFile access = null;
    private FileChannel channel = null;
    private boolean closed = false;
    private final Map<UUID, TarEntry> index = Maps.newLinkedHashMap();
    private final Set<UUID> references = Sets.newHashSet();
    private final SortedMap<UUID, List<UUID>> graph = Maps.newTreeMap();

    static final int getPaddingSize(int size) {
        int remainder = size % 512;
        if (remainder > 0) {
            return 512 - remainder;
        }
        return 0;
    }

    TarWriter(File file) {
        this(file, FileStoreMonitor.DEFAULT);
    }

    TarWriter(File file, FileStoreMonitor monitor) {
        this.file = file;
        this.monitor = monitor;
    }

    synchronized int count() {
        return this.index.size();
    }

    synchronized Set<UUID> getUUIDs() {
        return Sets.newHashSet(this.index.keySet());
    }

    synchronized boolean containsEntry(long msb, long lsb) {
        Preconditions.checkState(!this.closed);
        return this.index.containsKey(new UUID(msb, lsb));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    ByteBuffer readEntry(long msb, long lsb) throws IOException {
        TarEntry entry;
        Preconditions.checkState(!this.closed);
        TarWriter tarWriter = this;
        synchronized (tarWriter) {
            entry = this.index.get(new UUID(msb, lsb));
        }
        if (entry != null) {
            Preconditions.checkState(this.channel != null);
            ByteBuffer data = ByteBuffer.allocate(entry.size());
            this.channel.read(data, entry.offset());
            data.rewind();
            return data;
        }
        return null;
    }

    long writeEntry(long msb, long lsb, byte[] data, int offset, int size) throws IOException {
        Preconditions.checkNotNull(data);
        Preconditions.checkPositionIndexes(offset, offset + size, data.length);
        UUID uuid = new UUID(msb, lsb);
        CRC32 checksum = new CRC32();
        checksum.update(data, offset, size);
        String entryName = String.format("%s.%08x", uuid, checksum.getValue());
        byte[] header = TarWriter.newEntryHeader(entryName, size);
        log.debug("Writing segment {} to {}", (Object)uuid, (Object)this.file);
        return this.writeEntry(uuid, header, data, offset, size);
    }

    private synchronized long writeEntry(UUID uuid, byte[] header, byte[] data, int offset, int size) throws IOException {
        int pos;
        ByteBuffer segment;
        int refcount;
        long currentLength;
        Preconditions.checkState(!this.closed);
        if (this.access == null) {
            this.access = new RandomAccessFile(this.file, "rw");
            this.channel = this.access.getChannel();
        }
        long initialLength = this.access.getFilePointer();
        this.access.write(header);
        this.access.write(data, offset, size);
        int padding = TarWriter.getPaddingSize(size);
        if (padding > 0) {
            this.access.write(ZERO_BYTES, 0, padding);
        }
        Preconditions.checkState((currentLength = this.access.getFilePointer()) <= Integer.MAX_VALUE);
        TarEntry entry = new TarEntry(uuid.getMostSignificantBits(), uuid.getLeastSignificantBits(), (int)(currentLength - (long)size - (long)padding), size);
        this.index.put(uuid, entry);
        if (SegmentId.isDataSegmentId(uuid.getLeastSignificantBits()) && (refcount = (segment = ByteBuffer.wrap(data, offset, size)).get((pos = segment.position()) + 5) & 0xFF) != 0) {
            int refend = pos + 16 * (refcount + 1);
            ArrayList<UUID> list = Lists.newArrayListWithCapacity(refcount);
            for (int refpos = pos + 16; refpos < refend; refpos += 16) {
                UUID refid = new UUID(segment.getLong(refpos), segment.getLong(refpos + 8));
                if (!this.index.containsKey(refid)) {
                    this.references.add(refid);
                }
                list.add(refid);
            }
            Collections.sort(list);
            this.graph.put(uuid, list);
        }
        this.monitor.written(currentLength - initialLength);
        return currentLength;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void flush() throws IOException {
        File file = this.file;
        synchronized (file) {
            FileDescriptor descriptor = null;
            TarWriter tarWriter = this;
            synchronized (tarWriter) {
                if (this.access != null && !this.closed) {
                    descriptor = this.access.getFD();
                }
            }
            if (descriptor != null) {
                descriptor.sync();
            }
        }
    }

    boolean isDirty() {
        return this.access != null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws IOException {
        long currentPosition;
        long initialPosition;
        TarWriter tarWriter = this;
        synchronized (tarWriter) {
            Preconditions.checkState(!this.closed);
            this.closed = true;
        }
        if (this.access == null) {
            return;
        }
        File file = this.file;
        synchronized (file) {
            initialPosition = this.access.getFilePointer();
            this.writeGraph();
            this.writeIndex();
            this.access.write(ZERO_BYTES);
            this.access.write(ZERO_BYTES);
            currentPosition = this.access.getFilePointer();
            this.access.close();
        }
        this.monitor.written(currentPosition - initialPosition);
    }

    private void writeGraph() throws IOException {
        ArrayList<UUID> uuids = Lists.newArrayListWithCapacity(this.index.size() + this.references.size());
        uuids.addAll(this.index.keySet());
        uuids.addAll(this.references);
        Collections.sort(uuids);
        int graphSize = uuids.size() * 16 + 16;
        for (List<UUID> list : this.graph.values()) {
            graphSize += 4 + list.size() * 4 + 4;
        }
        int padding = TarWriter.getPaddingSize(graphSize);
        String graphName = this.file.getName() + ".gph";
        byte[] header = TarWriter.newEntryHeader(graphName, graphSize + padding);
        ByteBuffer buffer = ByteBuffer.allocate(graphSize);
        HashMap<UUID, Integer> refmap = Maps.newHashMap();
        int index = 0;
        for (UUID uUID : uuids) {
            buffer.putLong(uUID.getMostSignificantBits());
            buffer.putLong(uUID.getLeastSignificantBits());
            refmap.put(uUID, index++);
        }
        for (Map.Entry entry : this.graph.entrySet()) {
            buffer.putInt((Integer)refmap.get(entry.getKey()));
            for (UUID refid : (List)entry.getValue()) {
                buffer.putInt((Integer)refmap.get(refid));
            }
            buffer.putInt(-1);
        }
        CRC32 checksum = new CRC32();
        checksum.update(buffer.array(), 0, buffer.position());
        buffer.putInt((int)checksum.getValue());
        buffer.putInt(uuids.size());
        buffer.putInt(graphSize);
        buffer.putInt(170936074);
        this.access.write(header);
        if (padding > 0) {
            this.access.write(ZERO_BYTES, 0, padding);
        }
        this.access.write(buffer.array());
    }

    private void writeIndex() throws IOException {
        int indexSize = this.index.size() * 24 + 16;
        int padding = TarWriter.getPaddingSize(indexSize);
        String indexName = this.file.getName() + ".idx";
        byte[] header = TarWriter.newEntryHeader(indexName, indexSize + padding);
        ByteBuffer buffer = ByteBuffer.allocate(indexSize);
        TarEntry[] sorted = this.index.values().toArray(new TarEntry[this.index.size()]);
        Arrays.sort(sorted, TarEntry.IDENTIFIER_ORDER);
        for (TarEntry entry : sorted) {
            buffer.putLong(entry.msb());
            buffer.putLong(entry.lsb());
            buffer.putInt(entry.offset());
            buffer.putInt(entry.size());
        }
        CRC32 checksum = new CRC32();
        checksum.update(buffer.array(), 0, buffer.position());
        buffer.putInt((int)checksum.getValue());
        buffer.putInt(this.index.size());
        buffer.putInt(padding + indexSize);
        buffer.putInt(170937098);
        this.access.write(header);
        if (padding > 0) {
            this.access.write(ZERO_BYTES, 0, padding);
        }
        this.access.write(buffer.array());
    }

    private static byte[] newEntryHeader(String name, int size) {
        byte[] header = new byte[512];
        byte[] nameBytes = name.getBytes(Charsets.UTF_8);
        System.arraycopy(nameBytes, 0, header, 0, Math.min(nameBytes.length, 100));
        System.arraycopy(String.format("%07o", 256).getBytes(Charsets.UTF_8), 0, header, 100, 7);
        System.arraycopy(String.format("%07o", 0).getBytes(Charsets.UTF_8), 0, header, 108, 7);
        System.arraycopy(String.format("%07o", 0).getBytes(Charsets.UTF_8), 0, header, 116, 7);
        System.arraycopy(String.format("%011o", size).getBytes(Charsets.UTF_8), 0, header, 124, 11);
        long time = System.currentTimeMillis() / 1000L;
        System.arraycopy(String.format("%011o", time).getBytes(Charsets.UTF_8), 0, header, 136, 11);
        System.arraycopy(new byte[]{32, 32, 32, 32, 32, 32, 32, 32}, 0, header, 148, 8);
        header[156] = 48;
        int checksum = 0;
        for (int i = 0; i < header.length; ++i) {
            checksum += header[i] & 0xFF;
        }
        System.arraycopy(String.format("%06o\u0000 ", checksum).getBytes(Charsets.UTF_8), 0, header, 148, 8);
        return header;
    }

    synchronized void collectReferences(Set<UUID> referencedIds) {
        for (UUID uuid : Lists.reverse(Lists.newArrayList(this.index.keySet()))) {
            List refs;
            if (!referencedIds.remove(uuid) || (refs = (List)this.graph.get(uuid)) == null) continue;
            referencedIds.addAll(refs);
        }
    }

    public String toString() {
        return this.file.toString();
    }
}

