/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.address.api;

import java.util.Map;

import org.apache.sling.api.adapter.SlingAdaptable;
import org.apache.sling.api.resource.ModifiableValueMap;
import org.apache.sling.api.resource.PersistenceException;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ValueMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Represents an address. Use {@link AddressManager} to create address objects.
 */
public final class Address extends SlingAdaptable implements Comparable<Address> {

    // Most common address properties:

    /**
     * Property describing the address label.
     * Can be used to describe the address, e.g. "my winter appartment"
     */
    public static final String LABEL = "label";

    /**
     * Property describing the address title. E.g.: Mr.
     */
    public static final String TITLE = "title";

    /**
     * Property describing the address first name.
     */
    public static final String FIRST_NAME = "firstname";

    /**
     * Property describing the address last name.
     */
    public static final String LAST_NAME = "lastname";

    /**
     * Property describing the 1st line of the address street.
     */
    public static final String STREET_LINE1 = "street1";

    /**
     * Property describing the 2nd line of the address street.
     */
    public static final String STREET_LINE2 = "street2";

    /**
     * Property describing the address city.
     */
    public static final String CITY = "city";

    /**
     * Property describing the zip code.
     */
    public static final String ZIP_CODE = "zip";

    /**
     * Property describing the state (mainly used in the U.S.)
     */
    public static final String STATE = "state";

    /**
     * Property describing the country.
     */
    public static final String COUNTRY = "country";

    /**
     * Address path parameter name.
     */
    public static final String PARAM_ADDRESS_PATH = "addressPath";

    private static final Logger log = LoggerFactory.getLogger(Address.class);
    private Resource addressRes;
    private ValueMap properties;
    private ResourceResolver resolver;


    /**
     * Creates an address object.
     *
     * @param res The resource that represents the address.
     */
    public Address(Resource res) {
        if (res == null) {
            throw new IllegalArgumentException("Cannot instantiate address. Resource cannot be null.");
        }
        this.addressRes = res;
        this.properties = res.adaptTo(ValueMap.class);
        this.resolver = res.getResourceResolver();
    }

    /**
     * Returns the address path in the repository.
     *
     * @return The address path.
     */
    public String getPath() {
        return addressRes.getPath();
    }

    /**
     * Returns the address properties.
     *
     * @return The address properties as a {@link ValueMap}.
     */
    public ValueMap getProperties() {
        return properties;
    }

    /**
     * Returns a named address property converted to a given type.
     *
     * @param key The name of the property
     * @param type The type to which the property should be converted.
     * @return The named value converted to type T.
     */
    public <T> T getProperty(String key, Class<T> type) {
        return properties.get(key, type);
    }

    /**
     * Returns a named address property or a default value if the property can not be retrieved.
     *
     * @param key The name of the property
     * @param defaultValue The default value to use if the named property does
     *            not exist or cannot be converted to the requested type. The
     *            default value is also used to define the type to convert the
     *            value to. If this is <code>null</code> any existing property is
     *            not converted.
     * @return The named value converted to type T or the default value if
     *         non existing or can't be converted.
     */
    public <T> T getProperty(String key, T defaultValue) {
        return properties.get(key, defaultValue);
    }

    /**
     * Sets the address properties. All the properties are replaced with the existing ones.
     *
     * @param properties The new address properties.
     * @throws AddressException if persisting the properties fails.
     */
    public void setProperties(Map<String, Object> properties) throws AddressException {
        ModifiableValueMap addressProps = addressRes.adaptTo(ModifiableValueMap.class);
        addressProps.clear();
        addressProps.putAll(properties);
        try {
            resolver.commit();
        } catch (PersistenceException e) {
            throw new AddressException("Failed to persist the address properties", e);
        }
        this.properties = addressProps;
        log.debug("Properties were set for address {}", addressRes.getPath());
    }

    /**
     * Sets an address property.
     *
     * @param key The name of the property
     * @param value The value of the property
     * @throws AddressException if persisting the property fails.
     */
    public void setProperty(String key, Object value) throws AddressException {
        ModifiableValueMap addressProps = addressRes.adaptTo(ModifiableValueMap.class);
        addressProps.put(key, value);
        try {
            resolver.commit();
        } catch (PersistenceException e) {
            throw new AddressException("Failed to persist the address property", e);
        }
        this.properties = addressProps;
        log.debug("Property {} was set for address {}", key, addressRes.getPath());
    }

    /**
     * Returns the address label.
     */
    public String getLabel() {
        return getProperty(LABEL, "");
    }

    /**
     * Sets the address label.
     */
    public void setLabel(String label) throws AddressException {
        setProperty(LABEL, label);
    }

    /**
     * Returns the address title.
     */
    public String getTitle() {
        return getProperty(TITLE, "");
    }

    /**
     * Sets the address title.
     */
    public void setTitle(String title) throws AddressException {
        setProperty(TITLE, title);
    }

    /**
     * Returns the address first name.
     */
    public String getFirstName() {
        return getProperty(FIRST_NAME, "");
    }

    /**
     * Sets the address first name.
     */
    public void setFirstName(String firstName) throws AddressException {
        setProperty(FIRST_NAME, firstName);
    }

    /**
     * Returns the address last name.
     */
    public String getLastName() {
        return getProperty(LAST_NAME, "");
    }

    /**
     * Sets the address last name.
     */
    public void setLastName(String lastName) throws AddressException {
        setProperty(LAST_NAME, lastName);
    }

    /**
     * Returns the 1st line of the address street.
     */
    public String getStreetLine1() {
        return getProperty(STREET_LINE1, "");
    }

    /**
     * Sets the 1st line of the address street.
     */
    public void setStreetLine1(String streetLine1) throws AddressException {
        setProperty(STREET_LINE1, streetLine1);
    }

    /**
     * Returns the 2nd line of the address street.
     */
    public String getStreetLine2() {
        return getProperty(STREET_LINE2, "");
    }

    /**
     * Sets the 2nd line of the address street.
     */
    public void setStreetLine2(String streetLine1) throws AddressException {
        setProperty(STREET_LINE2, streetLine1);
    }

    /**
     * Returns the address city.
     */
    public String getCity() {
        return getProperty(CITY, "");
    }

    /**
     * Sets the address city.
     */
    public void setCity(String city) throws AddressException {
        setProperty(CITY, city);
    }

    /**
     * Returns the address zip code.
     */
    public String getZipCode() {
        return getProperty(ZIP_CODE, "");
    }

    /**
     * Sets the address  zip code.
     */
    public void setZipCode(String zipCode) throws AddressException {
        setProperty(ZIP_CODE, zipCode);
    }

    /**
     * Returns the address state.
     */
    public String getState() {
        return getProperty(STATE, "");
    }

    /**
     * Sets the address state.
     */
    public void setState(String state) throws AddressException {
        setProperty(STATE, state);
    }

    /**
     * Returns the address country.
     */
    public String getCountry() {
        return getProperty(COUNTRY, "");
    }

    /**
     * Sets the address country.
     */
    public void setCountry(String country) throws AddressException {
        setProperty(COUNTRY, country);
    }

    /**
     * Compares the current address with another one for sorting. The comparison is based on the last name if available,
     * on the address path otherwise.
     *
     * @param address The address which has to be compared with the current one
     * @return a negative integer, zero, or a positive integer as this address
     *         is less than, equal to, or greater than the specified one.
     * @see Comparable#compareTo(Object)
     */
    public int compareTo(Address address) {
        if (address == null) {
            return 1;
        }
        if (this.getLastName() != null && address.getLastName() != null && !this.getLastName().equals(address.getLastName())) {
            return this.getLastName().compareTo(address.getLastName());
        }
        return this.getPath().compareTo(address.getPath());
    }

    /**
     * {@inheritDoc}
     */
    @SuppressWarnings("unchecked")
    public <AdapterType> AdapterType adaptTo(Class<AdapterType> type) {
        if (type == Resource.class) {
            return (AdapterType)addressRes;
        } else if(type == ValueMap.class) {
            return (AdapterType) properties;
        }

        AdapterType ret = super.adaptTo(type);
        if (ret == null) {
            ret = addressRes.adaptTo(type);
        }

        return ret;
    }

}
