/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.cq.address.api;

import java.util.Collection;
import java.util.Map;

import org.apache.sling.api.resource.ResourceResolver;

public interface AddressProvider {

    /**
     * Returns the addresses belonging to a user.
     *
     * @param resolver The resource resolver used to access the address book
     * @param userID The user ID
     * @return The addresses or <code>null</code> if there are none
     */
    public Collection<Address> getAddresses(ResourceResolver resolver, String userID);

    /**
     * Returns the address based on a path if represented in the repository or based on an ID uniquely identifying
     * the address in an external provider.
     *
     * @param resolver The resource resolver used to access the address book
     * @param userID The user ID
     * @param path The path identifying the address or an ID uniquely identifying the address in an external provider
     *
     * @return The address or <code>null<code> if it does not exist or if it does not belong to the user
     */
    public Address getAddress(ResourceResolver resolver, String userID, String path);

    /**
     * Removes the address.
     *
     * @param resolver The resource resolver used to access the address book
     * @param userID The user ID
     * @param address The address
     * @param autoSave If <code>true</code> the operation is persisted
     * @throws AddressException when the address does not belong to the user or when the operation fails.
     */
    public void removeAddress(ResourceResolver resolver, String userID, Address address, boolean autoSave) throws AddressException;

    /**
     * Adds an address to the user.
     *
     * @param resolver The resource resolver used to access the address book
     * @param userID The user ID
     * @param properties The address properties
     * @param autoSave If <code>true</code> the operation is persisted
     * @return The new address
     * @throws AddressException when the operation fails.
     */
    public Address addAddress(ResourceResolver resolver, String userID, Map<String, Object> properties, boolean autoSave) throws AddressException;

    /**
     * Updates the address.
     *
     * @param resolver The resource resolver used to access the address book
     * @param userID The user ID
     * @param address The address
     * @param properties The address properties
     * @param autoSave If <code>true</code> the operation is persisted
     * @return The updated address
     * @throws AddressException when the address does not belong to the user or when the operation fails.
     */
    public Address updateAddress(ResourceResolver resolver, String userID, Address address, Map<String, Object> properties, boolean autoSave) throws AddressException;

    /**
     * Returns the generic default address of the user.
     *
     * @param resolver The resource resolver used to access the address book
     * @param userID The user ID
     */
    public Address getDefaultAddress(ResourceResolver resolver, String userID);

    /**
     * Returns the default address that is defined for the given type (e.g.: billing, shipping.).
     *
     * @param resolver The resource resolver used to access the address book
     * @param userID The user ID
     * @param type The type of the default address. E.g.: billing, shipping.
     *             If the  type is <code>null</code> or empty, the generic default address is returned.
     */
    public Address getDefaultAddress(ResourceResolver resolver, String userID, String type);

    /**
     * Sets the generic default address of the user.
     *
     * @param resolver The resource resolver used to access the address book
     * @param userID The user ID
     * @param address The address
     * @param autoSave If <code>true</code> the operation is persisted
     * @throws AddressException when the address does not belong to the user or when the operation fails.
     */
    public void setDefaultAddress(ResourceResolver resolver, String userID, Address address, boolean autoSave) throws AddressException;

    /**
     * Sets the default address for the given type (e.g.: billing, shipping.).
     *
     * @param resolver The resource resolver used to access the address book
     * @param userID The user ID
     * @param address The address
     * @param type The type of the default address. E.g.: billing, shipping.
     *             If the given type is <code>null</code> or empty, the generic default address is set.
     * @param autoSave If <code>true</code> the operation is persisted
     * @throws AddressException when the address does not belong to the user or when the operation fails.
     */
    public void setDefaultAddress(ResourceResolver resolver, String userID, Address address, String type, boolean autoSave) throws AddressException;

}

