/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.address.api.location;

import com.adobe.cq.address.api.Address;
import com.adobe.cq.address.api.AddressException;
import com.day.cq.commons.jcr.JcrConstants;
import org.apache.sling.api.adapter.SlingAdaptable;
import org.apache.sling.api.resource.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

/**
 * Represents a location.  Use {@link com.adobe.cq.address.api.location.LocationManager} to access location objects.
 */
public final class Location extends SlingAdaptable {

    private static final Logger log = LoggerFactory.getLogger(Location.class);

    private final Address address;

    private String[] defaultAddressFields = {
            Address.STREET_LINE1,
            Address.STREET_LINE2,
            Address.CITY,
            Address.STATE,
            Address.ZIP_CODE,
            Address.COUNTRY
    };

    public static final String LATITUDE = "latitude";
    public static final String LONGITUDE = "longitude";
    public static final String PHONE = "phone";
    public static final String HOURS = "hours";

    /**
     * Creates a location object.
     *
     * @param resource The resource representing a location.
     *
     * @throws IllegalArgumentException when Resource is <code>null</code>
     */
    public Location(final Resource resource) {
        if (resource == null) {
            throw new IllegalArgumentException("Cannot instantiate address. Resource cannot be null.");
        }
        this.address = resource.adaptTo(Address.class);
    }

    /**
     * Returns the location path in the repository.
     *
     * @return The address path.
     */
    public String getPath() {
        return address.getPath();
    }

    /**
     * Returns the title of this location as entered by the user.
     *
     * @return The location title or <code>null</code> if not defined.
     */
    public String getTitle() {
        return address.getProperty(JcrConstants.JCR_TITLE, "");
    }

    /**
     * Return the description of this location  as entered by the user.
     *
     * @return The description of this location  or <code>null</code> if not defined.
     */
    public String getDescription() {
        return address.getProperty(JcrConstants.JCR_DESCRIPTION, "");
    }

    /**
     * Sets the location coordinate properties.
     *
     * @param coord The new location properties.
     * @throws AddressException if persisting the coordinates fails.
     */
    public void setCoordinates(Coordinates coord) throws AddressException {
        address.setProperty(Location.LATITUDE, coord.getLat());
        address.setProperty(Location.LONGITUDE, coord.getLng());
    }

    /**
     * Returns coordinates for this location if latitude and longitude are both available.
     *
     * @return The location coordinates or <code>null</code> if not defined
     */
    public Coordinates getCoordinates() {
        Double lat = address.getProperty(LATITUDE, Double.class);
        Double lng = address.getProperty(LONGITUDE, Double.class);
        Coordinates coords = new Coordinates(lat, lng);
        if (!coords.validate()) {
            return null;
        }
        return coords;
    }

    /**
     * Returns the location phone number.
     */
    public String getPhone() {
        return address.getProperty(PHONE, "");
    }

    /**
     * Sets the location phone number.
     */
    public void setPhone(String phone) throws AddressException {
        address.setProperty(PHONE, phone);
    }

    /**
     * Returns the location hours, or null, if no hours property exists.
     */
    public String[] getHours() {
        return address.getProperty(HOURS, null);
    }

    /**
     * Sets the location hours.
     */
    public void setHours(String[] hours) throws AddressException {
        address.setProperty(HOURS, hours);
    }

    /**
     * The full string representation of a location using the default set of fields
     * * @return full address string
     */
    public String getFullAddress() {
        return getFullAddress(defaultAddressFields);
    }

    /**
     * The full string representation of a location using the provided array of property names.
     * @param fields Property names to use
     * @return full address string
     */
    public String getFullAddress(String[] fields) {
        StringBuilder sb = new StringBuilder();
        if (fields != null) {
            for (int i=0; i < fields.length; i++) {
                sb.append(this.address.getProperty(fields[i], "")).append(System.getProperty("line.separator"));
            }
        }
        return sb.toString();
    }

    //-------------< Object overrides >-------------------------------------------------------

    @Override
    public boolean equals(final Object obj) {
        return obj instanceof Location && ((Location) obj).getTitle().equals(getTitle());
    }

    @Override
    public int hashCode() {
        return getTitle().hashCode();
    }

    @SuppressWarnings("unchecked")
    public <AdapterType> AdapterType adaptTo(Class<AdapterType> type) {
        if (type == Address.class) {
            return (AdapterType) address;
        }

        AdapterType ret = super.adaptTo(type);
        if (ret == null) {
            ret = address.adaptTo(type);
        }

        return ret;
    }

}
