/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.address.api.location;

import com.adobe.cq.address.api.AddressException;

import java.util.List;
import java.util.Map;

/**
 * The <code>LocationManager</code> provides methods to retrieve, update, create and update locations.
 */
public interface LocationManager {

    /**
     * Returns the locations belonging to a location group.
     *
     * @param groupPath The location group path
     * @return The addresses or <code>null</code> if there are none
     */
    public List<Location> getLocations(String groupPath);

    /**
     * Returns a sorted list of locations ranked by closest distance to origin coordinates.
     *
     * @param groupPath The location group path
     * @param origin Latitude and longitude coordinates to rank locations against
     * @param geocode Flag that indicates if each location should be geocoded if it does not have stored coordinates
     *
     * @return ranked list of locations
     */
    public List<Location> rankLocations(String groupPath, Coordinates origin, boolean geocode);

    /**
     * Returns the location based on its path.
     *
     * @param path The location path
     *
     * @return The location or <code>null<code> if it does not exist
     */
    public Location getLocation(String path);

    /**
     * Removes the location.
     *
     * @param location The location address
     * @param autoSave If <code>true</code> the operation is persisted
     * @throws AddressException when the operation fails.
     * @throws IllegalArgumentException when the location is <code>null</code>
     */
    public void removeLocation(Location location, boolean autoSave) throws AddressException;

    /**
     * Adds a location to the location group.
     *
     * @param groupPath The location group path
     * @param properties The location properties
     * @param autoSave If <code>true</code> the operation is persisted
     * @return The new location
     * @throws AddressException when the operation fails.
     */
    public Location addLocation(String groupPath, Map<String, Object> properties, boolean autoSave) throws AddressException;

    /**
     * Updates the location.
     *
     * @param location The location address
     * @param properties The location properties
     * @param autoSave If <code>true</code> the operation is persisted
     * @return The updated location
     * @throws AddressException when the operation fails.
     * @throws IllegalArgumentException when the location is <code>null</code>
     */
    public Location updateLocation(Location location, Map<String, Object> properties, boolean autoSave) throws AddressException;

}
