/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.commerce.api;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;

import aQute.bnd.annotation.ProviderType;
import org.apache.commons.lang.StringUtils;

import com.adobe.cq.commerce.common.FacetParamHelper;

/**
 * Class that describes a commerce search query. Use the embedded {@link Builder} to create instances of this.
 */
@ProviderType
public class CommerceQuery {

    public static final String PARAM_QUERYTEXT = "q";
    public static final String PARAM_PAGE = "page";
    public static final String PARAM_PAGESIZE = "pagesize";
    public static final String PARAM_SORTID = "sort";
    
    protected String queryText;
    protected int page;
    protected int pageSize;
    protected String sortId;
    protected Map<String, List<String>> facets;
    
    protected CommerceQuery() {
        // protected default constructor
    }
    
    public CommerceQuery(HttpServletRequest request) {
        queryText = request.getParameter(PARAM_QUERYTEXT);
        
        if (StringUtils.isNumeric(request.getParameter(PARAM_PAGE))) {
            page = Integer.parseInt(request.getParameter(PARAM_PAGE));
        }
        
        if (StringUtils.isNumeric(request.getParameter(PARAM_PAGESIZE))) {
            pageSize = Integer.parseInt(request.getParameter(PARAM_PAGESIZE));
        }
        
        sortId = request.getParameter(PARAM_SORTID);
        
        facets = new HashMap<String, List<String>>();
        final Map<String, String[]> parameterMap = request.getParameterMap();
        for (Map.Entry<String, String[]> entry: parameterMap.entrySet()) {
            if (entry.getKey().startsWith(FacetParamHelper.PREFIX)) {
                final String facetId = entry.getKey().substring(FacetParamHelper.PREFIX.length());
                final List<String> values = Arrays.asList(entry.getValue());
                facets.put(facetId, values);
            }
        }
    }

    /**
     * A generic builder for queries executed via {@link CommerceService#search(CommerceQuery)}.
     */
    public static class Builder {
        private CommerceQuery query;

        public Builder() {
            query = new CommerceQuery();
        }

        /**
         * Set the actual query string. Allowed query terms depend
         * on the commerce provider.
         */
        public Builder setQueryText(String queryText) {
            query.queryText = queryText;
            return this;
        }

        /**
         * Set the current page within the search results page set.
         */
        public Builder setPage(int page) {
            query.page = page;
            return this;
        }

        /**
         * Set the number of hits to display per page of the search results.
         */
        public Builder setPageSize(int pageSize) {
            query.pageSize = pageSize;
            return this;
        }

        /**
         * Set the sort type's identifier. Allowed sort types are
         * depend on the commerce provider and should generally be set to
         * one of the values returned from {@link CommerceResult#getSorts()}.
         */
        public void setSortId(String sortId) {
            query.sortId = sortId;
        }

        /**
         * Build a query from this builder
         * 
         * @throws IllegalStateException
         *             if no queryText has been set
         */
        public CommerceQuery build() throws IllegalStateException {
            if (query.queryText == null) {
                throw new IllegalStateException("Cannot create query withthout queryText");
            }
            // swap variables so further changes to this instance are not possible through the builder
            final CommerceQuery result = query;
            query = null;
            return result;
        }
    }

    /**
     * Get the query string.
     */
    public String getQueryText() {
        return queryText;
    }

    /**
     * Get the result page's index.
     */
    public int getPage() {
        return page;
    }

    /**
     * Get the page size.
     */
    public int getPageSize() {
        return pageSize;
    }

    /**
     * Get the sort type's identifier.
     */
    public String getSortId() {
        return sortId;
    }

    /**
     * Get the list of facets applicable to current search.
     */
    public Map<String, List<String>> getFacets() {
        return facets;
    }
}
