/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.commerce.api;

import java.util.List;
import java.util.Map;

import aQute.bnd.annotation.ConsumerType;
import com.adobe.cq.commerce.api.collection.ProductCollection;
import com.adobe.cq.commerce.api.promotion.Promotion;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.SlingHttpServletResponse;

import com.adobe.cq.commerce.api.promotion.Voucher;
import com.day.cq.wcm.api.Page;
import org.apache.sling.api.resource.ResourceResolver;

/**
 * The <code>CommerceService</code> is the root of the Commerce API; other classes are fetched
 * through it.
 *
 * <p>In the case of external commerce providers, the CommerceService handles basic connectivity to
 * the commerce server, including command translation and issuance to the server's service API.</p>
 *
 * <p>Individual resources in the repository can nominate a particular implementation of the
 * {@link CommerceService} through their {@link CommerceConstants#PN_COMMERCE_PROVIDER}
 * properties.</p>
 *
 * <p>A {@link CommerceService} is then normally fetched using the adapter pattern:
 * <code>resource.adaptTo(CommerceService.class)</code></p>
 */
@ConsumerType
public interface CommerceService {

    /**
     * Returns a <code>CommerceSession</code> for the user identified by the given request.
     * @param request The scope's request.
     * @param response The scope's response.
     * @return A {@link CommerceSession} for the user.
     * @throws CommerceException
     */
    public CommerceSession login(SlingHttpServletRequest request, SlingHttpServletResponse response) throws CommerceException;

    /**
     * Test whether the commerce service (especially ones that rely on 3rd party services) is
     * currently available.
     * @param serviceType The type of the service that is tested to be available (e.g. payment,
     *                    fulfillment etc.).
     * @return true if available
     * @see CommerceConstants#SERVICE_COMMERCE
     */
    public boolean isAvailable(String serviceType);
    
    /**
     * Get the configured commerce server endpoint.
     * @return
     */
    public String getServer();

    /**
     * Set a context for this service instance and any session retrieved from it. Service and
     * session implementations may send the context with requests to remote commerce providers.
     *
     * <p><b>Note that the context is not persisted across requests and needs to be set again
     * when creating a new service instance.</b></p>
     *
     * @param context
     */
    public void setContext(Map<String, Object> context);

    /**
     * Get the context or an empty map if no context has been set before via {@link #setContext(Map)}.
     * @return the context
     */
    public Map<String, Object> getContext();

    /**
     * Get the product by path.
     * @param path
     * @return  The product or <code>null</code> if the path doesn't represent a product.
     */
    public Product getProduct(String path) throws CommerceException;

    /**
     * Get the voucher by path (or ID for external promotions).
     * @param path
     * @return The promotion or <code>null</code> if not found.
     */
    public Promotion getPromotion(String path) throws CommerceException;

    /**
     * Get the voucher by path (or ID for external vouchers).
     * @param path
     * @return The voucher or <code>null</code> if not found.
     */
    public Voucher getVoucher(String path) throws CommerceException;

    /**
     * Get the product collection by path (or ID for external product collections).
     * @param path
     * @return The product collection or <code>null</code> if not found.
     */
    public ProductCollection getProductCollection(String path) throws CommerceException;

    /**
     * Check for public availability of a product.
     * @param product
     * @return true if publicly available
     */
    public boolean isActivated(Product product) throws CommerceException;

    /**
     * Called during catalog creation or update to allow the commerce implementation to perform
     * implementation-specific actions.
     *
     * <p>The <code>catalogRolloutHook</code> is called after the root catalog page is created
     * or updated.</p>
     *
     * <p>Note: if changes are made, the appropriate lastModified dates should be updated.</p>
     *
     * @param blueprint The catalog blueprint.
     * @param catalog The catalog page created from the blueprint.
     */
    public void catalogRolloutHook(Page blueprint, Page catalog) throws CommerceException;

    /**
     * Called during catalog creation or update to allow the commerce implementation to perform
     * implementation-specific actions.
     *
     * <p>The <code>sectionRolloutHook</code> is called after each section page is created or
     * updated.</p>
     *
     * <p>Note: if changes are made, the appropriate lastModified dates should be updated.</p>
     *
     * @param blueprint The catalog section blueprint.
     * @param section The section page created from the blueprint.
     */
    public void sectionRolloutHook(Page blueprint, Page section) throws CommerceException;

    /**
     * Called during catalog creation or update to allow the commerce implementation to perform
     * implementation-specific actions.
     *
     * <p>The <code>productRolloutHook</code> is called after each product page is created or
     * updated.</p>
     *
     * <p>Note: if changes are made, the appropriate lastModified dates should be updated.</p>
     *
     * @param productData The productData for which the product page was created.
     * @param productPage The product page created from the section blueprint's product template.
     * @param productReference The product reference on the created/updated page.
     */
    public void productRolloutHook(Product productData, Page productPage, Product productReference) throws CommerceException;

    /**
     * Perform a search.
     * @param query The query representing the search parameters
     * @return a {@link CommerceResult} representing the search results
     */
    public CommerceResult search(CommerceQuery query) throws CommerceException;

    /**
     * Get the list of available promotions.
     * @param resourceResolver a Sling ResourceResolver instance
     * @return  A <code>List&lt;Promotion&gt;</code> containing the promotions.
     */
    public List<Promotion> getAvailablePromotions(ResourceResolver resourceResolver) throws CommerceException;

    /**
     * Get the list of predicates for selecting placed orders.
     * @return A <code>List</code> of order predicate names.
     */
    public List<String> getOrderPredicates() throws CommerceException;

    /*
     * ==================================================================================================
     * Deprecated methods.  For backwards-compatibility only.
     * ==================================================================================================
     */

    /**
     * Get the list of supported countries.
     * @return  A <code>List</code> containing ISO-country-code values.
     * @deprecated since 6.1, use {@link CommerceSession#getAvailableCountries()}
     */
    @Deprecated
    public List<String> getCountries() throws CommerceException;

    /**
     * Get the list of supported credit-card types.
     * @return  A <code>List</code> containing credit-card types.
     * @deprecated since 6.1, use {@link CommerceSession#getAvailablePaymentMethods()}
     */
    @Deprecated
    public List<String> getCreditCardTypes() throws CommerceException;


}
