/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.commerce.api;

import java.util.List;
import java.util.Locale;
import java.util.Map;

import aQute.bnd.annotation.ConsumerType;
import com.adobe.cq.commerce.api.promotion.VoucherInfo;
import com.adobe.cq.commerce.api.smartlist.SmartListManager;
import org.apache.commons.collections.Predicate;

import com.adobe.cq.commerce.api.promotion.PromotionInfo;
import com.adobe.cq.commerce.api.promotion.Voucher;

/**
 * The <code>CommerceSession</code> provides anonymous or authenticated access to a (possibly remote)
 * commerce server.
 *
 * <p>Session persistence is divided into two main parts: the cart entries (each one modeled by the
 * <code>CartEntry</code> interface), and the various order properties (collectively modeled by a
 * <code>Map&lt;String, Object></code>).</p>
 *
 * <p>In theory the various order properties could be updated as required via one (or more) calls to
 * <code>updateOrder()</code>, and then the order submitted via an empty call to <code>placeOrder()</code>.
 * However, in practice, security constraints (such as being forbidden from storing CCV values)
 * necessitate passing some of the order's properties in the final (<code>placeOrder()</code>) call.
 *
 * <p>It's up to the individual implementation to determine how many check-out steps there should be,
 * and how much information to include (only) in the final call.</p>
 */
@ConsumerType
public interface CommerceSession {
    /**
     * The property name for CartEntry quantity when passed in the property map.
     */
    public static final String PN_QUANTITY = "quantity";

    /**
     * Property name for read-only CartEntry.
     */
    public static final String PN_READONLY = "readonly";

    /**
     * Logs the current user out.
     */
    public void logout() throws CommerceException;

    /**
     * Forces a certain locale for the current user. Prices will be returned according the locale's
     * currency, if available.
     * @param locale
     */
    public void setUserLocale(Locale locale);

    /**
     * Return the forced user locale or <code>null</code>, if no locale was set before.
     * @return
     */
    public Locale getUserLocale();

    /**
     * Return the list of countries supported for this session/cart.
     *
     * While many implementations will in fact return a session-independent result, putting the
     * method on <code>CommerceSession</code> allows more sophisticated implementations to perform
     * session/cart-dependent processing (such as export restrictions on ITAR products).
     *
     * @return
     * @throws CommerceException
     */
    public List<String> getAvailableCountries() throws CommerceException;

    /**
     * Return the list of shipping methods supported for this session/cart.
     *
     * While many implementations will in fact return a session-independent result, putting the
     * method on <code>CommerceSession</code> allows more sophisticated implementations to perform
     * session/cart-dependent processing (such as air-freight restrictions on high-VOC products).
     *
     * @return
     * @throws CommerceException
     */
    public List<ShippingMethod> getAvailableShippingMethods() throws CommerceException;

    /**
     * Return the list of payment methods supported for this session/cart.
     *
     * While many implementations will in fact return a session-independent result, putting the
     * method on <code>CommerceSession</code> allows more sophisticated implementations to perform
     * session/cart-dependent processing (such as locale-specific payment vendors).
     *
     * @return
     * @throws CommerceException
     */
    public List<PaymentMethod> getAvailablePaymentMethods() throws CommerceException;

    /**
     * Get a list of all detailed price infos for a product  See {@link PriceInfo} for more details.
     * @param product
     * @return
     * @throws CommerceException
     */
    public List<PriceInfo> getProductPriceInfo(Product product) throws CommerceException;

    /**
     * Get a list of detailed price infos for a product filtered by the provided predicate
     * (usually a {@link com.adobe.cq.commerce.common.PriceFilter}).
     *
     * <p>See {@link PriceInfo} for more details.</p>
     *
     * @param product The product.
     * @param filter An optional predicate used for filtering the {@link PriceInfo}s.
     *               Pass <code>null</code> to fetch all {@link PriceInfo}s.
     * @return A list of all {@link PriceInfo}s which match the provided predicate.
     */
    public List<PriceInfo> getProductPriceInfo(Product product, Predicate filter) throws CommerceException;

    /**
     * Shortcut for getting a formatted string for the first applicable price for a product as
     * defined by the provided predicate (usually a {@link com.adobe.cq.commerce.common.PriceFilter}).
     *
     * <p>See {@link PriceInfo} for more details.</p>
     *
     * @param product The product
     * @param filter An optional predicate used for filtering the {@link PriceInfo}s.
     *               Pass <code>null</code> to fetch all {@link PriceInfo}s.
     * @return A formatted price string.
     */
    public String getProductPrice(Product product, Predicate filter) throws CommerceException;

    /**
     * Shortcut for getting a formatted string for the first price for a product.
     *
     * <p>See {@link PriceInfo} for more details.</p>
     *
     * @param product The product.
     * @return A formatted price string.
     */
    public String getProductPrice(Product product) throws CommerceException;

    /**
     * Get the number of cart entries.
     * @return
     * @throws CommerceException
     */
    public int getCartEntryCount() throws CommerceException;

    /**
     * Get the cart entries as a <code>List</code>.
     * @return
     * @throws CommerceException
     */
    public List<CartEntry> getCartEntries() throws CommerceException;

    /**
     * <code>CartEntry</code> represents shopping cart line-items with a certain product and quantity.
     */
    public interface CartEntry {
        /**
         * @return the (0-based) index of the entry in the cart
         */
        public int getEntryIndex();

        /**
         * @return the {@link Product} for this entry
         * @throws CommerceException
         */
        public Product getProduct() throws CommerceException;

        /**
         * @return the quantity
         */
        public int getQuantity();

        /**
         * <p>Get a list of detailed price info objects for a cart entry filtered by the provided
         * predicate (usually a {@link com.adobe.cq.commerce.common.PriceFilter}).</p>
         *
         * <p>Current components expect price infos with (at least) the following classifiers, in
         * this order:</p>
         * <ol>
         *     <li>"LINE" (line subtotal, generally without tax)</li>
         *     <li>"UNIT" (unit price, generally without tax)</li>
         * </ol>
         *
         * <p><strong>Example:</strong> to get the unit price of a cart entry, you could use the
         * following statement:</p>
         * <pre>
         * {@code
         * cartEntry.getPriceInfo(new PriceFilter("UNIT"));
         * }
         * </pre>
         *
         * @param filter An optional predicate used for filtering the {@link PriceInfo}s.
         *               Pass <code>null</code> to fetch all {@link PriceInfo}s.
         * @return a list of prices, can be empty if no price matches.
         * @see com.adobe.cq.commerce.common.PriceFilter
         */
        public List<PriceInfo> getPriceInfo(Predicate filter) throws CommerceException;

        /**
         * Shortcut for getting a formatted string of the first matching price.
         * @see #getPriceInfo(Predicate)
         *
         * @param filter An optional predicate used for filtering the {@link PriceInfo}s.
         * @return Formatted price string of first matching price
         */
        public String getPrice(Predicate filter) throws CommerceException;

        /**
         * Get a cart entry property with the given type.
         *
         * @param name The property's name
         * @param type The property's type
         * @return The property or <code>null</code> if it doesn't exist
         * or can not be converted to the given type.
         */
        public <T> T getProperty(String name, Class<T> type);

        /**
         * @return a formatted unit price
         * @deprecated since 5.6, use {@link #getPrice} instead.
         */
        @Deprecated
        public String getUnitPrice();
        /**
         * @return a formatted pre-tax price
         * @deprecated since 5.6, use {@link #getPrice} instead.
         */
        @Deprecated
        public String getPreTaxPrice();
        /**
         * @return a formatted tax value
         * @deprecated since 5.6, use {@link #getPrice} instead.
         */
        @Deprecated
        public String getTax();
        /**
         * @return a formatted total price
         * @deprecated since 5.6, use {@link #getPrice} instead.
         */
        @Deprecated
        public String getTotalPrice();

    }

    /**
     * <p>Get detailed price information about the cart filtered by the provided predicate
     * (usually a {@link com.adobe.cq.commerce.common.PriceFilter}).</p>
     *
     * <p>Current components expect the following prices, in this order:</p>
     * <ol>
     *     <li>"CART" (cart subtotal, generally pre-tax)</li>
     *     <li>"ORDER" (order total, generally including tax and shipping)</li>
     * </ol>
     *
     * <p><strong>Examples:</strong></p>
     * <pre>
     * {@code
     * // get the cart subtotal
     * session.getCartPrice(null);
     * // get cart total
     * session.getCartPrice(new PriceFilter("POST_TAX"));
     * // get order tax
     * session.getCartPrice(new PriceFilter("ORDER", "TAX"));
     * }
     * </pre>
     *
     * @param filter An optional predicate used for filtering the {@link PriceInfo}s.
     *               Pass <code>null</code> to fetch all {@link PriceInfo}s.
     * @return a list of prices, can be empty if no price matches.
     * @throws CommerceException
     */
    public List<PriceInfo> getCartPriceInfo(Predicate filter) throws CommerceException;

    /**
     * Shortcut for getting a formatted string of the first applicable price, according to the filter.
     * @see #getCartPriceInfo(Predicate)
     *
     * @param filter An optional predicate used for filtering the {@link PriceInfo}s.
     *               Pass <code>null</code> to fetch all {@link PriceInfo}s.
     * @return Formatted price string of first matching price
     * @throws CommerceException
     */
    public String getCartPrice(Predicate filter) throws CommerceException;

    /**
     * Add a new entry to the cart.
     *
     * @param product The product to add
     * @param quantity The number of items to add
     * @throws CommerceException
     */
    public void addCartEntry(Product product, int quantity) throws CommerceException;

    /**
     * Add a new entry to the cart.
     *
     * @param product The product to add
     * @param quantity The number of items to add
     * @param properties the Map of additional properties (key-value pairs)
     * @throws CommerceException
     */
    public void addCartEntry(Product product, int quantity, Map<String, Object> properties) throws CommerceException;
    
    /**
     * Modify quantity of an existing cart entry.
     *
     * @param entryNumber The cart entry's number
     * @param quantity The new quantity
     * @throws CommerceException
     */
    public void modifyCartEntry(int entryNumber, int quantity) throws CommerceException;

    /**
     * Modify properties of an existing cart entry.
     *
     * @param entryNumber the cart entry number
     * @param delta a Map of properties (key-value pairs) to be modified
     * @throws CommerceException
     */
    public void modifyCartEntry(int entryNumber, Map<String, Object> delta) throws CommerceException;

    /**
     * Delete an existing cart entry.
     *
     * @param entryNumber The cart entry's number
     * @throws CommerceException
     */
    public void deleteCartEntry(int entryNumber) throws CommerceException;

    /**
     * Apply a voucher to this session's cart.
     * 
     * @param code the voucher's code
     * @throws CommerceException
     */
    public void addVoucher(String code) throws CommerceException;
    
    /**
     * Remove a voucher that was previously added with {@link #addVoucher(String)}.
     * 
     * @param code the voucher's code
     * @throws CommerceException
     */
    public void removeVoucher(String code) throws CommerceException;

    /**
     * Get a list of vouchers that were added to this cart via {@link #addVoucher(String)}, or that
     * were assigned by the commerce engine.
     *
     * @return
     * @throws CommerceException
     */
    public List<VoucherInfo> getVoucherInfos() throws CommerceException;

    /**
     * Client-side promotion resolution allows a ClientContext store (the CartMgr in the OOTB
     * implementation) to resolve promotions and then adjust the current CommerceSession via
     * {@link #addPromotion(String)} and {@link #removePromotion(String)}.
     *
     * In contrast, if a CommerceSession wishes to resolve its own promotions on the server, it
     * should return <code>false</code> here and need only implement {@link #getPromotions()} to
     * return the resolved promotion info.
     *
     * @return <code>true</code> if underlying implementation has support
     */
    public boolean supportsClientsidePromotionResolution();

    /**
     * Apply a promotion to this session's cart.
     *
     * @param path A string uniquely identifying the promotion (generally a path)
     * @throws CommerceException
     */
    public void addPromotion(String path) throws CommerceException;

    /**
     * Remove a promotion.
     *
     * @param path A string uniquely identifying the promotion (generally a path)
     * @throws CommerceException
     */
    public void removePromotion(String path) throws CommerceException;

    /**
     * Get a list of promotions that are active for this cart.  Promotions might have been added
     * via {@link #addPromotion(String)}, or might have been assigned by the commerce engine.
     *
     * @return
     * @throws CommerceException
     */
    public List<PromotionInfo> getPromotions() throws CommerceException;

    /**
     * Get a unique ID which identifies the current cart/order.  Note that the semantics of this
     * ID are highly implementation-dependent.  Hybris, for example, will return null until the
     * first product is added to the cart.  Other systems may have different semantics.
     *
     * @return A unique ID
     * @throws CommerceException
     */
    public String getOrderId() throws CommerceException;

    /**
     * Get a subset of the current order details.  The OOTB commerce components assume support
     * for the following predicates:
     * <ul>
     *     <li>{@link CommerceConstants#BILLING_ADDRESS_PREDICATE}</li>
     *     <li>{@link CommerceConstants#SHIPPING_ADDRESS_PREDICATE}</li>
     * </ul>
     * <p>In addition, implementations would be expected to support predicates defined by
     * any available {@link ShippingMethod}s and {@link PaymentMethod}s.</p>
     *
     * @param predicate A <code>String</code> identifying a predicate which in turn identifies
     *                  the subset.
     * @return A <code>Map</code> of name/value pairs
     * @throws CommerceException
     */
    public Map<String, Object> getOrderDetails(String predicate) throws CommerceException;

    /**
     * Get the current order details.
     *
     * In order to facilitate usage within forms, it is suggested that implementations to prefix
     * any predicate details with their predicate (eg: "billing.street").
     *
     * @return A <code>Map</code> of name/value pairs
     * @throws CommerceException
     */
    public Map<String, Object> getOrder() throws CommerceException;

    /**
     * A predicate-based update call which makes it easier to update/clear a particular payment
     * method, shipping method, address, etc.
     *
     * @param details A <code>Map</code> containing the order properties to update.
     * @param predicate A predicate to assign to the properties.
     * @throws CommerceException
     */
    public void updateOrderDetails(Map<String, Object> details, String predicate) throws CommerceException;

    /**
     * Update a subset of the current order details.  Entries in the <code>Map</code> which do not
     * yet exist will be added to the order, while entries that already exist will be updated.
     *
     * @param delta A <code>Map</code> of name/value pairs
     * @throws CommerceException
     */
    public void updateOrder(Map<String, Object> delta) throws CommerceException;

    /**
     * Submit the current order.  Note that the semantics of submitting an order are somewhat
     * implementation-dependent.
     *
     * @param orderDetailsDelta An optional set of order details to update before submission.
     *                          Useful for passing properties which cannot be stored, such as
     *                          the CCV.
     * @throws CommerceException
     */
    public void placeOrder(Map<String, Object> orderDetailsDelta) throws CommerceException;

    /**
     * Get a list of previously placed orders.
     *
     * @param predicate An optional implementation-specific predicate name.  Implementations
     *                  should support at least {@link CommerceConstants#OPEN_ORDERS_PREDICATE}.
     * @param pageNumber the requested page number of the list of placed orders
     * @param pageSize the number of orders in a page
     * @param sortId the identifier of the sorting used to sort the list of placed orders
     *
     * @return a {@code PlacedOrderResult} object which contains the list of implementation-specific order summary info,
     * the pagination and sorting info of the returned orders. If {@code PaginationInfo} is null in the returned object
     * then the commerce implementation doesn't support pagination. If the returned object does not contain any
     * {@code CommerceSort} object then the returned orders are unsorted.
     *
     * @throws CommerceException
     */
    public PlacedOrderResult getPlacedOrders(String predicate, int pageNumber, int pageSize, String sortId) throws CommerceException;

    /**
     * Fetch a specific (previously-placed) order.
     *
     * @param orderId An implementation-specific id identifying the placed order
     * @return a {@link PlacedOrder}
     * @throws CommerceException
     */
    public PlacedOrder getPlacedOrder(String orderId) throws CommerceException;

    /**
     * Get the smart list manager for the commerce service implementation.
     *
     * @return A <code>SmartListManager</code> of order predicate names.
     */
    public SmartListManager getSmartListManager();

    /*
     * ==================================================================================================
     * Deprecated methods.  For backwards-compatibility only.
     * ==================================================================================================
     */

    /**
     * Get formatted price for a product.
     * @return
     * @deprecated since 5.6, use {@link #getProductPrice(Product)}
     */
    @Deprecated
    public String getPriceInfo(Product product) throws CommerceException;

    /**
     * Get formatted pre-tax price.
     * @return
     * @deprecated since 5.6, use {@link #getCartPrice(Predicate)} instead.
     */
    @Deprecated
    public String getCartPreTaxPrice() throws CommerceException;

    /**
     * Get formatted tax.
     * @return
     * @deprecated since 5.6, use {@link #getCartPrice(Predicate)} instead.
     */
    @Deprecated
    public String getCartTax() throws CommerceException;

    /**
     * Get formatted cart total.
     * @return
     * @deprecated since 5.6, use {@link #getCartPrice(Predicate)} instead.
     */
    @Deprecated
    public String getCartTotalPrice() throws CommerceException;

    /**
     * Get formatted shipping price.
     * @return
     * @deprecated since 5.6, use {@link #getCartPrice(Predicate)} instead.
     */
    @Deprecated
    public String getOrderShipping() throws CommerceException;

    /**
     * Get formatted total tax.
     * @return
     * @deprecated since 5.6, use {@link #getCartPrice(Predicate)} instead.
     */
    @Deprecated
    public String getOrderTotalTax() throws CommerceException;

    /**
     * Get formatted order total.
     * @return
     * @deprecated since 5.6, use {@link #getCartPrice(Predicate)} instead.
     */
    @Deprecated
    public String getOrderTotalPrice() throws CommerceException;

    /**
     * Get a list of vouchers that were added to this cart via {@link #addVoucher(String)}.
     * @return
     * @deprecated since 5.6.200; use {@link #getVoucherInfos()} instead
     */
    @Deprecated
    public List<Voucher> getVouchers() throws CommerceException;

    /**
     * Update a set of details within the current order.
     * @param delta
     * @throws CommerceException
     * @deprecated since 5.6, use {@link #updateOrder} instead.
     */
    @Deprecated
    public void updateOrderDetails(Map<String, String> delta) throws CommerceException;

    /**
     * Get the full details of the current order.
     * @return
     * @throws CommerceException
     * @deprecated since 5.6, use {@link #getOrder} instead.
     */
    @Deprecated
    public Map<String, String> getOrderDetails() throws CommerceException;

    /**
     * Submit the current order.
     * @param orderDetailsDelta
     * @throws CommerceException
     * @deprecated since 5.6, use {@link #placeOrder} instead.
     */
    @Deprecated
    public void submitOrder(Map<String, String> orderDetailsDelta) throws CommerceException;

}
