/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.commerce.api;

import java.math.BigDecimal;
import java.text.NumberFormat;
import java.util.Currency;
import java.util.HashMap;
import java.util.Locale;

import aQute.bnd.annotation.ProviderType;
import org.apache.sling.api.wrappers.ValueMapDecorator;

/**
 * <code>PriceInfo</code> provides information about prices for products, cart entries, and orders.
 *
 * <p>A product, cart entry, or order can have many prices associated with it.  Each price is identified
 * by a list of string classifiers.  For instance, a product might have the following prices:</p>
 * <ol>
 *     <li>"UNIT", "PRE_TAX", "USD"</li>
 *     <li>"UNIT", "PRE_TAX", "EUR"</li>
 *     <li>"SHIPPING", "WEIGHT_SURCHARGE", "SURFACE", "EUR"</li>
 *     <li>"SHIPPING", "WEIGHT_SURCHARGE", "AIR", "EUR"</li>
 * </ol>
 *
 * <p>A {@link com.adobe.cq.commerce.common.PriceFilter}</p> is normally used when fetching prices.
 * Price searches can return a list (generally <code>getPriceInfo()</code>) of all prices matching the
 * filter, or can return only the first (generally <code>getPrice()</code>).  Because of the second
 * form, more common (or default) price infos should be stored in front of less common (more specific)
 * price infos.</p>
 *
 * <p>For instance, if a product had the following prices:</p>
 * <ol>
 *     <li>"TAX"</li>
 *     <li>"TAX", "CALIFORNIA", "SANTA_CLARA"</li>
 *     <li>"TAX", "CALIFORNIA", "SAN_DIEGO"</li>
 * </ol>
 * <p>The first price would be returned from <code>getPrice(new PriceFilter("TAX"))</code>, while the
 * second would be returned from <code>getPrice(new PriceFilter("TAX", "CALIFORNIA"))</code>.</p>
 *
 * <p>In general, the set of price classifiers used by an e-commerce site are implementation-dependent.
 * However, the out-of-the-box commerce components do make a few assumptions about the existence of
 * the following classifiers:</p>
 * <ul>
 *     <li>"UNIT" (for products & cart entries)</li>
 *     <li>"LINE" (for cart entries)</li>
 *     <li>"SHIPPING" (for orders)</li>
 *     <li>"ORDER" (for orders)</li>
 * </ul>
 *
 * <p>A <code>PriceInfo</code> also supports the retrieval of a formatted string for displaying
 * the price, and a minimum quantity at which the price is valid.</p>
 */
@ProviderType
public class PriceInfo extends ValueMapDecorator {

    public static final String MIN_QUANTITY = "minQuantity";

    private BigDecimal amount;
    private Currency currency;
    private Locale locale;

    /**
     * Create price info with a minimum quantity of one.
     * @param amount The actual price value.
     * @param currency The currency.
     * @deprecated since 5.6.200; use {@link #PriceInfo(java.math.BigDecimal, java.util.Locale)} instead.
     */
    @Deprecated
    public PriceInfo(BigDecimal amount, Currency currency) {
        super(new HashMap<String, Object>());

        this.amount = amount;
        this.currency = currency;
        this.locale = null;
    }

    /**
     * Create price info with a minimum quantity of one, and the default currency for the given locale.
     * @param amount The actual price value.
     * @param locale The locale.
     */
    public PriceInfo(BigDecimal amount, Locale locale) {
        super(new HashMap<String, Object>());

        this.amount = amount;
        this.currency = Currency.getInstance(locale);
        this.locale = locale;
    }

    /**
     * Create price info with a minimum quantity of one, and the given currency and locale.
     * @param amount The actual price value.
     * @param locale The locale.
     */
    public PriceInfo(BigDecimal amount, Locale locale, Currency currency) {
        super(new HashMap<String, Object>());

        this.amount = amount;
        this.locale = locale;
        this.currency = currency;
    }

    /**
     * Get the price's value.
     * @return
     */
    public BigDecimal getAmount() {
        return amount;
    }

    /**
     * Convenience method for retrieving the minimum quantity property.
     * If no quantity property is set, a minimum quantity of 1 is assumed.
     * @return
     */
    public long getQuantity() {
        return get(MIN_QUANTITY, 1L);
    }

    /**
     * Get the locale.
     * @return
     */
    public Locale getLocale() {
        return locale;
    }

    /**
     * Get the currency.
     * @return
     */
    public Currency getCurrency() {
        return currency;
    }

    /**
     * Get the formatted string.
     * @return
     */
    public String getFormattedString() {
        if (amount == null) {
            return "";
        }

        NumberFormat nf;
        if (locale != null) {
            nf = NumberFormat.getCurrencyInstance(locale);
            nf.setCurrency(currency);
        } else {
            nf = NumberFormat.getCurrencyInstance();
            nf.setCurrency(currency);
            nf.setMinimumFractionDigits(2);
            nf.setMaximumFractionDigits(2);
        }
        return nf.format(amount);
    }

    /**
     * Override the standard toString method to return the formatted string.
     * @return
     */
    @Override
    public String toString() {
        return getFormattedString();
    }
}
