/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.commerce.api;

import aQute.bnd.annotation.ConsumerType;
import com.day.cq.commons.ImageResource;
import org.apache.sling.api.adapter.Adaptable;
import org.apache.sling.api.resource.Resource;

import java.util.Iterator;
import java.util.List;

/**
 * The <code>Product</code> interface reflects resources that represent products and
 * their variants in an e-commerce system.
 *
 * Products have properties.  Properties which drive product variation are known as
 * "variant axes".  Note that this does <i>not</i> include properties which change as a
 * <i>result</i> of variation, such as "shipping-weight" or "price".
 *
 * Each product variation must have a unique resource.  Hierarchy above those leaf
 * resources may or may not be present, depending on the PIM architecture, but if
 * present should also be modelled by the Product interface.
 */
@ConsumerType
public interface Product extends Adaptable {
    
    /** Base resource type for products. */
    public static final String RESOURCE_TYPE_PRODUCT = "commerce/components/product";

    /**
     * Get the underlying resource's path.  This <b>must</b> be unique for a particular
     * variation.
     * @return
     */
    public String getPath();

    /**
     * Get the path of the parent page which renders the product/variation.
     * @return
     */
    public String getPagePath();

    /**
     * Get a unique identifier for a product variation.
     * @return
     */
    public String getSKU();

    /**
     * Returns the product's title.
     * @return
     */
    public String getTitle();

    /**
     * Returns the product's title based on the given selector.
     *
     * <p>The selector is often a language token, but could also be a regional identifier
     * or some other discriminator.</p>
     *
     * @param selectorString
     * @return
     * @since 6.1
     */
    public String getTitle(String selectorString);

    /**
     * Returns the product's description.
     * @return
     */
    public String getDescription();

    /**
     * Returns the product's description based on the given selector.
     *
     * <p>The selector is often a language token, but could also be a regional identifier
     * or some other discriminator.</p>
     *
     * @param selectorString
     * @return
     * @since 6.1
     */
    public String getDescription(String selectorString);

    /**
     * Returns the URL to the product's thumbnail image.
     * @return
     */
    public String getThumbnailUrl();
    
    /**
     * Returns the URL to the product's thumbnail image, resized to the specified width.
     * @param width
     * @return
     */
    public String getThumbnailUrl(int width);

    /**
     * Returns the URL to the product's thumbnail image, based on the given selector string.
     *
     * <p>The selector is often a size or orientation specification, but could also be used
     * for language- or region-specific images.</p>
     *
     * @param selectorString
     * @return
     */
    public String getThumbnailUrl(String selectorString);

    /**
     * Get the primary asset for this product, or {@code null} if no asset was found.
     * @return
     */
    public Resource getAsset();

    /**
     * Get all the assets for this product. Can return an empty list if no assets were found.
     * @return
     */
    public List<Resource> getAssets();

    /**
     * Get the primary image for this product, or {@code null} if no image was found.
     * @return
     * @since 5.6.1
     */
    public ImageResource getImage();

    /**
     * Get all the images for this product. Can return an empty list if no images were found.
     * @return
     * @since 5.6.1
     */
    public List<ImageResource> getImages();

    /**
     * Returns a product property.
     * @param name The property name
     * @param type The type the property value should be converted to
     * @return
     */
    public <T> T getProperty(String name, Class<T> type);

    /**
     * Return a product property based on the given selector.
     *
     * <p>The selector is often a language token, but could also be a regional identifier
     * or some other discriminator.</p>
     *
     * @param name
     * @param selectorString
     * @param type
     * @param <T>
     * @return
     * @since 6.1
     */
    public <T> T getProperty(String name, String selectorString, Class<T> type);

    /**
     * Returns an iterator over the property names of the variant axes.  For instance:
     * <code>["color", "size"]</code>.
     * @return
     */
    public Iterator<String> getVariantAxes();
    
    /**
     * Tests a single axis (property) for variation.
     * @param axis  The name of the property
     * @return      <code>true</code> if the given axis is variant
     */
    public boolean axisIsVariant(String axis);
    
    /**
     * Returns an iterator over the list of <b>all</b> product variations to which this
     * particular product/variant belongs.
     *
     * Within a hierarchy this routine should return the leaf nodes, regardless of the
     * intervening nodes or the point of entry.  For instance, given:
<pre>
    coat/
        coat-blue/
            coat-blue-S
            coat-blue-M
        coat-green/
            coat-green-S
</pre>
     * it must return <code>[coat-blue-S, coat-blue-M, coat-green-S]</code> whether called
     * from the <code>Product</code> representing <code>coat</code> or <code>coat-green-S</code>.
     *
     * @return
     */
    public Iterator<Product> getVariants() throws CommerceException;

    /**
     * Returns an iterator over the list of product variants for this product filtered by
     * the <code>VariantFilter</code>.
     *
     * <p>Example usages:
     *
     * <p>To get the complete list of sizes, irrespective of color:
<pre>
    Iterator<Product> iter = product.getVariants(new EnumerateAxisFilter("size"));
</pre>
     * <p>To get the list of sub-variants of the color "blue":
<pre>
    Iterator<Product> iter = product.getVariants(new AxisFilter("color", "blue"));
</pre>
     * <p>
     * @see VariantFilter
     * @see com.adobe.cq.commerce.common.EnumerateAxisFilter
     * @see com.adobe.cq.commerce.common.AxisFilter
     *
     * @return
     */
    public Iterator<Product> getVariants(VariantFilter filter) throws CommerceException;

    /**
     * Returns the product which forms the base of a set of variations.  If called on
     * a <code>Product</code> which is already the base, will return <code>this</code>.
     * @return The base <code>Product</code>
     * @throws CommerceException
     */
    public Product getBaseProduct() throws CommerceException;

    /**
     * Returns the PIM Product (under /etc/commerce/products) which stores this product's
     * data.  (Will return null for products which don't refer to PIM data.)
     * @return The PIM <code>Product</code>
     */
    public Product getPIMProduct() throws CommerceException;

    /*
     * ==================================================================================================
     * Deprecated methods.  For backwards-compatibility only.
     * ==================================================================================================
     */

    /**
     * Returns the URL to the product's primary image.
     * @return
     * @deprecated since 5.6.1; use {@link #getImage()} instead.
     */
    @Deprecated
    public String getImageUrl();

    /**
     * Returns the JCR path to the product's primary image.
     * @return
     * @deprecated since 5.6.1; use {@link #getImage()} instead.
     */
    @Deprecated
    public String getImagePath();

    /**
     * Get a thumbnail for this product, or {@code null} if no thumbnail was found.
     * @return
     * @deprecated since 6.1; use {@link #getThumbnailUrl()} instead.
     */
    @Deprecated
    public ImageResource getThumbnail();

}
