/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.cq.commerce.api.asset;

import java.util.List;
import java.util.Map;

import aQute.bnd.annotation.ProviderType;
import com.adobe.cq.commerce.api.CommerceException;
import com.adobe.cq.commerce.api.Product;
import org.apache.sling.api.resource.Resource;

/**
 * The <code>ProductAssetManager</code> provides methods to retrieve, update, create and update product assets.
 *
 * A product asset is a resource in the repository and has properties that define its rendering, e.g. the path
 * to the referenced asset, the resource type to render it, a width, a height, ...
 */
@ProviderType
public interface ProductAssetManager {

    /**
     * Indicates whether the asset referenced by <code>assetReference</code> is supported.
     *
     * @param assetReference the asset path in the repository
     * @return <code>true</code> if the asset is supported, <code>false</code> otherwise
     */
    public boolean isSupportedReferencedAsset(String assetReference);

    /**
     * Indicates whether the product asset at path <code>productAssetPath</code> is supported.
     *
     * @param productAssetPath the path of the product asset
     * @return <code>true</code> if the product asset is supported, <code>false</code> otherwise
     */
    public boolean isSupportedAsset(String productAssetPath);

    /**
     * Returns the path to the referenced asset.
     *
     * @param productAssetPath the path of the product asset
     * @return The path to the referenced asset
     */
    public String getReferencedAsset(String productAssetPath);

    /**
     * Returns the product asset that best represents the product.
     *
     * @param product The product
     * @return The product asset or <code>null</code> if there is none
     */
    public Resource getAsset(Product product);

    /**
     * Returns the product assets belonging to the product.
     *
     * @param product The product
     * @return The product assets or <code>null</code> if there are none
     */
    public List<Resource> getAssets(Product product);

    /**
     * Returns the URL to the product's thumbnail image, based on the given selector string.
     *
     * @param product The product
     * @param selectorString The selector string
     * @return The thumbnail URL or <code>null</code> if there is none
     */
    public String getThumbnailUrl(Product product, String selectorString);

    /**
     * Returns the URL to the product's thumbnail image, based on the given selector string.
     *
     * @param productAssetPath The path to the product asset
     * @param selectorString The selector string
     * @return The product assets or <code>null</code> if there are none
     */
    public String getThumbnailUrl(String productAssetPath, String selectorString);

    /**
     * Adds a product asset to the product.
     *
     * @param product The product
     * @param assetReference The path to the referenced asset
     * @return The new product asset
     * @throws com.adobe.cq.commerce.api.CommerceException when the operation fails.
     */
    public Resource addAsset(Product product, String assetReference) throws CommerceException;

    /**
     * Adds a product asset to the product.
     *
     * @param product The product
     * @param productAssetProperties The rendering properties of the product asset
     * @return The new product asset
     * @throws CommerceException when the operation fails.
     */
    public Resource addAsset(Product product, Map<String, Object> productAssetProperties) throws CommerceException;

    /**
     * Updates the address.
     *
     * @param productAssetPath The path of the product asset
     * @param assetReference The path to the referenced asset
     * @return The updated product asset
     * @throws CommerceException when the operation fails.
     */
    public Resource updateAsset(String productAssetPath, String assetReference) throws CommerceException;

    /**
     * Updates the address.
     *
     * @param productAssetPath The path of the product asset
     * @param productAssetProperties The rendering properties of the product asset
     * @return The updated product asset
     * @throws CommerceException when the operation fails.
     */
    public Resource updateAsset(String productAssetPath, Map<String, Object> productAssetProperties) throws CommerceException;

    /**
     * Removes the product asset.
     *
     * @param productAssetPath The path of the product asset
     * @throws CommerceException when the operation fails.
     */
    public void removeAsset(String productAssetPath) throws CommerceException;

}